package com.biz.crm.supermarket.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.common.TpmGlobalDictConstants;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.confadmin.model.KmsDirectStoreEntity;
import com.biz.crm.config.KmsConfig;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.grabrule.mapper.KmsGrabRuleMapper;
import com.biz.crm.grabrule.model.KmsGrabRuleEntity;
import com.biz.crm.kaproduct.mapper.KmsTenantryDirectProductMapper;
import com.biz.crm.kaproduct.model.KmsTenantryDirectProductEntity;
import com.biz.crm.mdm.cusorg.MdmCusOrgFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectAccountReqVo;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectCustomerOrgReqVo;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectStoreReqVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectAccountRespVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectCustomerOrgRespVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectStoreRespVo;
import com.biz.crm.nebular.mdm.cusorg.CusOrgVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectAccountMapper;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectStoreMapper;
import com.biz.crm.supermarket.mapper.KmsTotalWarehouseStoreMapper;
import com.biz.crm.supermarket.model.KmsTenantryDirectAccountEntity;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.supermarket.model.KmsTenantryDirectStoreEntity;
import com.biz.crm.supermarket.model.KmsTotalWarehouseStoreEntity;
import com.biz.crm.util.AesEncoder;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.KmsOrgUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;


/**
 * @author maoshen
 * @date 2021/4/13.
 */
@Component
@Slf4j
public class KmsTenantryDirectHelper {


    @Resource
    private MdmCusOrgFeign mdmCusOrgFeign;

    @Resource
    private KmsTenantryDirectCustomerOrgMapper tenantryDirectCustomerOrgMapper;

    @Resource
    private KmsTenantryDirectAccountMapper tenantryDirectAccountMapper;

    @Resource
    private KmsTenantryDirectStoreMapper tenantryDirectStoreMapper;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;

    @Resource
    private KmsTenantryDirectProductMapper kmsTenantryDirectProductMapper;

    @Resource
    private MdmTerminalFeign mdmTerminalFeign;

    @Resource
    private KmsConfig kmsConfig;

    @Resource
    private KmsGrabRuleMapper kmsGrabRuleMapper;

    @Resource
    private KmsTotalWarehouseStoreMapper kmsTotalWarehouseStoreMapper;

    /**
     * 加密密码最大长度
     */
    private static final Integer MAX_PASS_LENGTH = 32;

    /**
     * 转换数据  map
     *
     * @return
     */
    public Map<String, Map<String, String>> getDictMaps() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(TpmGlobalDictConstants.ENABLE_STATUS);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 转换列表数据
     *
     * @param list
     */
    public void convertData(List<KmsTenantryDirectCustomerOrgRespVo> list) {
        //查询当前创建者所属公司
        List<String> createOrgCodes = list.stream().map(KmsTenantryDirectCustomerOrgRespVo::getCreateOrgCode)
                .distinct().collect(Collectors.toList());
        Map<String,String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
        Map<String, Map<String, String>> dataMap = this.getDictMaps();
        Set<String> cusOrgSet = Sets.newHashSet();
        list.forEach(o -> {
            if (StringUtils.isNotBlank(o.getEnableStatus())) {
                o.setEnableStatusName(Optional.ofNullable(dataMap.get(TpmGlobalDictConstants.ENABLE_STATUS))
                        .orElse(Maps.newHashMap()).get(o.getEnableStatus()));
            }
            cusOrgSet.add(o.getCustomerOrgId());
            o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
        });
        CusOrgVo vo = new CusOrgVo();
        vo.setIds(Lists.newArrayList(cusOrgSet));
        vo.setPageSize(cusOrgSet.size());
        List<CusOrgVo> cusOrgVos = ApiResultUtil.objResult(mdmCusOrgFeign.list(vo), true).getData();
        Map<String, String> map = cusOrgVos.stream().collect(Collectors
                .toMap(CusOrgVo::getCustomerOrgCode, CusOrgVo::getCustomerOrgName));
        list.forEach(x -> {
            if (map.containsKey(x.getCustomerOrgCode())) {
                x.setCustomerOrgName(map.get(x.getCustomerOrgCode()));
            }
        });
    }

    /**
     * 转换客户账号列表数据
     *
     * @param list
     */
    public void convertAccountDate(List<KmsTenantryDirectAccountRespVo> list) {
        Map<String, Map<String, String>> dataMap = this.getDictMaps();
        //查询当前创建者所属公司
        List<String> createOrgCodes = list.stream().map(KmsTenantryDirectAccountRespVo::getCreateOrgCode)
                .distinct().collect(Collectors.toList());
        Map<String,String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
        list.forEach(o -> {
            if (StringUtils.isNotBlank(o.getEnableStatus())) {
                o.setEnableStatusName(Optional.ofNullable(dataMap.get(TpmGlobalDictConstants.ENABLE_STATUS))
                        .orElse(Maps.newHashMap()).get(o.getEnableStatus()));
            }
            o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
        });
    }

    /**
     * 新增校验
     *
     * @param reqVo
     */
    public List<KmsTenantryDirectCustomerOrgEntity> checkSaveData(KmsTenantryDirectCustomerOrgReqVo reqVo) {
        this.checkTenantryDirect(reqVo);
        CusOrgVo cusOrgVo = ApiResultUtil.objResult(mdmCusOrgFeign.query(reqVo.getCustomerOrgId(), ""), true);
        reqVo.setCustomerOrgCode(cusOrgVo.getCustomerOrgCode());
        reqVo.setCustomerOrgName(cusOrgVo.getCustomerOrgName());
        return this.checkDataExist(reqVo);
    }

    /**
     * 检验必填数据
     *
     * @param reqVo
     */
    private void checkTenantryDirect(KmsTenantryDirectCustomerOrgReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getDirectId(), "商超系统不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemName(), "直营体系名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getCustomerOrgId(), "组织客户信息必填");
    }

    /**
     * 检验数据存在
     *
     * @param reqVo
     * @return
     */
    private List<KmsTenantryDirectCustomerOrgEntity> checkDataExist(KmsTenantryDirectCustomerOrgReqVo reqVo) {

        KmsTenantryDirectCustomerOrgReqVo customerOrgReqVoForId = new KmsTenantryDirectCustomerOrgReqVo();
        KmsTenantryDirectCustomerOrgReqVo customerOrgReqVoForCode = new KmsTenantryDirectCustomerOrgReqVo();
        if (StringUtils.isNotBlank(reqVo.getId())) {
            customerOrgReqVoForCode.setId(reqVo.getId());
            customerOrgReqVoForId.setId(reqVo.getId());
        }
        customerOrgReqVoForCode.setBsDirectSystemCode(reqVo.getBsDirectSystemCode());
        customerOrgReqVoForId.setDirectId(reqVo.getDirectId());
        List<KmsTenantryDirectCustomerOrgEntity> entitiesByCode = tenantryDirectCustomerOrgMapper.selectListByCompetenceForExist(customerOrgReqVoForCode);
        List<KmsTenantryDirectCustomerOrgEntity> entitiesById = tenantryDirectCustomerOrgMapper.selectListByCompetenceForExist(customerOrgReqVoForId);
        if (CollectionUtils.isNotEmpty(entitiesById)) {
            if (entitiesById.get(0).getEnableStatus().equals(CrmEnableStatusEnum.ENABLE.getCode()) &&
                    entitiesById.get(0).getDelFlag().equals(CrmDelFlagEnum.NORMAL.getCode())) {
                throw new BusinessException("直营信息已经被维护");
            } else {
                return entitiesById;
            }
        }
        if (CollectionUtils.isNotEmpty(entitiesByCode)) {
            throw new BusinessException("直营体系编码不能重复");
        }
        return Lists.newArrayList();
    }

    /**
     * 客户账号新增校验
     *
     * @param reqVo
     */
    public void checkSaveAccountData(KmsTenantryDirectAccountReqVo reqVo) {
        // 检验必填字段
        this.checkNoEmptyAccountData(reqVo);
        if (StringUtils.isNotBlank(reqVo.getId())) {
            KmsTenantryDirectAccountEntity entity = tenantryDirectAccountMapper.selectById(reqVo.getId());
            if (entity.getLoginPass().equals(reqVo.getLoginPass())) {
                reqVo.setLoginPass(new AesEncoder(kmsConfig.getEncryptAesKey()).decode(reqVo.getLoginPass()));
                if (reqVo.getConfirmPass().equals(entity.getLoginPass())) {
                    reqVo.setConfirmPass(new AesEncoder(kmsConfig.getEncryptAesKey()).decode(reqVo.getConfirmPass()));
                }
            }
        }
        AssertUtils.isTrue(StringUtils.equals(reqVo.getLoginPass(), reqVo.getConfirmPass()), "登录密码和确认密码不一致");
        reqVo.setLoginPass(new AesEncoder(kmsConfig.getEncryptAesKey()).encode(reqVo.getLoginPass()));
        if (reqVo.getLoginPass().length() > MAX_PASS_LENGTH) {
            throw new BusinessException("输入密码过长");
        }
        // 直营体系是否存在，且状态正常
        KmsTenantryDirectCustomerOrgReqVo customerOrgReqVo = new KmsTenantryDirectCustomerOrgReqVo();
        customerOrgReqVo.setBsDirectSystemCode(reqVo.getBsDirectSystemCode());
        customerOrgReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<KmsTenantryDirectCustomerOrgEntity> tenantryDirectCustomerOrgEntities = tenantryDirectCustomerOrgMapper
                .selectListByCompetence(customerOrgReqVo);
        AssertUtils.isNotEmpty(tenantryDirectCustomerOrgEntities, "没有正在维护中的直营体系");
        reqVo.setBsDirectSystemName(tenantryDirectCustomerOrgEntities.get(0).getBsDirectSystemName());
        reqVo.setBsDirectSystemId(tenantryDirectCustomerOrgEntities.get(0).getId());
        // 登录账号不能重复
        this.checkAccountExist(reqVo);
    }

    /**
     * 检验登录账号是否重复
     *
     * @param reqVo
     */
    private void checkAccountExist(KmsTenantryDirectAccountReqVo reqVo) {

        KmsTenantryDirectAccountReqVo accountReqVo = new KmsTenantryDirectAccountReqVo();
        if (StringUtils.isNotBlank(reqVo.getId())) {
            accountReqVo.setId(reqVo.getId());
        }
        accountReqVo.setBsDirectSystemId(reqVo.getBsDirectSystemId());
        accountReqVo.setLoginAcc(reqVo.getLoginAcc());
        List<KmsTenantryDirectAccountEntity> entities = tenantryDirectAccountMapper.selectListByCompetenceForExist(accountReqVo);
        if (CollectionUtils.isNotEmpty(entities)) {
            throw new BusinessException("该客户账号已经存在");
        }
    }

    /**
     * 客户账号非空字段检验
     *
     * @param reqVo
     */
    private void checkNoEmptyAccountData(KmsTenantryDirectAccountReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系不能为空");
        AssertUtils.isNotEmpty(reqVo.getLoginAcc(), "登录账号不能为空");
        AssertUtils.isNotEmpty(reqVo.getLoginPass(), "登录密码不能为空");
        AssertUtils.isNotEmpty(reqVo.getConfirmPass(), "确认密码不能为空");
    }


    /**
     * 客户账号修改检验
     *
     * @param reqVo
     */
    public void checkAccountUpdate(KmsTenantryDirectAccountReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        KmsTenantryDirectAccountEntity entity = tenantryDirectAccountMapper.selectById(reqVo.getId());
        AssertUtils.isNotNull(entity, "修改数据不存在");
    }

    /**
     * 转换
     *
     * @param list
     */
    public void convertStoreData(List<KmsTenantryDirectStoreRespVo> list) {
        Map<String, Map<String, String>> dataMap = this.getDictMaps();
        //查询当前创建者所属公司
        List<String> createOrgCodes = list.stream().map(KmsTenantryDirectStoreRespVo::getCreateOrgCode)
                .distinct().collect(Collectors.toList());
        Map<String,String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
        list.forEach(o -> {
            if (StringUtils.isNotBlank(o.getEnableStatus())) {
                o.setEnableStatusName(Optional.ofNullable(dataMap.get(TpmGlobalDictConstants.ENABLE_STATUS))
                        .orElse(Maps.newHashMap()).get(o.getEnableStatus()));
            }
            o.setOwnedCompany(orgCompanyMap.get(o.getCreateOrgCode()));
        });
    }

    /**
     * 非业务字段置为null
     *
     * @param entity
     */
    public void setPublicParamsNull(CrmBaseEntity entity) {
        entity.setId(null);
        entity.setCreateCode(null);
        entity.setCreateDate(null);
        entity.setCreateDateSecond(null);
        entity.setCreateName(null);
        entity.setCreateOrgCode(null);
        entity.setCreatePosCode(null);
        entity.setCreateOrgName(null);
        entity.setCreatePosName(null);
    }


    /**
     * 门店新增检验
     *
     * @param reqVo
     */
    public void checkSaveStoreData(KmsTenantryDirectStoreReqVo reqVo) {
        // 必填数据校验
        this.checkNoEmptyStoreDate(reqVo);
        //逻辑检验 (检验是否有维护中的直营体系）
        KmsTenantryDirectCustomerOrgReqVo customerOrgReqVo = new KmsTenantryDirectCustomerOrgReqVo();
        customerOrgReqVo.setBsDirectSystemId(reqVo.getBsDirectSystemId());
        customerOrgReqVo.setBsDirectSystemCode(reqVo.getBsDirectSystemCode());
        customerOrgReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<KmsTenantryDirectCustomerOrgEntity> tenantryDirectCustomerOrgEntities = tenantryDirectCustomerOrgMapper
                .selectListByCompetence(customerOrgReqVo);
        AssertUtils.isNotEmpty(tenantryDirectCustomerOrgEntities, "没有在维护中的直营体系");
        KmsTenantryDirectCustomerOrgEntity tenantryDirectCustomerOrgEntity = tenantryDirectCustomerOrgEntities.get(0);
        reqVo.setBsDirectSystemName(tenantryDirectCustomerOrgEntity.getBsDirectSystemName());
        reqVo.setBsDirectSystemId(tenantryDirectCustomerOrgEntity.getId());
        MdmTerminalVo terminalVo = ApiResultUtil.objResult(mdmTerminalFeign.query("", reqVo.getTerminalCode()), true);
//        AssertUtils.isTrue(StringUtils.equals(terminalVo.getDirectSystemId(), tenantryDirectCustomerOrgEntity.getTenantryDirectId()),
//                "企业门店不在直营体系运营范围中");
        reqVo.setTerminalName(terminalVo.getTerminalName());
        KmsDirectStoreEntity entity = kmsDirectStoreMapper.selectById(reqVo.getStoreId());
        if (entity == null) {
            throw new BusinessException("商超门店不存在");
        }
        if (!entity.getDirectId().equals(tenantryDirectCustomerOrgEntities.get(0).getDirectId())) {
            throw new BusinessException("商超门店不在维护中");
        }
        //校验重复
        this.checkStoreExist(reqVo);

    }

    private void checkStoreExist(KmsTenantryDirectStoreReqVo reqVo) {
        KmsTenantryDirectStoreReqVo storeReqVo = new KmsTenantryDirectStoreReqVo();
        storeReqVo.setBsDirectSystemId(reqVo.getBsDirectSystemId());
        storeReqVo.setTerminalCode(reqVo.getTerminalCode());
        storeReqVo.setStoreId(reqVo.getStoreId());
        storeReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        if (StringUtils.isNotBlank(reqVo.getId())) {
            storeReqVo.setId(reqVo.getId());
        }
        List<KmsTenantryDirectStoreEntity> entities = tenantryDirectStoreMapper
                .selectListByCompetenceForExist(storeReqVo);
        if (CollectionUtils.isNotEmpty(entities)) {
            throw new BusinessException("该数据已存在");
        }
    }

    /**
     * 门店新增必填数据检验
     */
    private void checkNoEmptyStoreDate(KmsTenantryDirectStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemId(), "直营体系Id[bsDirectSystemId]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系编码[bsDirectSystemCode]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemName(), "直营体系名称[bsDirectSystemName]不能为空");
        AssertUtils.isNotEmpty(reqVo.getStoreId(), "商超门店id不能为空");
        AssertUtils.isNotEmpty(reqVo.getStoreCode(), "商超门店编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getStoreName(), "商超门店名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getTerminalCode(), "企业门店编码不能为空");
    }

    /**
     * 修改检验数据存在
     *
     * @param reqVo
     */
    public void checkStoreUpdate(KmsTenantryDirectStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), CommonException.IDS_NULL);
        KmsTenantryDirectStoreEntity storeEntity = tenantryDirectStoreMapper.selectById(reqVo.getId());
        AssertUtils.isNotNull(storeEntity, "修改数据不存在");
    }

    public void checkDelete(List<String> ids) {
        if (CollectionUtils.isNotEmpty(ids)) {
            return;
        }
        List<KmsTenantryDirectAccountEntity> accounts = tenantryDirectAccountMapper.selectList(Wrappers.<KmsTenantryDirectAccountEntity>lambdaQuery()
                .in(KmsTenantryDirectAccountEntity::getBsDirectSystemId, ids)
                .eq(KmsTenantryDirectAccountEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
        StringBuilder stringBuffer = new StringBuilder();
        if (CollectionUtils.isNotEmpty(accounts)) {
            stringBuffer.append("直营体系下存在客户账号、");
        }
        List<KmsTenantryDirectProductEntity> products = kmsTenantryDirectProductMapper.selectList(Wrappers.<KmsTenantryDirectProductEntity>lambdaQuery()
                .eq(KmsTenantryDirectProductEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(KmsTenantryDirectProductEntity::getBsDirectSystemId, ids));
        if (CollectionUtils.isNotEmpty(products)) {
            stringBuffer.append("直营体系下存在客户账号、");
        }
        List<KmsTenantryDirectStoreEntity> stores = tenantryDirectStoreMapper.selectList(Wrappers.<KmsTenantryDirectStoreEntity>lambdaQuery()
                .in(KmsTenantryDirectStoreEntity::getBsDirectSystemId, ids)
                .eq(KmsTenantryDirectStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
        if (CollectionUtils.isNotEmpty(stores)) {
            stringBuffer.append("直营该体系下存在门店、");
        }
        List<KmsGrabRuleEntity> grabRules = kmsGrabRuleMapper.selectList(Wrappers.<KmsGrabRuleEntity>lambdaQuery()
                .eq(KmsGrabRuleEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(KmsGrabRuleEntity::getBsDirectSystemId, ids));
        if (CollectionUtils.isNotEmpty(grabRules)) {
            stringBuffer.append("直营体系下存在抓单规则、");
        }
        List<KmsTotalWarehouseStoreEntity> warehouseStores = kmsTotalWarehouseStoreMapper.selectList(Wrappers.<KmsTotalWarehouseStoreEntity>lambdaQuery()
                .in(KmsTotalWarehouseStoreEntity::getBsDirectSystemId, ids)
                .eq(KmsTotalWarehouseStoreEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode()));
        if (CollectionUtils.isNotEmpty(warehouseStores)) {
            stringBuffer.append("直营体系存在大仓、");
        }
        if (StringUtils.isNotBlank(stringBuffer)) {
            stringBuffer.append("不可删除");
            throw new BusinessException(stringBuffer.toString());
        }
    }

    /**
     * 密码
     */
    /*public static void main(String[] args) {
        String a  = new AesEncoder("v3DRk/QisHTNZWwp")
                .decode("1733089cf9adc6396f92d108f4a89ebe");
        log.info(a);

        String b = new AesEncoder("v3DRk/QisHTNZWwp").encode("**YAOqq..123456");
        log.info(b);
    }*/
}
