package com.biz.crm.supermarket.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.KmsGlobalDictConstants;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.mapper.KmsDirectStoreMapper;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.kaproduct.mapper.KmsTenantryDirectProductMapper;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.kms.kaproduct.resp.KmsTenantryDirectProductRespVo;
import com.biz.crm.nebular.kms.supermarket.req.KmsTotalWarehouseStoreReqVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectStoreRespVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTotalWarehouseStoreRespVo;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.supermarket.mapper.KmsTotalWarehouseStoreMapper;
import com.biz.crm.supermarket.model.KmsTotalWarehouseStoreEntity;
import com.biz.crm.supermarket.service.IKmsTotalWarehouseStoreService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 总仓门店接口实现
 *
 * @author lf
 * @date 2021-06-26 11:20:06
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsTotalWarehouseStoreServiceExpandImpl")
public class KmsTotalWarehouseStoreServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsTotalWarehouseStoreMapper, KmsTotalWarehouseStoreEntity> implements IKmsTotalWarehouseStoreService {

    @Resource
    private KmsTotalWarehouseStoreMapper kmsTotalWarehouseStoreMapper;

    @Resource
    private KmsTenantryDirectCustomerOrgMapper kmsTenantryDirectCustomerOrgMapper;

    @Resource
    private KmsDirectStoreMapper kmsDirectStoreMapper;

    @Resource
    private MdmTerminalFeign mdmTerminalFeign;

    @Resource
    private KmsTenantryDirectProductMapper kmsTenantryDirectProductMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;


    /**
     * 转换数据备用map
     */
    public Map<String, Map<String, String>> getDictMaps() {
        List<String> dictCodes = Lists.newArrayList();
        dictCodes.add(KmsGlobalDictConstants.KMS_PRODUCT_TYPE);
        dictCodes.add(KmsGlobalDictConstants.ENABLE_STATUS);
        //查询字典数据
        return DictUtil.getDictValueMapsByCodes(dictCodes);
    }

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsTotalWarehouseStoreRespVo> findList(KmsTotalWarehouseStoreReqVo reqVo) {
        Page<KmsTotalWarehouseStoreRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsTotalWarehouseStoreRespVo> list = kmsTotalWarehouseStoreMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            this.convertData(list);
        }
        return PageResult.<KmsTotalWarehouseStoreRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 转换列表数据
     *
     * @param list
     */
    private void convertData(List<KmsTotalWarehouseStoreRespVo> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        //查询当前创建者所属公司
        List<String> createOrgCodes = list.stream().map(KmsTotalWarehouseStoreRespVo::getCreateOrgCode)
                .distinct().collect(Collectors.toList());
        Map<String, String> orgCompanyMap = KmsOrgUtil.findCompanyByCreateOrgCodes(createOrgCodes);
        Map<String, Map<String, String>> dataMap = this.getDictMaps();
        list.forEach(x -> {
            if (StringUtils.isNotBlank(x.getProductType())) {
                x.setProductTypeDesc(Optional.ofNullable(dataMap.get(KmsGlobalDictConstants.KMS_PRODUCT_TYPE))
                        .orElse(Maps.newHashMap()).get(x.getProductType()));
            }
            if (StringUtils.isNotBlank(x.getEnableStatus())) {
                x.setEnableStatusName(Optional.ofNullable(dataMap.get(KmsGlobalDictConstants.ENABLE_STATUS))
                        .orElse(Maps.newHashMap()).get(x.getEnableStatus()));
            }
            x.setOwnedCompany(orgCompanyMap.get(x.getCreateOrgCode()));
        });
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsTotalWarehouseStoreRespVo
     */
    @Override
    public KmsTotalWarehouseStoreRespVo query(String id) {
        AssertUtils.isNotEmpty(id, "请选择操作数据");
        KmsTotalWarehouseStoreReqVo reqVo = new KmsTotalWarehouseStoreReqVo();
        reqVo.setId(id);
        List<KmsTotalWarehouseStoreRespVo> respVos = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(respVos)) {
            return null;
        }
        KmsTotalWarehouseStoreRespVo kmsTotalWarehouseStoreRespVo = respVos.get(0);
        if (!org.springframework.util.StringUtils.isEmpty(kmsTotalWarehouseStoreRespVo.getDirectProductId())) {
            KmsTenantryDirectProductRespVo productRespVo = kmsTenantryDirectProductMapper
                    .findById(kmsTotalWarehouseStoreRespVo.getDirectProductId());
            List<KmsTenantryDirectProductRespVo> productRespVos = Lists.newArrayList();
            productRespVos.add(productRespVo);
            kmsTotalWarehouseStoreRespVo.setDirectProductReqVos(productRespVos);
        }
        return kmsTotalWarehouseStoreRespVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsTotalWarehouseStoreReqVo reqVo) {
        this.checkNotEmpty(reqVo);
        this.checkDataExist(reqVo);
        List<KmsTotalWarehouseStoreEntity> entities = this.convert(reqVo);
        //this.checkDataExistAndSave(entities);
        this.saveBatch(entities);
        List<KmsTenantryDirectStoreRespVo> newDataList = CrmBeanUtil.copyList(entities, KmsTenantryDirectStoreRespVo.class);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForAdd(menuCode.toString(), newDataList.get(i).getId(), newDataList.get(i).getId(), newDataList.get(i));
        }
    }

    private void checkDataExist(KmsTotalWarehouseStoreReqVo reqVo) {
        KmsTotalWarehouseStoreReqVo selectReqVo = new KmsTotalWarehouseStoreReqVo();
        selectReqVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        selectReqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        selectReqVo.setBsDirectSystemId(reqVo.getBsDirectSystemId());
        selectReqVo.setTerminalCode(reqVo.getTerminalCode());
        selectReqVo.setProductType(reqVo.getProductType());
        List<KmsTotalWarehouseStoreEntity> entities = kmsTotalWarehouseStoreMapper
                .selectListWithDataPurview(selectReqVo);
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        if (StringUtils.isBlank(entities.get(0).getProductCode())) {
            throw new BusinessException("数据已存在");
        }
        if (CollectionUtils.isNotEmpty(reqVo.getDirectProductReqVos())) {
            Set<String> productCodeSet = entities.stream()
                    .map(KmsTotalWarehouseStoreEntity::getProductCode).collect(Collectors.toSet());
            Set<String> productExistSet = Sets.newHashSet();
            reqVo.getDirectProductReqVos().forEach(x -> {
                if (productCodeSet.contains(x.getProductCode())) {
                    productExistSet.add(x.getProductCode());
                }
            });
            if (CollectionUtils.isNotEmpty(productExistSet)) {
                throw new BusinessException(productExistSet.toString() + "的物料以维护");
            }
        }
        if (CollectionUtils.isEmpty(reqVo.getDirectProductReqVos())) {
            throw new BusinessException("数据已存在");
        }
    }

    /**
     * 检验 数据重复
     *
     * @param entities
     */
    private void checkDataExistAndSave(List<KmsTotalWarehouseStoreEntity> entities) {
        /*if (CollectionUtils.isEmpty(entities)) {
            return;
        }
        Set<String> productIdSet = Sets.newHashSet();
        Set<String> bsId = Sets.newHashSet();
        entities.forEach(x -> {
            productIdSet.add(x.getProductId());
        });
        KmsTotalWarehouseStoreReqVo reqVo = new KmsTotalWarehouseStoreReqVo();
        reqVo.setProductIds(Lists.newArrayList(productIdSet));
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<KmsTotalWarehouseStoreEntity> storeEntities = kmsTotalWarehouseStoreMapper
                .selectListWithDataPurview(reqVo);
        Map<String, KmsTotalWarehouseStoreEntity> entityMap = Maps.newHashMap();
        storeEntities.forEach(x -> {
            entityMap.put(x.getBsDirectSystemId() + x.getProductId(), x);
        });
        List<KmsTotalWarehouseStoreEntity> saveEntities = Lists.newArrayList();
        entities.forEach(x -> {
            if (!entityMap.containsKey(x.getBsDirectSystemId() + x.getProductId())) {
                saveEntities.add(x);
            }
        });
        if (CollectionUtils.isEmpty(saveEntities)) {
            throw new BusinessException("选择的" + entities.size() + "家送达方已维护");
        }
        this.saveBatch(saveEntities);
        */
    }


    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsTotalWarehouseStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "请选择操作数据");
        this.checkNotEmpty(reqVo);
        KmsTotalWarehouseStoreEntity entityByDb = kmsTotalWarehouseStoreMapper.selectById(reqVo.getId());
        KmsTotalWarehouseStoreRespVo oldData = CrmBeanUtil.copy(entityByDb, KmsTotalWarehouseStoreRespVo.class);
        AssertUtils.isNotNull(entityByDb, "修改数据不存在");
        BeanUtils.copyProperties(reqVo, entityByDb);
        this.updateById(entityByDb);
        KmsTotalWarehouseStoreRespVo newData = this.query(reqVo.getId());
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 新增修改校验数据填写
     *
     * @param reqVo
     */
    private void checkNotEmpty(KmsTotalWarehouseStoreReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemId(), "直营体系不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(), "直营体系不能为空");
        AssertUtils.isNotEmpty(reqVo.getDirectStoreId(), "直营门店ID不能为空");
        AssertUtils.isNotEmpty(reqVo.getTerminalCode(), "企业门店编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getTerminalName(), "企业门店名称不能为空");
    }

    /**
     * 转换 待保存 数据
     *
     * @param reqVo
     */
    private List<KmsTotalWarehouseStoreEntity> convert(KmsTotalWarehouseStoreReqVo reqVo) {
        List<KmsTotalWarehouseStoreEntity> entities = Lists.newArrayList();
        if (org.springframework.util.CollectionUtils.isEmpty(reqVo.getDirectProductReqVos())) {
            entities.add(CrmBeanUtil.copy(reqVo, KmsTotalWarehouseStoreEntity.class));
        } else {
            reqVo.getDirectProductReqVos().forEach(o -> {
                KmsTotalWarehouseStoreEntity entity = CrmBeanUtil.copy(reqVo, KmsTotalWarehouseStoreEntity.class);
                entity.setDirectProductId(o.getId());
                entity.setProductCode(o.getProductCode());
                entity.setProductName(o.getProductName());
                entity.setKaProductCode(o.getKaProductCode());
                entities.add(entity);
            });
        }
        return entities;
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTotalWarehouseStoreEntity> kmsTotalWarehouseStoreEntities = kmsTotalWarehouseStoreMapper
                .selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsTotalWarehouseStoreEntities)) {
            kmsTotalWarehouseStoreEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsTotalWarehouseStoreEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为启用
        List<KmsTotalWarehouseStoreEntity> kmsTotalWarehouseStoreEntities = kmsTotalWarehouseStoreMapper.selectBatchIds(ids);
        List<KmsTotalWarehouseStoreRespVo> oldDataList = CrmBeanUtil.copyList(kmsTotalWarehouseStoreEntities, KmsTotalWarehouseStoreRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTotalWarehouseStoreEntities)) {
            kmsTotalWarehouseStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        List<KmsTotalWarehouseStoreRespVo> newDataList = CrmBeanUtil.copyList(kmsTotalWarehouseStoreEntities, KmsTotalWarehouseStoreRespVo.class);
        this.updateBatchById(kmsTotalWarehouseStoreEntities);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(),
                    newDataList.get(i).getId(), newDataList.get(i).getId(), oldDataList.get(i), newDataList.get(i));
        }
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        //设置状态为禁用
        List<KmsTotalWarehouseStoreEntity> kmsTotalWarehouseStoreEntities = kmsTotalWarehouseStoreMapper
                .selectBatchIds(ids);
        List<KmsTotalWarehouseStoreRespVo> oldDataList = CrmBeanUtil.copyList(kmsTotalWarehouseStoreEntities, KmsTotalWarehouseStoreRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTotalWarehouseStoreEntities)) {
            kmsTotalWarehouseStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        List<KmsTotalWarehouseStoreRespVo> newDataList = CrmBeanUtil.copyList(kmsTotalWarehouseStoreEntities, KmsTotalWarehouseStoreRespVo.class);
        this.updateBatchById(kmsTotalWarehouseStoreEntities);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(),
                    newDataList.get(i).getId(), newDataList.get(i).getId(), oldDataList.get(i), newDataList.get(i));
        }
    }
}
