package com.biz.crm.grabrule.helper;

import com.biz.crm.base.BusinessException;
import com.biz.crm.confadmin.service.IKmsAdminService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.grabrule.model.KmsGrabRuleParamEntity;
import com.biz.crm.grabrule.model.KmsGrabRuleStoreEntity;
import com.biz.crm.nebular.job.req.XxlJobInfoReqVo;
import com.biz.crm.nebular.kms.confadmin.base.BaseClassifyConstants;
import com.biz.crm.nebular.kms.confadmin.base.BaseDateFlagConstants;
import com.biz.crm.nebular.kms.confadmin.base.ComomStatus;
import com.biz.crm.nebular.kms.confadmin.req.BsGrabInterfaceParamReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.BsGrabInterfaceParamRespVo;
import com.biz.crm.nebular.kms.confadmin.resp.BsGrabInterfaceSubParamOptionRespVo;
import com.biz.crm.nebular.kms.grabrule.req.KmsGrabRuleParamReqVo;
import com.biz.crm.nebular.kms.grabrule.req.KmsGrabRuleReqVo;
import com.biz.crm.nebular.kms.grabrule.req.KmsGrabRuleStoreReqVo;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.supermarket.service.IKmsTenantryDirectCustomerOrgService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Project crm
 * @PackageName com.biz.crm.grabrule.helper
 * @ClassName KmsGrabRuleHelper
 * @Author Gavin
 * @Date 2021/5/7 下午7:02
 * @Description 抓单参数助手
 */
@Slf4j
@Component
public class KmsGrabRuleHelper {

    @Resource
    private IKmsAdminService kmsAdminService;

    @Resource
    private IKmsTenantryDirectCustomerOrgService kmsTenantryDirectCustomerOrgService;

    /**
     * 参数校验
     * @param reqVo
     */
    public void check(KmsGrabRuleReqVo reqVo) {
        AssertUtils.isNotNull(reqVo.getStartTime(),"开始时间[startTime]不能为空");
        AssertUtils.isNotNull(reqVo.getEndTime(),"结束时间[endTime]不能为空");
        AssertUtils.isNotNull(reqVo.getIntervalTime(),"时间间隔[intervalTime]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemId(),"直营体系Id[bsDirectSystemId]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemCode(),"直营体系编码[bsDirectSystemCode]不能为空");
        AssertUtils.isNotEmpty(reqVo.getBsDirectSystemName(),"直营体系名称[bsDirectSystemName]不能为空");
        AssertUtils.isNotEmpty(reqVo.getAccountId(),"登录账号ID[accountId]不能为空");
        AssertUtils.isNotEmpty(reqVo.getOrderType(),"单据类型orderType不能为空");
        AssertUtils.isNotNull(reqVo.getFileDownloadWay(),"单据下载方式不能为空");
        AssertUtils.isNotNull(reqVo.getFileClassify(),"单据下载方式不能为空");
        AssertUtils.isNotNull(reqVo.getOrderDownloadWay(),"单据下载方式不能为空");
        AssertUtils.isNotEmpty(reqVo.getRuleName(),"抓单规则名称不能为空");
        String isAllStore = reqVo.getIsAllStore();
        AssertUtils.isNotEmpty(reqVo.getIsAllStore(),"是否全部门店标示不能为空");
        if(StringUtils.isNotEmpty(isAllStore)){
            if(!YesNoEnum.yesNoEnum.YES.getValue().equals(isAllStore) && !YesNoEnum.yesNoEnum.NO.getValue().equals(isAllStore)){
                throw new BusinessException("是否全部门店标示数据错误（Y/N）");
            }
        }
        //门店
        List<KmsGrabRuleStoreReqVo> storeReqVos = reqVo.getRuleStoreList();
        if(YesNoEnum.yesNoEnum.NO.getValue().equals(isAllStore)){
            if(CollectionUtil.listEmpty(storeReqVos)) {
                throw new BusinessException("抓单门店不能为空");
            }else{
                storeReqVos.forEach(o -> {
                    AssertUtils.isNotEmpty(o.getKmsStoreId(),"抓单门店ID不能为空");
                });
            }
        }
        //抓单参数
        List<KmsGrabRuleParamReqVo> ruleParamReqVos = reqVo.getRuleParamList();
        if(CollectionUtil.listNotEmptyNotSizeZero(ruleParamReqVos)){
            ruleParamReqVos.forEach(o -> {
                AssertUtils.isNotEmpty(o.getFieldId(),"参数主键fieldId不能为空");
                AssertUtils.isNotEmpty(o.getFieldEn(),"抓单属性名[fieldEn]不能为空");
                AssertUtils.isNotEmpty(o.getFieldCn(),"抓单属性描述[fieldCn]不能为空");
            });
        }
    }

    /**
     * 构建门店实体
     * @param reqVo
     * @return
     */
    public List<KmsGrabRuleStoreEntity> buildRuleStore(KmsGrabRuleReqVo reqVo) {
        return reqVo.getRuleStoreList().stream().map(o -> {
            o.setGrabRuleId(reqVo.getId());
            return CrmBeanUtil.copy(o,KmsGrabRuleStoreEntity.class);
        }).collect(Collectors.toList());
    }

    /**
     * 构建参数实体
     * @param reqVo
     * @return
     */
    public List<KmsGrabRuleParamEntity> saveGrabRuleParams(KmsGrabRuleReqVo reqVo) {
        List<KmsGrabRuleParamEntity> ruleParamEntities = Lists.newArrayList();
        List<KmsGrabRuleParamReqVo> ruleParamReqVos = reqVo.getRuleParamList();
        if(CollectionUtil.listNotEmptyNotSizeZero(ruleParamReqVos)) {
            //查询参数信息
            KmsTenantryDirectCustomerOrgEntity directCustomerOrgEntity = kmsTenantryDirectCustomerOrgService.lambdaQuery()
                    .eq(KmsTenantryDirectCustomerOrgEntity::getId, reqVo.getBsDirectSystemId()).one();
            BsGrabInterfaceParamReqVo params = new BsGrabInterfaceParamReqVo();
            params.setDirectId(directCustomerOrgEntity.getDirectId());
            params.setOrderType(reqVo.getOrderType());
            List<BsGrabInterfaceParamRespVo> grabOrderParams = kmsAdminService.findGrabOrderParams(params);
            Map<String, BsGrabInterfaceParamRespVo> paramRespVoMap = grabOrderParams.stream()
                    .collect(Collectors.toMap(BsGrabInterfaceParamRespVo::getFieldId, Function.identity()));
            //参数数量必须相同
            if(grabOrderParams.size() != ruleParamReqVos.size()){
                throw new BusinessException("参数数量不符，请重新获取参数信息!");
            }
            ruleParamEntities = ruleParamReqVos.stream().map(o -> {
                BsGrabInterfaceParamRespVo paramRespVo = paramRespVoMap.get(o.getFieldId());
                //判断是否为必填
                if (ComomStatus.YES.equals(paramRespVo.getIsRequired()) && StringUtils.isEmpty(o.getFieldValue())) {
                    throw new BusinessException("参数[" + o.getFieldCn() + "]为必填字段!");
                }
                o.setGrabRuleId(reqVo.getId());
                //日期控件特殊处理
                if (BaseClassifyConstants.FieldClassify.DATE.getValue().equals(o.getFieldClassify())
                        || BaseClassifyConstants.FieldClassify.DATERANGE.getValue().equals(o.getFieldClassify())) {
                    this.dateHandle(o);
                }
                return CrmBeanUtil.copy(o, KmsGrabRuleParamEntity.class);
            }).collect(Collectors.toList());
        }else{
            throw new BusinessException("抓单参数信息不能为空！");
        }
        return ruleParamEntities;
    }

    /**
     * 日期控件单独处理
     * @param o
     */
    protected void dateHandle(KmsGrabRuleParamReqVo field) {
        List<BsGrabInterfaceSubParamOptionRespVo> subOptions = field.getSubOptions();
        if (CollectionUtil.listNotEmptyNotSizeZero(subOptions)) {
            if (StringUtils.isNotEmpty(field.getFieldValue())) {
                Instant now = Instant.now();
                subOptions.forEach(y -> {
                    if (BaseDateFlagConstants.DateFlagClassify.START.getValue().equals(y.getDateFlagClassify())) {
                        field.setStartTimeFieldEn(y.getOptionValue());
                    } else if (BaseDateFlagConstants.DateFlagClassify.END.getValue().equals(y.getDateFlagClassify())) {
                        field.setEndTimeFieldEn(y.getOptionValue());
                    }
                });
            }
        }else{
            throw new BusinessException("subOptions子参数信息不能为空");
        }
    }

    /**
     * 构建定时对象
     * @param cron
     * @return
     */
    public XxlJobInfoReqVo buildXxlJobInfo(KmsGrabRuleReqVo reqVo) {
        UserRedis userRedis = UserUtils.getUser();
        XxlJobInfoReqVo xxlJobInfoReqVo = new XxlJobInfoReqVo();
        xxlJobInfoReqVo.setJobDesc(reqVo.getRuleName());
        xxlJobInfoReqVo.setAuthor(userRedis.getUsername());
        xxlJobInfoReqVo.setScheduleConf(buildCron(reqVo));
        return xxlJobInfoReqVo;
    }

    /**
     * 构建定时任务表达式
     * @param reqVo
     * @return
     */
    protected static String buildCron(KmsGrabRuleReqVo reqVo) {
        Integer startTime = reqVo.getStartTime();
        Integer endTime = reqVo.getEndTime();
        Integer intervalTime = reqVo.getIntervalTime();
        if(startTime < 0 || startTime > 23 || endTime < 0 || endTime > 23 || startTime >= endTime){
            throw new BusinessException("开始时间结束时间规范错误(0-23)");
        }
        if(intervalTime <= 0 ){
            throw new BusinessException("时间间隔必须大于零");
        }
        int minute = 60;//60分钟
        //商
        int quotient = intervalTime / minute;
        //余数
        int remainder = intervalTime % minute;
        if(quotient == 0){
            endTime = endTime - 1;//避免执行时间超过结束时间
            int gap = (endTime - startTime) * minute;
            if(intervalTime > gap){
                throw new BusinessException("时间间隔错误!");
            }
            String hour = startTime == endTime ? startTime.toString() : startTime + "-" + endTime;
            //表达式 0 0/20 15-19 * * ?
            String cron = "0 0/" + remainder + " " + hour + " * * ?";
            return cron;
        }else{
            int gap = (endTime - startTime) * minute;
            if(intervalTime > gap){
                throw new BusinessException("时间间隔错误!");
            }
            if(remainder == 0){
                List<String> hours = Lists.newArrayList();
                //时间大于一小时
                for (int i = startTime,j = endTime;i <= j; i = i + quotient){
                    hours.add(String.valueOf(i));
                }
                String hour = hours.stream().collect(Collectors.joining(","));
                //表达式 0 0 15,19 * * ?
                String cron = "0 " + "0 " + hour + " * * ?";
                return cron;
            }else{
                endTime = endTime - 1;//避免执行时间超过结束时间
                List<String> hours = Lists.newArrayList();
                //时间大于一小时
                for (int i = startTime,j = endTime;i <= j; i = i + quotient){
                    hours.add(String.valueOf(i));
                }
                String hour = hours.stream().collect(Collectors.joining(","));
                //表达式 0 20 15,19 * * ?
                String cron = "0 " + remainder + " " + hour + " * * ?";
                return cron;
            }
        }
    }
}