package com.biz.crm.grabrule.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.mapper.KmsDirectThreadMapper;
import com.biz.crm.confadmin.service.IKmsAdminService;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.grabrule.helper.KmsGrabRuleHelper;
import com.biz.crm.grabrule.mapper.KmsGrabRuleMapper;
import com.biz.crm.grabrule.model.KmsGrabRuleEntity;
import com.biz.crm.grabrule.model.KmsGrabRuleParamEntity;
import com.biz.crm.grabrule.model.KmsGrabRuleStoreEntity;
import com.biz.crm.grabrule.service.IKmsGrabRuleParamService;
import com.biz.crm.grabrule.service.IKmsGrabRuleService;
import com.biz.crm.grabrule.service.IKmsGrabRuleStoreService;
import com.biz.crm.job.XxlJobFeign;
import com.biz.crm.nebular.job.req.XxlJobInfoReqVo;
import com.biz.crm.nebular.job.resp.XxlJobInfoRespVo;
import com.biz.crm.nebular.kms.confadmin.req.BsGrabInterfaceParamReqVo;
import com.biz.crm.nebular.kms.confadmin.resp.BsGrabInterfaceParamRespVo;
import com.biz.crm.nebular.kms.confadmin.resp.KmsDirectThreadRespVo;
import com.biz.crm.nebular.kms.grabrule.req.KmsGrabRuleReqVo;
import com.biz.crm.nebular.kms.grabrule.req.KmsGrabRuleStoreReqVo;
import com.biz.crm.nebular.kms.grabrule.resp.KmsGrabRuleRespVo;
import com.biz.crm.nebular.kms.grabrule.resp.KmsGrabRuleStoreRespVo;
import com.biz.crm.supermarket.model.KmsTenantryDirectCustomerOrgEntity;
import com.biz.crm.supermarket.service.IKmsTenantryDirectCustomerOrgService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 抓单规则配置表接口实现
 *
 * @author Gavin
 * @date 2021-04-16 17:00:42
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsGrabRuleServiceExpandImpl")
public class KmsGrabRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsGrabRuleMapper, KmsGrabRuleEntity> implements IKmsGrabRuleService {

    @Resource
    private KmsGrabRuleMapper kmsGrabRuleMapper;

    @Resource
    private IKmsGrabRuleStoreService kmsGrabRuleStoreService;

    @Resource
    private IKmsGrabRuleParamService kmsGrabRuleParamService;

    @Resource
    private IKmsAdminService kmsAdminService;

    @Resource
    private IKmsTenantryDirectCustomerOrgService kmsTenantryDirectCustomerOrgService;

    @Resource
    private XxlJobFeign xxlJobFeign;

    @Resource
    private KmsGrabRuleHelper kmsGrabRuleHelper;

    @Resource
    private KmsDirectThreadMapper kmsDirectThreadMapper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsGrabRuleRespVo> findList(KmsGrabRuleReqVo reqVo) {
        Page<KmsGrabRuleRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsGrabRuleRespVo> list = kmsGrabRuleMapper.findList(page, reqVo);
        return PageResult.<KmsGrabRuleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return kmsGrabRuleRespVo
     */
    @Override
    public KmsGrabRuleRespVo query(KmsGrabRuleReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "查询数据ID不能为空");
        KmsGrabRuleRespVo respVo = kmsGrabRuleMapper.findById(reqVo);
        AssertUtils.isNotNull(respVo, "未查询到数据信息!");
        String ruleId = respVo.getId();
        //查询门店
        String isAllStore = respVo.getIsAllStore();
        if (YesNoEnum.yesNoEnum.NO.getValue().equals(isAllStore)) {
            List<KmsGrabRuleStoreRespVo> ruleStoreRespVos = kmsGrabRuleStoreService.findByGrabRuleId(ruleId);
            respVo.setRuleStoreList(ruleStoreRespVos);
        }
        //查询单据下载方式
        KmsDirectThreadRespVo threadRespVo = kmsDirectThreadMapper.findByGrabRuleId(ruleId);
        AssertUtils.isNotNull(threadRespVo, "未查询到单据下载方式");
        respVo.setFileClassify(threadRespVo.getFileClassify());
        respVo.setOrderDownloadWay(threadRespVo.getOrderDownloadWay());
        respVo.setFileDownloadWay(threadRespVo.getFileDownloadWay());
        //查询参数
        KmsTenantryDirectCustomerOrgEntity directCustomerOrgEntity = kmsTenantryDirectCustomerOrgService.lambdaQuery()
                .eq(KmsTenantryDirectCustomerOrgEntity::getId, respVo.getBsDirectSystemId()).one();
        BsGrabInterfaceParamReqVo params = new BsGrabInterfaceParamReqVo();
        params.setDirectId(directCustomerOrgEntity.getDirectId());
        respVo.setDirectId(directCustomerOrgEntity.getDirectId());
        params.setOrderType(respVo.getOrderType());
        List<BsGrabInterfaceParamRespVo> grabOrderParams = kmsAdminService.findGrabOrderParams(params);
        List<KmsGrabRuleParamEntity> grabRuleParamEntities = kmsGrabRuleParamService.lambdaQuery()
                .eq(KmsGrabRuleParamEntity::getGrabRuleId, ruleId).list();
        Map<String, KmsGrabRuleParamEntity> paramMap = grabRuleParamEntities.stream()
                .collect(Collectors.toMap(KmsGrabRuleParamEntity::getFieldId, Function.identity()));
        grabOrderParams.forEach(o -> {
            KmsGrabRuleParamEntity pe = paramMap.get(o.getFieldId());
            if (!ObjectUtils.isEmpty(pe)) {
                o.setFieldValue(pe.getFieldValue());
                o.setOptionName(pe.getOptionName());
            }
        });
        respVo.setRuleParamList(grabOrderParams);
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsGrabRuleReqVo reqVo) {
        //参数校验
        kmsGrabRuleHelper.check(reqVo);
        KmsGrabRuleEntity entity = CrmBeanUtil.copy(reqVo, KmsGrabRuleEntity.class);
        this.save(entity);
        KmsGrabRuleRespVo newData = CrmBeanUtil.copy(entity, KmsGrabRuleRespVo.class);
        String grabRuleId = entity.getId();
        reqVo.setId(grabRuleId);
        //保存门店
        List<KmsGrabRuleStoreReqVo> ruleStoreReqVos = reqVo.getRuleStoreList();
        if (CollectionUtil.listNotEmptyNotSizeZero(ruleStoreReqVos)) {
            List<KmsGrabRuleStoreEntity> storeEntities = kmsGrabRuleHelper.buildRuleStore(reqVo);
            kmsGrabRuleStoreService.saveBatch(storeEntities);
            newData.setRuleStoreList(CrmBeanUtil.copyList(storeEntities, KmsGrabRuleStoreRespVo.class));
        }
        //保存参数
        List<KmsGrabRuleParamEntity> ruleParamEntities = kmsGrabRuleHelper.saveGrabRuleParams(reqVo);
        kmsGrabRuleParamService.saveBatch(ruleParamEntities);
        newData.setRuleParamList(CrmBeanUtil.copyList(ruleParamEntities, BsGrabInterfaceParamRespVo.class));
        //新增定时任务
        XxlJobInfoReqVo infoReqVo = kmsGrabRuleHelper.buildXxlJobInfo(reqVo);
        infoReqVo.setExecutorParam(grabRuleId);
        XxlJobInfoRespVo xxlJobInfoRespVo = ApiResultUtil.objResult(xxlJobFeign.add(infoReqVo));
        AssertUtils.isNotNull(xxlJobInfoRespVo, "生成任务失败");
        String taskId = xxlJobInfoRespVo.getId();
        if (ObjectUtils.isEmpty(taskId)) {
            throw new BusinessException("生成任务失败!");
        }
        entity.setTaskId(taskId);
        this.updateById(entity);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForAdd(menuCode.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsGrabRuleReqVo reqVo) {
        kmsGrabRuleHelper.check(reqVo);
        AssertUtils.isNotEmpty(reqVo.getId(), "数据ID不能为空");
        KmsGrabRuleEntity entity = this.getById(reqVo.getId());
        AssertUtils.isNotNull(entity, "未查询到数据信息");
        BeanUtils.copyProperties(reqVo, entity);
        KmsGrabRuleRespVo oldData = this.query(reqVo);
        this.updateById(entity);
        //删除门店信息
        kmsGrabRuleStoreService.lambdaUpdate().eq(KmsGrabRuleStoreEntity::getGrabRuleId, entity.getId()).remove();
        //保存门店
        List<KmsGrabRuleStoreReqVo> ruleStoreReqVos = reqVo.getRuleStoreList();
        if (CollectionUtil.listNotEmptyNotSizeZero(ruleStoreReqVos)) {
            List<KmsGrabRuleStoreEntity> storeEntities = kmsGrabRuleHelper.buildRuleStore(reqVo);
            kmsGrabRuleStoreService.saveBatch(storeEntities);
        }
        //删除参数信息
        kmsGrabRuleParamService.lambdaUpdate().eq(KmsGrabRuleParamEntity::getGrabRuleId, entity.getId()).remove();
        //保存参数
        List<KmsGrabRuleParamEntity> ruleParamEntities = kmsGrabRuleHelper.saveGrabRuleParams(reqVo);
        kmsGrabRuleParamService.saveBatch(ruleParamEntities);
        //更新定时任务
        XxlJobInfoReqVo infoReqVo = kmsGrabRuleHelper.buildXxlJobInfo(reqVo);
        infoReqVo.setId(entity.getTaskId());
        xxlJobFeign.update(infoReqVo);
        KmsGrabRuleRespVo newData = this.query(reqVo);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCode.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(KmsGrabRuleReqVo reqVo) {
        List<KmsGrabRuleEntity> kmsGrabRuleEntities = kmsGrabRuleMapper.selectBatchIds(reqVo.getIds());
        List<String> taskIds = kmsGrabRuleEntities.stream().map(KmsGrabRuleEntity::getTaskId).collect(Collectors.toList());
        this.removeByIds(reqVo.getIds());
        Result remove = xxlJobFeign.remove(taskIds);
    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsGrabRuleReqVo reqVo) {
        //设置状态为启用
        List<KmsGrabRuleEntity> kmsGrabRuleEntities = kmsGrabRuleMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsGrabRuleEntities)) {
            kmsGrabRuleEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        List<String> taskIds = kmsGrabRuleEntities.stream().map(KmsGrabRuleEntity::getTaskId).collect(Collectors.toList());
        this.updateBatchById(kmsGrabRuleEntities);
        xxlJobFeign.start(taskIds);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsGrabRuleReqVo reqVo) {
        //设置状态为禁用
        List<KmsGrabRuleEntity> kmsGrabRuleEntities = kmsGrabRuleMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsGrabRuleEntities)) {
            kmsGrabRuleEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        List<String> taskIds = kmsGrabRuleEntities.stream().map(KmsGrabRuleEntity::getTaskId).collect(Collectors.toList());
        this.updateBatchById(kmsGrabRuleEntities);
        xxlJobFeign.pause(taskIds);
    }
}
