package com.biz.crm.returnform.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.nebular.kms.returnform.req.KmsReturnFormReqVo;
import com.biz.crm.nebular.kms.returnform.resp.KmsReturnFormExportRespVo;
import com.biz.crm.nebular.kms.returnform.resp.KmsReturnFormRespVo;
import com.biz.crm.returnform.service.IKmsReturnFormService;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.util.List;

/**
 * 退货单信息表
 *
 * @author Gavin
 * @date 2021-05-12 18:23:07
 */
@Slf4j
@RestController
@RequestMapping("/kmsreturnform")
@Api(tags = "商超单据-退货单")
public class KmsReturnFormController {
    @Autowired
    private IKmsReturnFormService kmsReturnFormService;

    @Resource
    private RedissonUtil redissonUtil;

    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/list")
    @CrmLog
    public Result<PageResult<KmsReturnFormRespVo>> list(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        PageResult<KmsReturnFormRespVo> result = kmsReturnFormService.findList(kmsReturnFormReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "查询列表导出")
    @PostMapping("/listForExport")
    @CrmLog
    public Result<PageResult<KmsReturnFormExportRespVo>> listForExport(@RequestBody KmsReturnFormReqVo reqVo) {
        PageResult<KmsReturnFormExportRespVo> result = kmsReturnFormService.listForExport(reqVo);
        return Result.ok(result);
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @PostMapping("/query")
    @CrmLog
    public Result<KmsReturnFormRespVo> query(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        KmsReturnFormRespVo respVo = kmsReturnFormService.query(kmsReturnFormReqVo);
        return Result.ok(respVo);
    }

    /**
     * 新增
     */
    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        kmsReturnFormService.save(kmsReturnFormReqVo);
        return Result.ok();
    }

    /**
     * 更新
     */
    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        kmsReturnFormService.update(kmsReturnFormReqVo);
        return Result.ok("修改成功");
    }

    /**
     * 删除
     */
    @ApiOperation(value = "删除")
    @PostMapping("/delete")
    @CrmLog
    public Result delete(@RequestBody List<String> ids) {
        kmsReturnFormService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        kmsReturnFormService.enableBatch(kmsReturnFormReqVo);
        return Result.ok("启用成功");
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody KmsReturnFormReqVo kmsReturnFormReqVo) {
        kmsReturnFormService.disableBatch(kmsReturnFormReqVo);
        return Result.ok("禁用成功");
    }

    /**
     * 手动转退货单
     */
    @ApiOperation(value = "手动转退货单")
    @PostMapping("/manualTrans")
    @CrmLog
    public Result manualTrans(@RequestBody List<String> ids) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_TRANS_FORM);
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                KmsReturnFormReqVo kmsReturnFormReqVo = new KmsReturnFormReqVo();
                kmsReturnFormReqVo.setIds(ids);
                kmsReturnFormService.manualTrans(kmsReturnFormReqVo);
            } else {
                throw new BusinessException("有人正在进行单据转换，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw new BusinessException("操作失败:" + e.getMessage());
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }
}
