package com.biz.crm.salesdata.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.nebular.kms.salesdata.req.KmsSalesDataReqVo;
import com.biz.crm.nebular.kms.salesdata.resp.KmsSalesDataRespVo;
import com.biz.crm.salesdata.service.IKmsSalesDataService;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 销售数据表
 *
 * @author gavin
 * @date 2021-11-24 14:22:40
 */
@Slf4j
@RestController
@RequestMapping("/kmssalesdata")
@Api(tags = "商超单据-销售数据")
public class KmsSalesDataController {
    @Autowired
    private IKmsSalesDataService kmsSalesDataService;

    @Resource
    private RedissonUtil redissonUtil;

    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/list")
    @CrmLog
    public Result<PageResult<KmsSalesDataRespVo>> list(@RequestBody KmsSalesDataReqVo kmsSalesDataReqVo) {
        PageResult<KmsSalesDataRespVo> result = kmsSalesDataService.findList(kmsSalesDataReqVo);
        return Result.ok(result);
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @PostMapping("/query")
    @CrmLog
    public Result<KmsSalesDataRespVo> query(@RequestParam(value = "id") String id) {
        KmsSalesDataRespVo respVo = kmsSalesDataService.query(id);
        return Result.ok(respVo);
    }

    /**
     * 新增
     */
    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody KmsSalesDataReqVo kmsSalesDataReqVo) {
        kmsSalesDataService.save(kmsSalesDataReqVo);
        return Result.ok();
    }

    /**
     * 更新
     */
    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody KmsSalesDataReqVo kmsSalesDataReqVo) {
        kmsSalesDataService.update(kmsSalesDataReqVo);
        return Result.ok("修改成功");
    }

    /**
     * 删除
     */
    @ApiOperation(value = "删除")
    @PostMapping("/delete")
    @CrmLog
    public Result deleteBatch(@RequestBody List<String> ids) {
        kmsSalesDataService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody KmsSalesDataReqVo kmsSalesDataReqVo) {
        kmsSalesDataService.enableBatch(kmsSalesDataReqVo);
        return Result.ok("启用成功");
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody KmsSalesDataReqVo kmsSalesDataReqVo) {
        kmsSalesDataService.disableBatch(kmsSalesDataReqVo);
        return Result.ok("禁用成功");
    }

    /**
     * 手动转换销售数据
     */
    @ApiOperation(value = "手动转换销售数据")
    @PostMapping("/manualTrans")
    @CrmLog
    public Result manualTrans(@RequestBody List<String> ids) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_TRANS_FORM + "_" + KmsAdminEnum.OrderType.SALES_DATA.getValue());
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                KmsSalesDataReqVo kmsSalesDataReqVo = new KmsSalesDataReqVo();
                kmsSalesDataReqVo.setIds(ids);
                kmsSalesDataService.manualTrans(kmsSalesDataReqVo);
            } else {
                throw new BusinessException("有人正在进行数据转换，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw e;
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }
}
