package com.biz.crm.stockdata.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.constant.KmsConstant;
import com.biz.crm.eunm.kms.KmsAdminEnum;
import com.biz.crm.nebular.kms.stockdata.req.KmsStockDataReqVo;
import com.biz.crm.nebular.kms.stockdata.resp.KmsStockDataRespVo;
import com.biz.crm.stockdata.service.IKmsStockDataService;
import com.biz.crm.util.RedissonUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 库存数据表
 *
 * @author gavin
 * @date 2021-11-24 14:22:01
 */
@Slf4j
@RestController
@RequestMapping("/kmsstockdata")
@Api(tags = "商超单据-库存数据")
public class KmsStockDataController {
    @Autowired
    private IKmsStockDataService kmsStockDataService;

    @Resource
    private RedissonUtil redissonUtil;

    /**
     * 列表
     */
    @ApiOperation(value = "查询列表")
    @PostMapping("/list")
    @CrmLog
    @CrmDictMethod
    public Result<PageResult<KmsStockDataRespVo>> list(@RequestBody KmsStockDataReqVo kmsStockDataReqVo) {
        PageResult<KmsStockDataRespVo> result = kmsStockDataService.findList(kmsStockDataReqVo);
        return Result.ok(result);
    }

    /**
     * 查询
     */
    @ApiOperation(value = "查询")
    @PostMapping("/query")
    @CrmLog
    @CrmDictMethod
    public Result<KmsStockDataRespVo> query(@RequestParam(value = "id", required = false) String id) {
        KmsStockDataRespVo respVo = kmsStockDataService.query(id);
        return Result.ok(respVo);
    }

    /**
     * 新增
     */
    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody KmsStockDataReqVo kmsStockDataReqVo) {
        kmsStockDataService.save(kmsStockDataReqVo);
        return Result.ok();
    }

    /**
     * 更新
     */
    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody KmsStockDataReqVo kmsStockDataReqVo) {
        kmsStockDataService.update(kmsStockDataReqVo);
        return Result.ok("修改成功");
    }

    /**
     * 删除
     */
    @ApiOperation(value = "删除")
    @PostMapping("/delete")
    @CrmLog
    public Result deleteBatch(@RequestBody List<String> ids) {
        kmsStockDataService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    /**
     * 启用
     */
    @ApiOperation(value = "启用")
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody KmsStockDataReqVo kmsStockDataReqVo) {
        kmsStockDataService.enableBatch(kmsStockDataReqVo);
        return Result.ok("启用成功");
    }

    /**
     * 禁用
     */
    @ApiOperation(value = "禁用")
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody KmsStockDataReqVo kmsStockDataReqVo) {
        kmsStockDataService.disableBatch(kmsStockDataReqVo);
        return Result.ok("禁用成功");
    }

    /**
     * 手动转换库存数据
     */
    @ApiOperation(value = "手动转换库存数据")
    @PostMapping("/manualTrans")
    @CrmLog
    public Result manualTrans(@RequestBody List<String> ids) {
        RLock rLock = null;
        try {
            rLock = redissonUtil.createLock(KmsConstant.MANUAL_TRANS_FORM + "_" + KmsAdminEnum.OrderType.INVENTORY_DATA.getValue());
            if (redissonUtil.tryLock(rLock, 0, 3600)) {
                KmsStockDataReqVo kmsStockDataReqVo = new KmsStockDataReqVo();
                kmsStockDataReqVo.setIds(ids);
                kmsStockDataService.manualTrans(kmsStockDataReqVo);
            } else {
                throw new BusinessException("有人正在进行数据转换，请稍后重试!");
            }
        } catch (BusinessException e) {
            throw e;
        } catch (Exception e) {
            throw e;
        } finally {
            redissonUtil.unLock(rLock);
        }
        return Result.ok("操作成功");
    }
}
