package com.biz.crm.stockdata.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.kms.rawdata.resp.KmsInventoryDataFormDetailDwRespVo;
import com.biz.crm.nebular.kms.stockdata.req.KmsStockDataReqVo;
import com.biz.crm.nebular.kms.stockdata.resp.KmsStockDataRespVo;
import com.biz.crm.stockdata.helper.KmsStockDataHelper;
import com.biz.crm.stockdata.mapper.KmsStockDataMapper;
import com.biz.crm.stockdata.model.KmsStockDataEntity;
import com.biz.crm.stockdata.service.IKmsStockDataService;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectCustomerOrgMapper;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.List;

/**
 * 库存数据表接口实现
 *
 * @author gavin
 * @date 2021-11-24 14:22:01
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsStockDataServiceExpandImpl")
public class KmsStockDataServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsStockDataMapper, KmsStockDataEntity> implements IKmsStockDataService {

    @Resource
    private KmsStockDataMapper kmsStockDataMapper;

    @Resource
    private KmsStockDataHelper kmsStockDataHelper;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private KmsTenantryDirectCustomerOrgMapper orgMapper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsStockDataRespVo> findList(KmsStockDataReqVo reqVo) {
        Page<KmsStockDataRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsStockDataRespVo> list = kmsStockDataMapper.findList(page, reqVo);
        return PageResult.<KmsStockDataRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsStockDataRespVo
     */
    @Override
    public KmsStockDataRespVo query(String id) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsStockDataReqVo reqVo) {
        KmsStockDataEntity entity = CrmBeanUtil.copy(reqVo, KmsStockDataEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsStockDataReqVo reqVo) {
        KmsStockDataEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {

    }

    /**
     * 启用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(KmsStockDataReqVo reqVo) {
        //设置状态为启用
        List<KmsStockDataEntity> kmsStockDataEntities = kmsStockDataMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsStockDataEntities)) {
            kmsStockDataEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsStockDataEntities);
    }

    /**
     * 禁用
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(KmsStockDataReqVo reqVo) {
        //设置状态为禁用
        List<KmsStockDataEntity> kmsStockDataEntities = kmsStockDataMapper.selectBatchIds(reqVo.getIds());
        if (CollectionUtils.isNotEmpty(kmsStockDataEntities)) {
            kmsStockDataEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsStockDataEntities);
    }

    /**
     * 自动转换库存数据
     *
     * @param o
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoTransData(KmsInventoryDataFormDetailDwRespVo dwRespVo) {
        //匹配处理库存版本信息
        KmsStockDataEntity stockDataEntity = kmsStockDataHelper.transFormVersion(dwRespVo);
        if (!ObjectUtils.isEmpty(stockDataEntity)) {
            //数据转换
            kmsStockDataHelper.transData(stockDataEntity);
            //保存信息
            this.saveOrUpdate(stockDataEntity);
        }
    }

    /**
     * 手动转换库存数据
     *
     * @param kmsStockDataReqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void manualTrans(KmsStockDataReqVo kmsStockDataReqVo) {
        List<String> ids = kmsStockDataReqVo.getIds();
        AssertUtils.isNotEmpty(ids, "请选择待转换数据");
        //查询验收单状态，待确认、转换失败
        List<KmsStockDataEntity> kmsStockDataEntities = this.lambdaQuery()
                .in(KmsStockDataEntity::getId, ids).list();
        List<KmsInventoryDataFormDetailDwRespVo> detailDwRespVos = CrmBeanUtil.copyList(kmsStockDataEntities, KmsInventoryDataFormDetailDwRespVo.class);
        detailDwRespVos.forEach(dwRespVo -> {
            //匹配处理验收单版本信息
            KmsStockDataEntity kmsStockDataEntity = kmsStockDataHelper.transFormVersion(dwRespVo);
            if (!ObjectUtils.isEmpty(kmsStockDataEntity)) {
                //数据转换
                kmsStockDataHelper.transData(kmsStockDataEntity);
                //保存信息
                this.saveOrUpdate(kmsStockDataEntity);
            }
        });
    }
}
