package com.biz.crm.supermarket.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.confadmin.model.KmsDirectStoreAreaEntity;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.exception.CommonException;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.kms.supermarket.req.KmsTenantryDirectStoreReqVo;
import com.biz.crm.nebular.kms.supermarket.resp.KmsTenantryDirectStoreRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.supermarket.mapper.KmsTenantryDirectStoreMapper;
import com.biz.crm.supermarket.model.KmsTenantryDirectStoreEntity;
import com.biz.crm.supermarket.service.IKmsTenantryDirectStoreService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.Result;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;

/**
 * 企业门店与商超门店关系接口实现
 *
 * @author Gavin
 * @date 2021-04-15 11:31:07
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "KmsTenantryDirectStoreServiceExpandImpl")
public class KmsTenantryDirectStoreServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<KmsTenantryDirectStoreMapper, KmsTenantryDirectStoreEntity> implements IKmsTenantryDirectStoreService {

    @Resource
    private KmsTenantryDirectStoreMapper kmsTenantryDirectStoreMapper;

    @Resource
    private KmsTenantryDirectHelper helper;

    @Resource
    private MdmTerminalFeign mdmTerminalFeign;

    @Resource
    private CrmLogSendUtil crmLogSendUtil;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<KmsTenantryDirectStoreRespVo> findList(KmsTenantryDirectStoreReqVo reqVo) {
        Page<KmsTenantryDirectStoreRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<KmsTenantryDirectStoreRespVo> list = kmsTenantryDirectStoreMapper.findList(page, reqVo);
        if (CollectionUtils.isNotEmpty(list)) {
            helper.convertStoreData(list);
        }
        return PageResult.<KmsTenantryDirectStoreRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     *
     * @param id
     * @return kmsTenantryDirectStoreRespVo
     */
    @Override
    public KmsTenantryDirectStoreRespVo query(String id) {
        AssertUtils.isNotEmpty(id, CommonException.IDS_NULL);
        KmsTenantryDirectStoreReqVo reqVo = new KmsTenantryDirectStoreReqVo();
        reqVo.setId(id);
        List<KmsTenantryDirectStoreRespVo> respVos = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(respVos)) {
            throw new BusinessException("查询数据不存在");
        }
        return respVos.get(0);
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(KmsTenantryDirectStoreReqVo reqVo) {
        helper.checkSaveStoreData(reqVo);
        KmsTenantryDirectStoreEntity entity = CrmBeanUtil.copy(reqVo, KmsTenantryDirectStoreEntity.class);
        this.save(entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        KmsTenantryDirectStoreRespVo newData = CrmBeanUtil.copy(entity, KmsTenantryDirectStoreRespVo.class);
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), newData.getId(), newData.getId(), newData);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(KmsTenantryDirectStoreReqVo reqVo) {
        helper.checkStoreUpdate(reqVo);
        helper.checkSaveStoreData(reqVo);
        KmsTenantryDirectStoreEntity entity = CrmBeanUtil.copy(reqVo, KmsTenantryDirectStoreEntity.class);
        KmsTenantryDirectStoreRespVo oldData = this.query(reqVo.getId());
        kmsTenantryDirectStoreMapper.updateByKey(entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        KmsTenantryDirectStoreRespVo newData = this.query(reqVo.getId());
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), newData.getId(), newData.getId(), oldData, newData);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectStoreEntity> kmsTenantryDirectStoreEntities = kmsTenantryDirectStoreMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectStoreEntities)) {
            kmsTenantryDirectStoreEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectStoreEntities);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        //设置状态为启用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectStoreEntity> kmsTenantryDirectStoreEntities = kmsTenantryDirectStoreMapper.selectBatchIds(ids);
        List<KmsTenantryDirectStoreRespVo> oldDataList = CrmBeanUtil.copyList(kmsTenantryDirectStoreEntities, KmsTenantryDirectStoreRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectStoreEntities)) {
            kmsTenantryDirectStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectStoreEntities);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<KmsTenantryDirectStoreRespVo> newDataList = CrmBeanUtil.copyList(kmsTenantryDirectStoreEntities, KmsTenantryDirectStoreRespVo.class);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(i).getId(), newDataList.get(i).getId(), oldDataList.get(i), newDataList.get(i));
        }
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        AssertUtils.isNotEmpty(ids, CommonException.IDS_NULL);
        List<KmsTenantryDirectStoreEntity> kmsTenantryDirectStoreEntities = kmsTenantryDirectStoreMapper.selectBatchIds(ids);
        List<KmsTenantryDirectStoreRespVo> oldDataList = CrmBeanUtil.copyList(kmsTenantryDirectStoreEntities, KmsTenantryDirectStoreRespVo.class);
        if (CollectionUtils.isNotEmpty(kmsTenantryDirectStoreEntities)) {
            kmsTenantryDirectStoreEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(kmsTenantryDirectStoreEntities);
        Object menuCode = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        List<KmsTenantryDirectStoreRespVo> newDataList = CrmBeanUtil.copyList(kmsTenantryDirectStoreEntities, KmsTenantryDirectStoreRespVo.class);
        for (int i = 0; i < newDataList.size(); i++) {
            crmLogSendUtil.sendForUpdate(menuCode.toString(), newDataList.get(i).getId(), newDataList.get(i).getId(), oldDataList.get(i), newDataList.get(i));
        }
    }


    /**
     * 企业门店与商超门店关系新增获取企业门店集合
     *
     * @param mdmTerminalVo
     * @return
     */
    @Override
    public List<MdmTerminalVo> findTerminalList(MdmTerminalVo mdmTerminalVo) {
        mdmTerminalVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        mdmTerminalVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        Result<List<MdmTerminalVo>> listResult;
        try {
            listResult = mdmTerminalFeign.listCondition(mdmTerminalVo);
        } catch (Exception e) {
            log.error("{}", e);
            throw new BusinessException(" 新增企业门店与商超门店关系 获取门店列表,调用mdm接口失败");
        }
//        AssertUtils.isNotEmpty(mdmTerminalVo.getDirectSystemId(), "直营体系编码不能为空");
//        List<MdmTerminalVo> list = ApiResultUtil.objResult(listResult, true)
//                .stream().filter(o -> StringUtils.equals(o.getDirectSystemId(), mdmTerminalVo.getDirectSystemId()))
//                .collect(Collectors.toList());
        return ApiResultUtil.objResult(listResult, true);
    }
}
