package com.biz.crm.log.util;

import com.alibaba.fastjson.JSON;
import com.biz.crm.base.BusinessException;
import com.biz.crm.eunm.log.LogEunm;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.log.CrmLogDto;
import com.biz.crm.util.DateUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2020/12/16 16:55
 **/
@Component
public class CrmLogSendUtil {

    @Autowired
    private RocketMQProducer rocketMQProducer;

    /**
     * 修改发送处理日志
     * @param templateCode 日志模板编码
     * @param onlyKey 记录唯一键（一般是id）
     * @param modularCode 模块编码（取值字典）
     * @param oldObject 更新之前的对象
     * @param newObject 更新之后的对象
     */
    public void sendForUpdate(String templateCode,String onlyKey,String modularCode, Object oldObject,Object newObject,String operationUserId,String operationUserName){
        if(null == oldObject || null == newObject){
            throw new BusinessException("更新日志比对前/后值为空！");
        }
        if(StringUtils.isEmpty(modularCode)){
            throw new BusinessException("模块编码为空！");
        }
        if(StringUtils.isEmpty(templateCode)){
            throw new BusinessException("日志模板编码为空！");
        }
        if(StringUtils.isEmpty(onlyKey)){
            throw new BusinessException("日志唯一标记为空！");
        }
        if(oldObject instanceof Collection || newObject instanceof Collection){
            throw new BusinessException("暂不支持集合日志对比！");
        }
        CrmLogDto crmLogDto = new CrmLogDto(templateCode,LogEunm.OperationTypeEunm.UPDATE.getCode(),onlyKey,modularCode,oldObject,newObject,operationUserId,operationUserName);
        crmLogDto.setOperationTime(DateUtil.format(DateUtil.getDate(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        RocketMQMessageBody msg = new RocketMQMessageBody();
        msg.setTag(RocketMQConstant.CRM_MQ_TAG.CRMLOG);
        msg.setMsgBody(JSON.toJSONString(crmLogDto));
        rocketMQProducer.convertAndSend(msg);
    }

    /**
     * 删除发送处理日志
     * @param modularCode 模块编码（取值字典）
     * @param map value --> 删除的对象Json字符串,key删除对象的描述（用于搜索）
     */
    public void sendForDel(String modularCode, Map<String,Object> map,String operationUserId,String operationUserName){
        if(CollectionUtils.isEmpty(map)){
            return;
        }
        if(StringUtils.isEmpty(modularCode)){
            throw new BusinessException("模块编码为空！");
        }
        CrmLogDto crmLogDto = new CrmLogDto(LogEunm.OperationTypeEunm.DEL.getCode(),
                "",
                map.keySet().iterator().next(),
                modularCode,
                map.values().iterator().next(),
                "",
                operationUserId,
                operationUserName);
        crmLogDto.setOperationTime(DateUtil.format(DateUtil.getDate(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        RocketMQMessageBody msg = new RocketMQMessageBody();
        msg.setTag(RocketMQConstant.CRM_MQ_TAG.CRMLOG);
        msg.setMsgBody(JSON.toJSONString(crmLogDto));
        rocketMQProducer.convertAndSend(msg);
    }


    /**
     * 批量删除发送处理日志
     * @param modularCode 模块编码（取值字典）
     * @param list map :
     *             value --> 删除的对象Json字符串,key删除对象的描述（用于搜索）
     */
    public void sendForDelBatch(String modularCode, List<Map<String,Object>> list,String operationUserId,String operationUserName){
        if(CollectionUtils.isEmpty(list)){
            return;
        }
        if(StringUtils.isEmpty(modularCode)){
            throw new BusinessException("模块编码为空！");
        }
        CrmLogDto crmLogDto = new CrmLogDto(LogEunm.OperationTypeEunm.DEL.getCode(),
                "",
                "",
                modularCode,
                list,
                "",
                operationUserId,
                operationUserName);
        crmLogDto.setOperationTime(DateUtil.format(DateUtil.getDate(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        RocketMQMessageBody msg = new RocketMQMessageBody();
        msg.setTag(RocketMQConstant.CRM_MQ_TAG.CRMLOG);
        msg.setMsgBody(JSON.toJSONString(crmLogDto));
        rocketMQProducer.convertAndSend(msg);
    }


    /**
     * 新增发送处理日志
     * @param modularCode 模块编码（取值字典）
     * @param map value --> 删除的对象Json字符串,key -->删除对象的描述（用于搜索）
     */
    public void sendForAdd(String modularCode, Map<String,Object> map,String operationUserId,String operationUserName){
        if(CollectionUtils.isEmpty(map)){
            return;
        }
        if(StringUtils.isEmpty(modularCode)){
            throw new BusinessException("模块编码为空！");
        }
        CrmLogDto crmLogDto = new CrmLogDto(LogEunm.OperationTypeEunm.ADD.getCode(),
                "",
                map.keySet().iterator().next(),
                modularCode,
                "",
                map.values().iterator().next(),
                operationUserId,
                operationUserName);
        crmLogDto.setOperationTime(DateUtil.format(DateUtil.getDate(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        RocketMQMessageBody msg = new RocketMQMessageBody();
        msg.setTag(RocketMQConstant.CRM_MQ_TAG.CRMLOG);
        msg.setMsgBody(JSON.toJSONString(crmLogDto));
        rocketMQProducer.convertAndSend(msg);
    }

    /**
     * 批量新增发送处理日志
     * @param modularCode 模块编码（取值字典）
     * @param list map :
     *             value --> 新增的对象Json字符串,key --> 新增对象的描述（用于搜索）
     */
    public void sendForAddBatch(String modularCode, List<Map<String,Object>> list,String operationUserId,String operationUserName){
        if(CollectionUtils.isEmpty(list)){
            return;
        }
        if(StringUtils.isEmpty(modularCode)){
            throw new BusinessException("模块编码为空！");
        }
        for(Map<String,Object> map : list){
            this.sendForAdd(modularCode,map,operationUserId,operationUserName);
        }
        CrmLogDto crmLogDto = new CrmLogDto(LogEunm.OperationTypeEunm.ADD.getCode(),
                "",
                "",
                modularCode,
                "",
                list,
                operationUserId,
                operationUserName);
        crmLogDto.setOperationTime(DateUtil.format(DateUtil.getDate(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        RocketMQMessageBody msg = new RocketMQMessageBody();
        msg.setTag(RocketMQConstant.CRM_MQ_TAG.CRMLOG);
        msg.setMsgBody(JSON.toJSONString(crmLogDto));
        rocketMQProducer.convertAndSend(msg);
    }

}
