package com.biz.crm.admin.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.admin.core.complete.XxlJobCompleter;
import com.biz.crm.admin.core.context.XxlJobHelper;
import com.biz.crm.admin.core.exception.XxlJobException;
import com.biz.crm.admin.core.model.XxlJobGroup;
import com.biz.crm.admin.core.model.XxlJobInfo;
import com.biz.crm.admin.core.model.XxlJobLog;
import com.biz.crm.admin.core.scheduler.XxlJobScheduler;
import com.biz.crm.admin.core.util.I18nUtil;
import com.biz.crm.admin.mapper.XxlJobGroupDao;
import com.biz.crm.admin.mapper.XxlJobInfoDao;
import com.biz.crm.admin.mapper.XxlJobLogDao;
import com.biz.crm.common.PageResult;
import com.biz.crm.core.biz.ExecutorBiz;
import com.biz.crm.core.biz.model.KillParam;
import com.biz.crm.core.biz.model.LogParam;
import com.biz.crm.core.biz.model.LogResult;
import com.biz.crm.core.biz.model.ReturnT;
import com.biz.crm.core.util.DateUtil;
import com.biz.crm.nebular.job.req.XxlJobLogReqVo;
import com.biz.crm.nebular.job.resp.XxlJobLogRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.List;

/**
 * index controller
 *
 * @author xuxueli 2015-12-19 16:13:16
 */
@Controller
@RequestMapping("/joblog")
@Api(tags = "调度日志表;")
public class JobLogController {
    private static Logger logger = LoggerFactory.getLogger(JobLogController.class);

    @Resource
    private XxlJobGroupDao xxlJobGroupDao;
    @Resource
    public XxlJobInfoDao xxlJobInfoDao;
    @Resource
    public XxlJobLogDao xxlJobLogDao;


    @Resource
    private XxlJobHelper xxlJobHelper;

        public String index(HttpServletRequest request, Model model, @RequestParam(required = false, defaultValue = "0") String jobId) {

        // 执行器列表
        List<XxlJobGroup> jobGroupList_all = xxlJobGroupDao.findAll();

        // filter group
        List<XxlJobGroup> jobGroupList = JobInfoController.filterJobGroupByRole(request, jobGroupList_all);
        if (jobGroupList == null || jobGroupList.size() == 0) {
            throw new XxlJobException(I18nUtil.getString("jobgroup_empty"));
        }

        model.addAttribute("JobGroupList", jobGroupList);

        // 任务
        if (StringUtils.isNotEmpty(jobId)) {
            XxlJobInfo jobInfo = xxlJobInfoDao.loadById(jobId);
            if (jobInfo == null) {
                throw new RuntimeException(I18nUtil.getString("jobinfo_field_id") + I18nUtil.getString("system_unvalid"));
            }

            model.addAttribute("jobInfo", jobInfo);

            // valid permission
            JobInfoController.validPermission(request, jobInfo.getJobGroup());
        }

        return "joblog/joblog.index";
    }

    @PostMapping("/getJobsByGroup")
    @ResponseBody
    @ApiOperation("根据执行器获取任务列表")
    public Result<List<XxlJobInfo>> getJobsByGroup(@RequestBody XxlJobLogReqVo vo) {
        List<XxlJobInfo> list = xxlJobInfoDao.getJobsByGroup(vo.getJobGroup());
        return Result.ok(list);
    }

    @PostMapping("/pageList")
    @ResponseBody
    @ApiOperation("获取调度日志列表")
    public Result<PageResult<XxlJobLogRespVo>> pageList(@RequestBody XxlJobLogReqVo vo) {

        // valid permission
        // 仅管理员支持查询全部；普通用户仅支持查询有权限的 jobGroup
        //JobInfoController.validPermission(request, jobGroup);

        Page<XxlJobLogRespVo> page = PageUtil.buildPage(vo.getPageNum(), vo.getPageSize());
        List<XxlJobLogRespVo> list = xxlJobLogDao.pageList(page, vo);
        return Result.ok(PageResult.<XxlJobLogRespVo>builder()
                .data(xxlJobHelper.convertJobLog(list))
                .count(page.getTotal())
                .build());
    }


    @PostMapping("/logDetailCat")
    @ResponseBody
    @ApiOperation("获取执行日志")
    public Result logDetailCat(@RequestBody XxlJobLogReqVo vo) {
        try {
            ExecutorBiz executorBiz = XxlJobScheduler.getExecutorBiz(vo.getExecutorAddress());
            ReturnT<LogResult> logResult = executorBiz.log(new LogParam(vo.getTriggerTime().getTime(), vo.getLogId(), vo.getFromLineNum()));

            // is end
            if (logResult.getContent() != null && logResult.getContent().getFromLineNum() > logResult.getContent().getToLineNum()) {
                XxlJobLog jobLog = xxlJobLogDao.load(vo.getLogId());
                if (jobLog.getHandleCode() > 0) {
                    logResult.getContent().setEnd(true);
                }
            }

            return Result.ok(logResult.getContent());
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    @PostMapping("/logKill")
    @ResponseBody
    @ApiOperation("终止任务")
    public Result logKill(@RequestBody XxlJobLogReqVo vo) {
        // base check
        XxlJobLog log = xxlJobLogDao.load(vo.getId());
        XxlJobInfo jobInfo = xxlJobInfoDao.loadById(log.getJobId());
        AssertUtils.isNotNull(jobInfo, I18nUtil.getString("jobinfo_glue_jobid_unvalid"));
        AssertUtils.isTrue(ReturnT.SUCCESS_CODE != log.getTriggerCode(), I18nUtil.getString("joblog_kill_log_limit"));
        // request of kill
        ReturnT<String> runResult = null;
        try {
            ExecutorBiz executorBiz = XxlJobScheduler.getExecutorBiz(log.getExecutorAddress());
            runResult = executorBiz.kill(new KillParam(jobInfo.getId()));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            runResult = new ReturnT<String>(500, e.getMessage());
        }

        if (ReturnT.SUCCESS_CODE == runResult.getCode()) {
            log.setHandleCode(ReturnT.FAIL_CODE);
            log.setHandleMsg(I18nUtil.getString("joblog_kill_log_byman") + ":" + (runResult.getMsg() != null ? runResult.getMsg() : ""));
            log.setHandleTime(new Date());
            XxlJobCompleter.updateHandleInfoAndFinish(log);
            return Result.ok(runResult.getMsg());
        } else {
            return Result.error(runResult.getMsg());
        }
    }

    @PostMapping("/clearLog")
    @ResponseBody
    @ApiOperation("清理日志")
    public ReturnT<String> clearLog(String jobGroup, String jobId, int type) {

        Date clearBeforeTime = null;
        int clearBeforeNum = 0;
        if (type == 1) {
            clearBeforeTime = DateUtil.addMonths(new Date(), -1);    // 清理一个月之前日志数据
        } else if (type == 2) {
            clearBeforeTime = DateUtil.addMonths(new Date(), -3);    // 清理三个月之前日志数据
        } else if (type == 3) {
            clearBeforeTime = DateUtil.addMonths(new Date(), -6);    // 清理六个月之前日志数据
        } else if (type == 4) {
            clearBeforeTime = DateUtil.addYears(new Date(), -1);    // 清理一年之前日志数据
        } else if (type == 5) {
            clearBeforeNum = 1000;        // 清理一千条以前日志数据
        } else if (type == 6) {
            clearBeforeNum = 10000;        // 清理一万条以前日志数据
        } else if (type == 7) {
            clearBeforeNum = 30000;        // 清理三万条以前日志数据
        } else if (type == 8) {
            clearBeforeNum = 100000;    // 清理十万条以前日志数据
        } else if (type == 9) {
            clearBeforeNum = 0;            // 清理所有日志数据
        } else {
            return new ReturnT<String>(ReturnT.FAIL_CODE, I18nUtil.getString("joblog_clean_type_unvalid"));
        }

        List<String> logIds = null;
        do {
            Page<String> page =PageUtil.buildPage(1000/clearBeforeNum,1000);
            logIds = xxlJobLogDao.findClearLogIds(jobGroup, jobId, clearBeforeTime, page);
            if (logIds != null && logIds.size() > 0) {
                xxlJobLogDao.clearLog(logIds);
            }
        } while (logIds != null && logIds.size() > 0);

        return ReturnT.SUCCESS;
    }

}
