package com.biz.crm.business.alibaba.dingtalk.local.service.impl;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.business.common.auth.sdk.service.UrlApiService;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.business.common.log.sdk.dto.ExternalLogDetailDto;
import com.biz.crm.business.common.log.sdk.service.ExternalLogVoService;
import com.biz.crm.business.common.log.sdk.util.ExternalLogUtil;
import com.biz.crm.business.alibaba.dingtalk.sdk.constant.DingTalkConstant;
import com.biz.crm.business.alibaba.dingtalk.sdk.service.DingTalkService;
import com.dingtalk.api.DefaultDingTalkClient;
import com.dingtalk.api.DingTalkClient;
import com.dingtalk.api.request.OapiGettokenRequest;
import com.dingtalk.api.request.OapiMessageCorpconversationAsyncsendV2Request;
import com.dingtalk.api.request.OapiV2UserGetbymobileRequest;
import com.dingtalk.api.response.OapiGettokenResponse;
import com.dingtalk.api.response.OapiMessageCorpconversationAsyncsendV2Response;
import com.dingtalk.api.response.OapiV2UserGetbymobileResponse;
import com.taobao.api.ApiException;
import com.taobao.api.Constants;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.List;
import java.util.Objects;

/**
 * 发送钉钉消息
 *
 * @author huxmld
 * @version v1.0.0
 * @date 2022.11.23 17:31
 */
@Service
@Slf4j
public class DingTalkServiceImpl implements DingTalkService {

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private ExternalLogVoService externalLogVoService;

    @Autowired(required = false)
    private UrlApiService urlApiService;

    private UrlAddressVo urlAddressVo;

    public void init() {
        urlAddressVo = urlApiService.getUrlAddressByAccount(DingTalkConstant.DING_TALK_USER_NAME);
        Assert.hasLength(urlAddressVo.getUrl(), "钉钉接口URL不能为空!");
        Assert.hasLength(urlAddressVo.getAccessId(), "钉钉参数[agentId]不能为空!");
        Assert.hasLength(urlAddressVo.getAccessKey(), "钉钉参数[appKey]不能为空!");
        Assert.hasLength(urlAddressVo.getSecretKey(), "钉钉参数[appSecret]不能为空!");
    }

    /**
     * 发送文本类型消息
     *
     * @param userId     钉钉用户ID/用户账号集合
     * @param msgContent 消息内容
     * @return com.biz.crm.business.common.sdk.model.Result
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.23 17:47
     */
    @Override
    public Result<OapiMessageCorpconversationAsyncsendV2Response> sendTextMsgByUserId(String userId, String msgContent) {
        return this.sendTextMsg(userId, msgContent);
    }

    /**
     * 发送文本类型消息
     *
     * @param userIdList 钉钉用户ID/用户账号集合
     * @param msgContent 消息内容
     * @return com.biz.crm.business.common.sdk.model.Result
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.23 17:47
     */
    @Override
    public Result<OapiMessageCorpconversationAsyncsendV2Response> sendTextMsgByUserIdList(List<String> userIdList, String msgContent) {
        return this.sendTextMsg(StringUtils.join(userIdList,","), msgContent);
    }

    /**
     * 发送工作通知
     *
     * @param userIds    钉钉钉钉用户ID/用户账号集合
     * @param msgContent 消息内容
     */
    private Result<OapiMessageCorpconversationAsyncsendV2Response> sendTextMsg(String userIds, String msgContent) {
        Result<String> accessTokenResult = this.getDingTalkAccessToken();
        if (!accessTokenResult.isSuccess()) {
            return Result.error(accessTokenResult.getMessage());
        }
        Result<OapiMessageCorpconversationAsyncsendV2Response> result = new Result<>();
        try {
            DingTalkClient client = new DefaultDingTalkClient(urlAddressVo.getUrl() + DingTalkConstant.SEND_MSG_INTERFACE);
            OapiMessageCorpconversationAsyncsendV2Request request = new OapiMessageCorpconversationAsyncsendV2Request();
            request.setAgentId(Long.parseLong(urlAddressVo.getAccessId()));
            request.setUseridList(userIds);
            request.setToAllUser(false);
            OapiMessageCorpconversationAsyncsendV2Request.Msg msg = new OapiMessageCorpconversationAsyncsendV2Request.Msg();
            msg.setMsgtype("text");
            msg.setText(new OapiMessageCorpconversationAsyncsendV2Request.Text());
            msg.getText().setContent(msgContent);
            request.setMsg(msg);
            ExternalLogDetailDto detailDto = ExternalLogUtil.buildLogSaveInfo(request, urlAddressVo);
            detailDto.setMethod(DingTalkConstant.SEND_MSG_INTERFACE);
            detailDto.setMethodMsg("发送钉钉通知消息");
            externalLogVoService.addOrUpdateLog(detailDto, true);

            OapiMessageCorpconversationAsyncsendV2Response response = client.execute(request, accessTokenResult.getResult());
            if (DingTalkConstant.SUCCESS_CODE.equals(response.getErrcode())) {
                result.setResult(response);
                result.setMessage(response.getMsg());
            } else {
                result.setSuccess(false);
                result.setMessage(response.getSubMsg());
            }
            ExternalLogUtil.buildLogResult(detailDto, result);
            externalLogVoService.addOrUpdateLog(detailDto, false);
        } catch (ApiException e) {
            log.error(e.getMessage(), e);
            result.setSuccess(false);
            result.setMessage(e.getMessage());
        }
        return result;

    }

    /**
     * 获取钉钉token
     *
     * @return com.biz.crm.business.common.sdk.model.Result
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.29 12:32
     */
    @Override
    public Result<String> getDingTalkAccessToken() {
        this.init();
        String token = (String) redisService.get(DingTalkConstant.DING_TALK_TOKEN);
        Result<String> result = new Result<>();
        if (StringUtils.isNotBlank(token)) {
            result.setResult(token);
            return result;
        }

        DingTalkClient client = new DefaultDingTalkClient(urlAddressVo.getUrl() + DingTalkConstant.ACCESS_TOKEN_INTERFACE);
        OapiGettokenRequest request = new OapiGettokenRequest();
        request.setAppkey(urlAddressVo.getAccessKey());
        request.setAppsecret(urlAddressVo.getSecretKey());
        request.setHttpMethod(Constants.METHOD_GET);
        try {
            OapiGettokenResponse response = client.execute(request);
            result.setMessage(response.getErrmsg());
            if (DingTalkConstant.SUCCESS_CODE.equals(response.getErrcode())) {
                result.setResult(response.getAccessToken());
                if (Objects.nonNull(response.getExpiresIn())
                        && response.getExpiresIn() - 100 > 0) {
                    redisService.set(DingTalkConstant.DING_TALK_TOKEN, response.getAccessToken(),
                            response.getExpiresIn() - 100);
                }
            } else {
                result.setSuccess(false);
                result.setMessage(response.getErrmsg());
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            result.setSuccess(false);
            result.setMessage(e.getMessage());
        }

        return result;
    }

    /**
     * 根据手机号获取userId
     *
     * @param mobile 手机号
     * @return java.lang.String
     * @author huxmld
     * @version v1.0.0
     * @date 2022.12.3 18:25
     */
    @Override
    public String getUserIdByMobile(String mobile) {
        Assert.hasLength(mobile, "手机号不能为空!");
        Result<String> result = this.getDingTalkAccessToken();
        Assert.isTrue(result.isSuccess(), "获取钉钉Token失败!");
        try {
            DingTalkClient client = new DefaultDingTalkClient(urlAddressVo.getUrl() + DingTalkConstant.USER_ID_BY_MOBILE_INTERFACE);
            OapiV2UserGetbymobileRequest req = new OapiV2UserGetbymobileRequest();
            req.setMobile(mobile);
            OapiV2UserGetbymobileResponse rsp = client.execute(req, result.getResult());
            return rsp.getBody();
        } catch (ApiException e) {
            log.error(e.getMessage(), e);
            throw new IllegalArgumentException(e.getMessage());
        }
    }
}
