package com.biz.crm.business.common.auth.feign.service.internal;

import com.biz.crm.business.common.auth.sdk.constans.AuthConstant;
import com.biz.crm.business.common.auth.sdk.eunm.BusinessKeyEnum;
import com.biz.crm.business.common.auth.sdk.exception.SignException;
import com.biz.crm.business.common.auth.sdk.service.AuthSignService;
import com.biz.crm.business.common.auth.sdk.service.UrlApiService;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import javax.xml.bind.DatatypeConverter;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 签名验证 service 实现
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.12 20:57
 */
@Slf4j
@Service
public class AuthSignServiceImpl implements AuthSignService {

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private UrlApiService urlApiService;

    /**
     * header参数签名 获取加密字符串
     *
     * @param accessKey
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/16 11:29
     */
    @Override
    public UrlAddressVo getSignInfoForAccessKey(String accessKey) {
        Assert.hasLength(accessKey, "参数[ak]不能为空!");
        UrlAddressVo urlAddressVo = urlApiService.getUrlAddressByAccessKey(accessKey);
        Assert.hasLength(urlAddressVo.getSecretKey(), "密匙未配置,请联系管理员!");
        //单位时间  访问次数限制
        long accessTimesMax = urlAddressVo.getAccessTimesMax();
        String redisKey = AuthConstant.EXTERNAL_ACCESS_LIMIT_LOCK + accessKey;
        long accessTimes = redisService.hIncr(redisKey, urlAddressVo.getAccessKey(), 1L);
        long expireDate = redisService.getExpire(redisKey);
        if (expireDate <= 0) {
            redisService.expire(redisKey, AuthConstant.MAX_SECONDS);
        }
        Assert.isTrue(accessTimes < accessTimesMax,
                "[" + SignException.ACCESS_LIMIT.getCode() + "]" + SignException.ACCESS_LIMIT.getDescription());
        return urlAddressVo;
    }


    /**
     * 接口权限验证
     *
     * @param urlAddressVo
     * @param dataMap
     * @return boolean
     * @author huxmld
     * @version v1.0.0
     * @date 2024-01-19 03:41
     */
    @Override
    public void valAuthority(UrlAddressVo urlAddressVo, Map<String, Object> dataMap) {
        Assert.notNull(urlAddressVo, "[" + SignException.NO_PERMISSION.getCode() + "]" + SignException.NO_PERMISSION.getDescription());
        Assert.notEmpty(dataMap, "[" + SignException.NOT_NULL.getCode() + "]" + SignException.NOT_NULL.getDescription());
        //全部权限
        if (BooleanEnum.TRUE.getCapital().equals(urlAddressVo.getIsAll())) {
            return;
        }
        Map<String, List<String>> authorityMap = urlAddressVo.getAuthorityMap();
        Assert.notEmpty(authorityMap, SignException.NO_PERMISSION.getDescription() + "[" + SignException.NO_PERMISSION.getCode() + "]");
        /**
         * 当前只有一种权限类型,所以直接判断空
         * 后续多个需要判断同时不为空,才能过
         */
        List<String> businessInterfaceList = authorityMap.getOrDefault(BusinessKeyEnum.BUSINESS_INTERFACE.getCode(), Lists.newArrayList());
        Assert.notEmpty(businessInterfaceList, "[" + SignException.NO_PERMISSION.getCode() + "]" + SignException.NO_PERMISSION.getDescription());

        String requestUri = (String) dataMap.get(AuthConstant.REQUEST_URI_MAP_KEY);
        Assert.isTrue(businessInterfaceList.contains(requestUri), "[" + SignException.NO_PERMISSION.getCode() + "]" + SignException.NO_PERMISSION.getDescription());
    }

    /**
     * 验证签名是否过期
     *
     * @param signExpireDate 过期时间|单位分钟
     * @param timestamp      时间戳
     * @return java.lang.Boolean
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.5 13:08
     */
    @Override
    public Boolean verifySignExpired(Integer signExpireDate, String timestamp) {
        if (Objects.isNull(signExpireDate)
                || StringUtils.isEmpty(timestamp)) {
            return false;
        }
        long timestampLong = 0;
        try {
            timestampLong = Long.parseLong(timestamp);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return timestampLong + signExpireDate * 60L * 1000 >= System.currentTimeMillis();
    }

    /**
     * 参数排序
     *
     * @param parameterMap
     * @return java.lang.String
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.12 20:57
     */
    @Override
    public String parameterBodySort(Map<String, Object> parameterMap) {
        String result = "";
        try {

            List<Map.Entry<String, Object>> infoIds = new ArrayList<>(parameterMap.entrySet());
            // 对所有传入参数按照字段名的 ASCII 码从小到大排序（字典序）
            infoIds.sort(Map.Entry.comparingByKey());
            // 构造签名键值对的格式
            StringBuilder stringBuilder = new StringBuilder();
            for (Map.Entry<String, Object> item : infoIds) {
                if (StringUtils.isNotEmpty(item.getKey())) {
                    String key = item.getKey();
                    Object val = item.getValue();
                    if (!(val == null || val == "")) {
                        stringBuilder.append(key).append("=").append(val).append("&");
                    }
                }
            }
            result = stringBuilder.toString();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        }
        return result;
    }

    /**
     * 参数签名
     *
     * @param parameterMap
     * @return java.lang.String
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.12 20:57
     */
    @Override
    public String parameterBodySign(Map<String, Object> parameterMap) {
        Assert.notEmpty(parameterMap, "签名字符串不能为空!");
        try {
            String sortEnd = this.parameterBodySort(parameterMap);
            return DigestUtils.md5Hex(sortEnd);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * header参数 签名
     *
     * @param timestamp 时间戳
     * @param accessKey
     * @param secretKey
     * @return java.lang.String
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.1 14:31
     */
    @Override
    public String parameterHeaderSign(String timestamp, String accessKey, String secretKey) {
        Assert.hasLength(timestamp, "参数[timestamp]不能为空!");
        Assert.hasLength(accessKey, "参数[ak]不能为空!");
        Assert.hasLength(secretKey, "参数[sk]不能为空!");
        String signStr = timestamp + accessKey + secretKey;
        MessageDigest md5 = null;
        try {
            md5 = MessageDigest.getInstance("MD5");
            byte[] digest = md5.digest(signStr.getBytes(
                    StandardCharsets.UTF_8));
            return DatatypeConverter.printHexBinary(digest).toLowerCase();
        } catch (NoSuchAlgorithmException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        }

    }
}
