package com.biz.crm.business.common.auth.feign.service.internal;

import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.auth.feign.feign.UrlAddressVoFeign;
import com.biz.crm.business.common.auth.sdk.constans.AuthConstant;
import com.biz.crm.business.common.auth.sdk.service.UrlApiService;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 获取 加密字符串 service
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.11.3 14:44
 */
@Slf4j
@Service
public class UrlApiServiceImpl implements UrlApiService {

    @Autowired(required = false)
    private UrlAddressVoFeign urlAddressVoFeign;
    /**
     * 执行内容缓存
     */
    private static volatile Cache<String, UrlAddressVo> cache = null;

    public UrlApiServiceImpl() {
        if (cache == null) {
            synchronized (UrlApiServiceImpl.class) {
                while (cache == null) {
                    cache = CacheBuilder.newBuilder().initialCapacity(10000)
                            .expireAfterWrite(AuthConstant.CACHE_TIME, TimeUnit.MINUTES).maximumSize(100000).build();
                }
            }
        }
    }

    /**
     * 通过(账号|数据字典)获取
     *
     * @param account|account 数据字典|账号
     * @return com.biz.crm.external.business.common.vo.UrlAddressVo
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.3 14:45
     */
    @Override
    public UrlAddressVo getUrlAddressByAccount(String account) {
        Assert.hasLength(account, "参数[account]不能为空!");
        UrlAddressVo urlAddressVo = cache.getIfPresent(account);
        if (Objects.nonNull(urlAddressVo)) {
            return urlAddressVo;
        }
        Result<UrlAddressVo> result = urlAddressVoFeign.getUrlAddressByAccount(account);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        if (Objects.nonNull(result.getResult())) {
            cache.put(account, result.getResult());
        }
        return result.getResult();
    }

    /**
     * 通过系统标记[accessKey] 获取
     *
     * @param accessKey 系统标记
     * @return com.biz.crm.external.business.common.vo.UrlAddressVo
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.3 14:45
     */
    @Override
    public UrlAddressVo getUrlAddressByAccessKey(String accessKey) {
        Assert.hasLength(accessKey, "参数[ak]不能为空!");
        UrlAddressVo urlAddressVo = cache.getIfPresent(accessKey);
        if (Objects.nonNull(urlAddressVo)) {
            return urlAddressVo;
        }
        Result<UrlAddressVo> result = urlAddressVoFeign.getUrlAddressByAccessKey(accessKey);
        Assert.isTrue(result.isSuccess(), result.getMessage());
        if (Objects.nonNull(result.getResult())) {
            cache.put(accessKey, result.getResult());
        }
        return result.getResult();

    }

    /**
     * 根据系统标记获取系统账号信息
     *
     * @param accessKey 系统标记
     * @return java.lang.String
     * @author huxmld
     * @version v1.0.0
     * @date 2022.12.10 17:20
     */
    @Override
    public String getUserNameByAccessKey(String accessKey) {
        return this.getUrlAddressByAccessKey(accessKey).getUserName();
    }
}
