package com.biz.crm.business.common.auth.feign.aop;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.auth.sdk.constans.AuthConstant;
import com.biz.crm.business.common.auth.sdk.dto.SignBaseDto;
import com.biz.crm.business.common.auth.sdk.exception.SignException;
import com.biz.crm.business.common.auth.sdk.service.AuthSignService;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.business.common.base.util.ExceptionStackMsgUtil;
import com.biz.crm.business.common.log.sdk.constant.ExternalLogGlobalConstants;
import com.biz.crm.business.common.log.sdk.dto.ExternalLogDetailDto;
import com.biz.crm.business.common.log.sdk.service.ExternalLogVoService;
import com.biz.crm.business.common.log.sdk.util.ExternalLogUtil;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 签名验证  AOP
 * <p>
 *
 * @describe 签名
 * @author huxmld
 * @version v1.0.0
 * @date 2021/8/12 15:14
 */
@Aspect
@Component
@Slf4j
public class SignBodyAspect {

    protected static final ThreadLocal<ConcurrentHashMap<Long, ExternalLogDetailDto>> LOG_VO_MAP = new ThreadLocal<>();

    @Autowired(required = false)
    private AuthSignService authSignService;

    @Autowired(required = false)
    private ExternalLogVoService externalLogVoService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 前置通知：
     * 1. 在执行目标方法之前执行，比如请求接口之前的签名验证;
     * 2. 在前置通知中设置保存请求日志信息，如开始时间，请求参数等
     */
    @Before(value = "@annotation(com.biz.crm.business.common.auth.sdk.aop.SignBody) || @within(com.biz.crm.business.common.auth.sdk.aop.SignBodyGlobal)")
    public void doBefore(JoinPoint joinPoint) {
        ExternalLogDetailDto detailDto = null;
        try {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            Assert.notNull(requestAttributes, "请求不合法!");
            HttpServletRequest request = requestAttributes.getRequest();
            String requestMethod = request.getMethod();
            Map<String, Object> headMap = new HashMap<>(16);
            Enumeration headerNames = request.getHeaderNames();
            while (headerNames.hasMoreElements()) {
                String paraName = (String) headerNames.nextElement();
                String paraValue = request.getHeader(paraName);
                headMap.put(paraName, paraValue);
            }
            Map<String, Object> dataMap = new HashMap<>(16);
            StringBuffer reqBody = new StringBuffer();
            if ("GET".equalsIgnoreCase(requestMethod)) {
                Enumeration enu = request.getParameterNames();
                while (enu.hasMoreElements()) {
                    String paraName = (String) enu.nextElement();
                    String paraValue = request.getParameter(paraName);
                    dataMap.put(paraName, paraValue);
                }
                reqBody.append(JSON.toJSONString(dataMap));
            } else {
                Optional.ofNullable(joinPoint.getArgs()).ifPresent(x -> {
                    if (x.length > 1) {
                        reqBody.append(JSON.toJSONString(x));
                    } else {
                        for (Object o : x) {
                            reqBody.append(JSON.toJSONString(o));
                        }
                    }
                });
            }
            Method method = ((MethodSignature) joinPoint.getSignature()).getMethod();
            String timestamp = request.getHeader("timestamp");
            String accessKey = request.getHeader("ak");
            String token = request.getHeader("token");
            SignBaseDto dto = JSON.parseObject(reqBody.toString(), SignBaseDto.class);
            dto = Optional.ofNullable(dto).orElse(new SignBaseDto());
            detailDto = ExternalLogUtil.buildLogSaveInfo(dto);
            // 日志存缓存
            addLogVo(detailDto);
            detailDto.setReqHead(JSON.toJSONString(headMap));
            detailDto.setReqJson(reqBody.toString());
            detailDto.setAccessKey(accessKey);
            detailDto.setMethod(method.getName());
            String requestUri = request.getRequestURI();
            detailDto.setRequestUri(requestUri);
            dataMap.put(AuthConstant.REQUEST_URI_MAP_KEY, requestUri);
            ApiOperation apiOperation = method.getAnnotation(ApiOperation.class);
            AtomicReference<String> methodMsg = new AtomicReference<>("");
            Optional.ofNullable(apiOperation)
                    .filter(x -> !StringUtils.isEmpty(x.value()))
                    .ifPresent(x -> methodMsg.set(x.value()));
            detailDto.setMethodMsg(methodMsg.get());
            detailDto.setInvokeType(ExternalLogGlobalConstants.EXTERNAL_INTERFACE);
            // 日志存缓存
            addLogVo(detailDto);
            loginUserService.refreshAuthentication(null);
            externalLogVoService.addOrUpdateLog(detailDto, true);
            UrlAddressVo urlAddressVo = authSignService.getSignInfoForAccessKey(accessKey);
            detailDto.setUserName(urlAddressVo.getUserName());
            detailDto.setFullName(urlAddressVo.getFullName());
            // 日志存缓存
            addLogVo(detailDto);
            String secretKey = request.getHeader("sk");
            Assert.isTrue(StringUtils.isEmpty(secretKey), "[sk]不允许传递!");
            authSignService.valAuthority(urlAddressVo, dataMap);
            loginUserService.refreshAuthentication(urlAddressVo);

            Assert.hasLength(token, "参数[token]不能为空!");
            Assert.hasLength(timestamp, "参数[timestamp]不能为空!");
            Assert.hasLength(accessKey, "参数[ak]不能为空!");
            valBaseInfoEmpty(dto);
            Assert.isTrue(authSignService.verifySignExpired(urlAddressVo.getSignExpireDate(), timestamp),
                    "[" + SignException.SIGN_EXPIRED.getCode() + "]" + SignException.SIGN_EXPIRED.getDescription());

            dataMap.put(accessKey, urlAddressVo.getAccessKey() + urlAddressVo.getSecretKey());
            dataMap.put("timestamp", timestamp);
            // 签名
            String parameterSign = authSignService.parameterBodySign(dataMap);
            Assert.isTrue(StringUtils.equalsIgnoreCase(parameterSign, token),
                    "[" + SignException.SIGN_ERROR.getCode() + "]" + SignException.SIGN_ERROR.getDescription());

        } catch (Exception e) {
            log.error(e.getMessage(), e);
            if (Objects.nonNull(detailDto)) {
                detailDto.setExceptionStack(ExceptionStackMsgUtil.stackMsg(e));
                detailDto.setTipMsg(e.getMessage());
                detailDto.setRespJson(JSON.toJSONString(Result.error(e.getMessage())));
                if (!AuthConstant.NOT_ES_LOG.contains(detailDto.getMethod())) {
                    externalLogVoService.addOrUpdateLog(detailDto, false);
                } else {
                    log.warn("外部系统日志[{}]", JSON.toJSONString(detailDto));
                }
            }
            throw e;
        }
    }

    /**
     * 返回通知：
     * 1. 在执行目标方法之后执行，比如请求接口之后的日志保存;
     * 2. 在返回通知设置返回信息，如返回时间，返回参数等
     */
    @AfterReturning(value = "@annotation(com.biz.crm.business.common.auth.sdk.aop.SignBody) || @within(com.biz.crm.business.common.auth.sdk.aop.SignBodyGlobal)",
            returning = "result")
    public void AfterReturning(Object result) {
        ExternalLogDetailDto detailDto = getLogVo();
        if (detailDto != null) {
            ExternalLogUtil.buildLogResult(detailDto, result);
            externalLogVoService.addOrUpdateLog(detailDto, false);
        }
    }

    /**
     * 异常通知：
     * 1. 在执行目标方法抛出异常之后;
     * 2. 在异常通知设置返回的异常信息
     */
    @AfterThrowing(value = "@annotation(com.biz.crm.business.common.auth.sdk.aop.SignBody) || @within(com.biz.crm.business.common.auth.sdk.aop.SignBodyGlobal)", throwing = "e")
    public void AfterThrowing(Exception e) {
        ExternalLogDetailDto detailDto = getLogVo();
        if (detailDto != null) {
            detailDto.setExceptionStack(ExceptionStackMsgUtil.stackMsg(e));
            detailDto.setTipMsg(e.getMessage());
            detailDto.setRespJson(JSON.toJSONString(Result.error(e.getMessage())));
            if (!AuthConstant.NOT_ES_LOG.contains(detailDto.getMethod())) {
                externalLogVoService.addOrUpdateLog(detailDto, false);
            } else {
                log.warn("外部系统日志[{}]", JSON.toJSONString(detailDto));
            }
        }
        log.error("======> AOP签名验证信息 异常信息  <======");
        log.error(e.getMessage(), e);
    }


    /*----------------------------------线程缓存 start----------------------------------*/

    /**
     * 添加日志对象
     *
     * @param respVo
     * @return
     * @describe 日志
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/13 12:33
     */
    private void addLogVo(ExternalLogDetailDto respVo) {
        Thread thread = Thread.currentThread();
        ConcurrentHashMap<Long, ExternalLogDetailDto> logMap = LOG_VO_MAP.get();
        if (logMap == null) {
            logMap = new ConcurrentHashMap<>(16);
        }
        logMap.put(thread.getId(), respVo);
        LOG_VO_MAP.set(logMap);
    }

    /**
     * 获取日志对象
     *
     * @return
     * @describe 日志
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/13 12:33
     */
    private ExternalLogDetailDto getLogVo() {
        Thread thread = Thread.currentThread();
        ConcurrentHashMap<Long, ExternalLogDetailDto> logMap = LOG_VO_MAP.get();
        if (logMap == null) {
            logMap = new ConcurrentHashMap<>(16);
        }
        ExternalLogDetailDto detailDto = logMap.get(thread.getId());
        logMap.remove(thread.getId());
        LOG_VO_MAP.set(logMap);
        return detailDto;
    }

    /**
     * 验证请求对象字段非空
     *
     * @param dto
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/16 11:29
     */
    private void valBaseInfoEmpty(SignBaseDto dto) {

        Assert.notNull(dto, "参数不能为空!");
        Assert.hasLength(dto.getMethod(), "参数[method]不能为空!");
        Assert.hasLength(dto.getUuid(), "参数[uuid]不能为空!");

    }

}