package com.biz.crm.business.common.auth.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.auth.local.entity.UrlAddress;
import com.biz.crm.business.common.auth.local.entity.UrlAddressAuthority;
import com.biz.crm.business.common.auth.local.repository.UrlAddressAuthorityRepository;
import com.biz.crm.business.common.auth.local.repository.UrlAddressRepository;
import com.biz.crm.business.common.auth.local.service.UrlAddressService;
import com.biz.crm.business.common.auth.sdk.constans.AuthConstant;
import com.biz.crm.business.common.auth.sdk.dto.UrlAddressAuthorityDto;
import com.biz.crm.business.common.auth.sdk.eunm.BusinessKeyEnum;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressAuthorityVo;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.enums.DelFlagStatusEnum;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 获取 加密字符串 service
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.11.3 14:44
 */
@Slf4j
@Service
public class UrlAddressServiceImpl implements UrlAddressService {

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private UrlAddressRepository urlAddressRepository;

    @Autowired(required = false)
    private UrlAddressAuthorityRepository urlAddressAuthorityRepository;

    /**
     * 新增
     *
     * @param dto
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(UrlAddress dto) {
        UrlAddress urlAddress = this.valBaseInfo(dto);
        this.valSave(urlAddress);
        saveAuthorityList(dto.getInterfaceList(), dto.getUserName());
        this.urlAddressRepository.save(urlAddress);
    }

    /**
     * 保存的验证
     *
     * @param entity 对象
     */
    private void valSave(UrlAddress entity) {
        //企业微信存在一个账号(ak)多个密钥的情况
        int countAccessKey = this.urlAddressRepository.lambdaQuery()
                .eq(UrlAddress::getAccessKey, entity.getAccessKey())
                .count();

        Assert.isTrue(countAccessKey < 1,
                "ak[" + entity.getAccessKey() + "]已存在");
        int countUserName = this.urlAddressRepository.lambdaQuery()
                .eq(UrlAddress::getUserName, entity.getUserName())
                .count();
        Assert.isTrue(countUserName < 1,
                "账号[" + entity.getUserName() + "]已存在");
    }


    /**
     * 保存权限类型
     *
     * @param interfaceList 权限详情
     */
    private void saveAuthorityList(List<UrlAddressAuthorityDto> interfaceList, String userName) {
        if (StringUtils.isEmpty(userName)) {
            return;
        }
        urlAddressAuthorityRepository.delByUserName(userName);
        if (CollectionUtil.isEmpty(interfaceList)) {
            return;
        }
        List<UrlAddressAuthority> authorityList = Lists.newArrayList();
        String tenantCode = TenantUtils.getTenantCode();
        String delFlag = DelFlagStatusEnum.NORMAL.getCode();
        String enableCode = EnableStatusEnum.ENABLE.getCode();
        interfaceList.forEach(vo -> {
            UrlAddressAuthority addressAuthority = new UrlAddressAuthority();
            addressAuthority.setUserName(userName);
            String businessKey = vo.getBusinessKey();
            Assert.hasLength(businessKey, "权限类型不能为空!" + BusinessKeyEnum.values());
            BusinessKeyEnum businessKeyEnum = BusinessKeyEnum.codeToEnum(businessKey);
            Assert.notNull(businessKeyEnum, "权限类型[" + businessKey + "]不合法!");
            addressAuthority.setBusinessKey(businessKey);
            addressAuthority.setBusinessCode(vo.getBusinessCode());
            addressAuthority.setBusinessDesc(vo.getBusinessDesc());
            addressAuthority.setTenantCode(tenantCode);
            addressAuthority.setDelFlag(delFlag);
            addressAuthority.setEnableStatus(enableCode);
            authorityList.add(addressAuthority);
        });
        if (CollectionUtil.isNotEmpty(authorityList)) {
            urlAddressAuthorityRepository.saveBatch(authorityList);
        }
    }

    /**
     * 编辑
     *
     * @param dto 对象
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(UrlAddress dto) {
        UrlAddress urlAddress = this.valBaseInfo(dto);
        this.valUpdate(urlAddress);
        saveAuthorityList(dto.getInterfaceList(), dto.getUserName());
        this.urlAddressRepository.updateById(urlAddress);
    }

    /**
     * 更新的验证
     *
     * @param entity
     */
    private void valUpdate(UrlAddress entity) {
        entity.setModifyAccount(null);
        entity.setModifyName(null);
        entity.setModifyTime(null);
        Assert.hasLength(entity.getId(), "ID不能为空!");
        UrlAddress oldEntity = this.urlAddressRepository.getById(entity.getId());
        Assert.notNull(oldEntity, "数据不存在,请刷新重试!");
        int countAccessKey = this.urlAddressRepository.lambdaQuery()
                .eq(UrlAddress::getAccessKey, entity.getAccessKey())
                .ne(UrlAddress::getId, entity.getId())
                .count();
        Assert.isTrue(countAccessKey < 1,
                "ak[" + entity.getAccessKey() + "]已存在,请修改!");
        int countUserName = this.urlAddressRepository.lambdaQuery()
                .eq(UrlAddress::getUserName, entity.getUserName())
                .ne(UrlAddress::getId, entity.getId())
                .count();
        Assert.isTrue(countUserName < 1,
                "账号[" + entity.getUserName() + "]已存在,请修改!");
        urlAddressAuthorityRepository.delByUserName(oldEntity.getUserName());
        List<String> delKeyList = Lists.newArrayList();
        delKeyList.add(this.getRedisKey(oldEntity.getAccessKey()));
        delKeyList.add(this.getRedisKey(oldEntity.getUserName()));
        redisService.del(delKeyList);
    }

    /**
     * 基础验证
     *
     * @param dto
     * @return
     */
    private UrlAddress valBaseInfo(UrlAddress dto) {
        Assert.notNull(dto, "信息不能为空!");
        Assert.hasLength(dto.getUserName(), "参数[账号信息]不能为空!");
        Assert.hasLength(dto.getUrl(), "参数[URL信息]不能为空!");
        Assert.hasLength(dto.getAccessKey(), "参数[ak]不能为空!");
        Assert.hasLength(dto.getSecretKey(), "参数[sk]不能为空!");
        String isAll = dto.getIsAll();
        Assert.hasLength(dto.getIsAll(), "是否所有接口权限不能为空!");

        String url = dto.getUrl();
        if (url.endsWith(AuthConstant.URL_TAIL)) {
            dto.setUrl(url.substring(0, url.length() - 1));
        }

        String mobileUrl = dto.getMobileUrl();
        if (StringUtils.isNotEmpty(mobileUrl)
                && mobileUrl.endsWith(AuthConstant.URL_TAIL)) {
            dto.setMobileUrl(mobileUrl.substring(0, url.length() - 1));
        }

        if (!BooleanEnum.TRUE.getCapital().equals(isAll)
                && !BooleanEnum.FALSE.getCapital().equals(isAll)) {
            throw new IllegalArgumentException("isAll不合法,必须为[Y/N]");
        }
        if (CollectionUtil.isNotEmpty(dto.getInterfaceList())) {
            dto.getInterfaceList().forEach(vo -> {
                BusinessKeyEnum businessKeyEnum = BusinessKeyEnum.codeToEnum(vo.getBusinessKey());
                Assert.notNull(businessKeyEnum, "权限类型[" + vo.getBusinessKey() + "]不合法!");
                switch (businessKeyEnum) {
                    case BUSINESS_INTERFACE:
                        break;
                    default:
                        throw new IllegalArgumentException("此类型未定义!");
                }
            });
        }
        if (StringUtils.isEmpty(dto.getTenantCode())) {
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        if (Objects.isNull(dto.getCacheTime())) {
            dto.setCacheTime(AuthConstant.DEFAULT_CACHE_TIME);
        }
        if (Objects.isNull(dto.getCacheRedisTime())) {
            dto.setCacheRedisTime(AuthConstant.DEFAULT_CACHE_TIME);
        }
        if (Objects.isNull(dto.getSignExpireDate())) {
            dto.setSignExpireDate(AuthConstant.DEFAULT_SIGN_EXPIRE_DATE);
        }
        if (Objects.isNull(dto.getAccessTimesMax())) {
            dto.setAccessTimesMax(AuthConstant.DEFAULT_ACCESS_TIMES_MAX);
        }
        if (StringUtils.isEmpty(dto.getTenantCode())){
            dto.setTenantCode(TenantUtils.getTenantCode());
        }
        return dto;
    }

    /**
     * 删除
     *
     * @param ids 主键集合
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择数据!");
        List<UrlAddress> list = this.urlAddressRepository.findByIds(ids);
        Assert.notEmpty(list, "选择数据不存在,请刷新重试!");
        this.urlAddressRepository.updateDelFlagByIds(ids);
        List<String> delKeyList = Lists.newArrayList();
        list.forEach(entity -> {
            delKeyList.add(this.getRedisKey(entity.getAccessKey()));
            delKeyList.add(this.getRedisKey(entity.getUserName()));
        });
        redisService.del(delKeyList);
    }

    /**
     * 启用
     *
     * @param ids 主键集合
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择数据!");
        List<UrlAddress> list = this.urlAddressRepository.findByIds(ids);
        Assert.notEmpty(list, "选择数据不存在,请刷新重试!");
        this.urlAddressRepository.updateEnableStatusByIds(ids, EnableStatusEnum.ENABLE);
        List<String> delKeyList = Lists.newArrayList();
        list.forEach(entity -> {
            delKeyList.add(this.getRedisKey(entity.getAccessKey()));
            delKeyList.add(this.getRedisKey(entity.getUserName()));
        });
        redisService.del(delKeyList);
    }

    /**
     * 禁用
     *
     * @param ids 主键集合
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "请选择数据!");
        List<UrlAddress> list = this.urlAddressRepository.findByIds(ids);
        Assert.notEmpty(list, "选择数据不存在,请刷新重试!");
        this.urlAddressRepository.updateEnableStatusByIds(ids, EnableStatusEnum.DISABLE);
        List<String> delKeyList = Lists.newArrayList();
        list.forEach(entity -> {
            delKeyList.add(this.getRedisKey(entity.getAccessKey()));
            delKeyList.add(this.getRedisKey(entity.getUserName()));
        });
        redisService.del(delKeyList);
    }

    /**
     * 详情
     *
     * @param id 主键
     * @return void
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.16 19:28
     */
    @Override
    public UrlAddressVo findById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        UrlAddress urlAddress = urlAddressRepository.findById(id);
        if (Objects.isNull(urlAddress)) {
            return null;
        }
        UrlAddressVo urlAddressVo = this.nebulaToolkitService.copyObjectByWhiteList(urlAddress, UrlAddressVo.class,
                HashSet.class, ArrayList.class);
        List<UrlAddressAuthority> urlAddressAuthorityList = urlAddressAuthorityRepository.getByUserName(urlAddressVo.getUserName());
        if (CollectionUtil.isEmpty(urlAddressAuthorityList)) {
            return urlAddressVo;
        }
        List<UrlAddressAuthorityVo> authorityVoList = (List<UrlAddressAuthorityVo>) this.nebulaToolkitService.copyCollectionByWhiteList(urlAddressAuthorityList,
                UrlAddressAuthority.class, UrlAddressAuthorityVo.class, HashSet.class, ArrayList.class);
        Map<String, List<UrlAddressAuthorityVo>> authorityVoMap = authorityVoList.stream().collect(
                Collectors.groupingBy(UrlAddressAuthorityVo::getBusinessKey));
        Map<String, List<String>> authorityMap = urlAddressAuthorityList.stream().collect(
                Collectors.groupingBy(UrlAddressAuthority::getBusinessKey,
                        Collectors.mapping(UrlAddressAuthority::getBusinessCode, Collectors.toList())));
        urlAddressVo.setAuthorityMap(authorityMap);
        urlAddressVo.setInterfaceList(authorityVoMap.getOrDefault(BusinessKeyEnum.BUSINESS_INTERFACE.getCode(), Lists.newArrayList()));
        return urlAddressVo;
    }


    /**
     * 获取redis的key
     *
     * @param key
     * @return java.lang.String
     * @author huxmld
     * @version v1.0.0
     * @date 2024/5/9 16:54
     */
    @Override
    public String getRedisKey(String key) {
        Assert.hasLength(key, "redis key must not null!");
        return AuthConstant.EXTERNAL_CACHE_REDIS + key;
    }


}
