package com.biz.crm.business.common.auth.local.service.internal;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.auth.local.entity.UrlAddress;
import com.biz.crm.business.common.auth.local.entity.UrlAddressAuthority;
import com.biz.crm.business.common.auth.local.repository.UrlAddressAuthorityRepository;
import com.biz.crm.business.common.auth.local.repository.UrlAddressRepository;
import com.biz.crm.business.common.auth.local.service.UrlAddressService;
import com.biz.crm.business.common.auth.sdk.constans.AuthConstant;
import com.biz.crm.business.common.auth.sdk.exception.SignException;
import com.biz.crm.business.common.auth.sdk.service.UrlApiService;
import com.biz.crm.business.common.auth.sdk.vo.UrlAddressVo;
import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 获取 加密字符串 service
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.11.3 14:44
 */
@Slf4j
@Service
public class UrlApiServiceImpl implements UrlApiService {


    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    @Autowired(required = false)
    private UrlAddressRepository urlAddressRepository;

    @Autowired(required = false)
    private UrlAddressAuthorityRepository urlAddressAuthorityRepository;

    @Autowired(required = false)
    private UrlAddressService urlAddressService;

    /**
     * 执行内容缓存
     */
    private static volatile Cache<String, UrlAddressVo> cache = null;

    public UrlApiServiceImpl() {
        if (cache == null) {
            synchronized (UrlApiServiceImpl.class) {
                while (cache == null) {
                    cache = CacheBuilder.newBuilder().initialCapacity(10000)
                            .expireAfterWrite(AuthConstant.CACHE_TIME, TimeUnit.MINUTES).maximumSize(100000).build();
                }
            }
        }
    }

    /**
     * 通过(账号|数据字典)获取
     *
     * @param account|account 数据字典|账号
     * @return com.biz.crm.external.business.common.vo.UrlAddressVo
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.3 14:45
     */
    @Override
    public UrlAddressVo getUrlAddressByAccount(String account) {
        Assert.hasLength(account, "参数[account]不能为空!");
        String redisKey = urlAddressService.getRedisKey(account);
        UrlAddressVo urlAddressVo = cache.getIfPresent(account);
        if (Objects.isNull(urlAddressVo)) {
            urlAddressVo = (UrlAddressVo) redisService.get(redisKey);
            if (Objects.nonNull(urlAddressVo)) {
                cache.put(account, urlAddressVo);
            }
        }

        if (Objects.isNull(urlAddressVo)) {
            UrlAddress urlAddress = urlAddressRepository.getUrlAddressByAccount(account);
            urlAddressVo = new UrlAddressVo();
            urlAddressVo.setUserName(account);
            cache.put(account, urlAddressVo);
            urlAddressVo = this.buildUrlVo(urlAddress);
            cache.put(account, urlAddressVo);
            this.urlVerification(urlAddressVo);
            cache.put(account, urlAddressVo);
            redisService.set(redisKey, urlAddressVo, DateUtil.HOUR_IN_SECOND * urlAddressVo.getCacheRedisTime());
        }
        return urlAddressVo;
    }

    /**
     * 通过系统标记[accessKey] 获取
     *
     * @param accessKey 系统标记
     * @return com.biz.crm.external.business.common.vo.UrlAddressVo
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.3 14:45
     */
    @Override
    public UrlAddressVo getUrlAddressByAccessKey(String accessKey) {
        Assert.hasLength(accessKey, "参数[ak]不能为空!");
        String redisKey = urlAddressService.getRedisKey(accessKey);
        UrlAddressVo urlAddressVo = cache.getIfPresent(accessKey);
        if (Objects.isNull(urlAddressVo)) {
            urlAddressVo = (UrlAddressVo) redisService.get(redisKey);
            if (Objects.nonNull(urlAddressVo)) {
                cache.put(accessKey, urlAddressVo);
            }
        }

        if (Objects.isNull(urlAddressVo)) {
            UrlAddress urlAddress = urlAddressRepository.getUrlAddressByAccessKey(accessKey);
            urlAddressVo = new UrlAddressVo();
            urlAddressVo.setAccessKey(accessKey);
            cache.put(accessKey, urlAddressVo);
            urlAddressVo = buildUrlVo(urlAddress);
            cache.put(accessKey, urlAddressVo);
            this.urlVerification(urlAddressVo);
            cache.put(accessKey, urlAddressVo);
            redisService.set(redisKey, urlAddressVo, DateUtil.HOUR_IN_SECOND * urlAddressVo.getCacheRedisTime());
        }
        return urlAddressVo;

    }

    /**
     * 根据系统标记获取系统账号信息
     *
     * @param accessKey 系统标记
     * @return java.lang.String
     * @author huxmld
     * @version v1.0.0
     * @date 2022.12.10 17:20
     */
    @Override
    public String getUserNameByAccessKey(String accessKey) {
        return this.getUrlAddressByAccessKey(accessKey).getUserName();
    }

    /**
     * URL信息验证
     *
     * @param urlAddressVo url信息
     */
    private void urlVerification(UrlAddressVo urlAddressVo) {
        Assert.notNull(urlAddressVo, "[" + SignException.NO_PERMISSION.getCode() + "]" + SignException.NO_PERMISSION.getDescription());
        Assert.isTrue(EnableStatusEnum.ENABLE.getCode().equals(urlAddressVo.getEnableStatus()),
                "[" + SignException.AK_DISABLE.getCode() + "]" + SignException.AK_DISABLE.getDescription());
        Assert.hasLength(urlAddressVo.getAccessKey(),
                "当前ak[" + urlAddressVo.getAccessKey() + "]未配置accessKey信息!");
        Assert.hasLength(urlAddressVo.getSecretKey(),
                "当前ak[" + urlAddressVo.getAccessKey() + "]未配置secretKey信息!");
        Assert.hasLength(urlAddressVo.getUserName(),
                "当前ak[" + urlAddressVo.getAccessKey() + "]未配置账号信息!");
        Assert.hasLength(urlAddressVo.getUrl(),
                "当前ak[" + urlAddressVo.getAccessKey() + "]未配置URL信息,访问失败!");
        String url = urlAddressVo.getUrl();
        if (url.endsWith(AuthConstant.URL_TAIL)) {
            urlAddressVo.setUrl(url.substring(0, url.length() - 1));
        }
        String mobileUrl = urlAddressVo.getMobileUrl();
        if (StringUtils.isNotEmpty(mobileUrl)
                && mobileUrl.endsWith(AuthConstant.URL_TAIL)) {
            urlAddressVo.setMobileUrl(mobileUrl.substring(0, mobileUrl.length() - 1));
        }
        List<UrlAddressAuthority> urlAddressAuthorityList = urlAddressAuthorityRepository.getByUserName(urlAddressVo.getUserName());
        if (CollectionUtil.isEmpty(urlAddressAuthorityList)) {
            return;
        }

        Map<String, List<String>> authorityMap = urlAddressAuthorityList.stream().collect(
                Collectors.groupingBy(UrlAddressAuthority::getBusinessKey,
                        Collectors.mapping(UrlAddressAuthority::getBusinessCode, Collectors.toList())));
        urlAddressVo.setAuthorityMap(authorityMap);

    }

    /**
     * 构建url信息Vo
     *
     * @param urlAddress
     * @return
     */
    private UrlAddressVo buildUrlVo(UrlAddress urlAddress) {
        Assert.notNull(urlAddress, "[" + SignException.NO_PERMISSION.getCode() + "]" + SignException.NO_PERMISSION.getDescription());
        UrlAddressVo urlAddressVo = this.nebulaToolkitService.copyObjectByWhiteList(urlAddress, UrlAddressVo.class,
                HashSet.class, ArrayList.class);
        //系统缓存时间,默认3分钟
        if (Objects.isNull(urlAddressVo.getCacheTime())) {
            urlAddressVo.setCacheTime(AuthConstant.DEFAULT_CACHE_TIME);
        }
        //Redis缓存时间,默认3小时
        if (Objects.isNull(urlAddressVo.getCacheRedisTime())) {
            urlAddressVo.setCacheRedisTime(AuthConstant.DEFAULT_CACHE_TIME);
        }
        //签名有效期,默认3分钟
        if (Objects.isNull(urlAddressVo.getSignExpireDate())) {
            urlAddressVo.setSignExpireDate(AuthConstant.DEFAULT_SIGN_EXPIRE_DATE);
        }
        //访问次数限制,默认400
        if (Objects.isNull(urlAddressVo.getAccessTimesMax())) {
            urlAddressVo.setAccessTimesMax(AuthConstant.DEFAULT_ACCESS_TIMES_MAX);
        }
        urlAddressVo.setCacheTime(Math.max(urlAddressVo.getCacheTime(), 1));
        urlAddressVo.setCacheRedisTime(Math.max(urlAddressVo.getCacheRedisTime(), 1));
        urlAddressVo.setSignExpireDate(Math.max(urlAddressVo.getSignExpireDate(), 1));
        urlAddressVo.setAccessTimesMax(Math.max(urlAddressVo.getAccessTimesMax(), 1));
        return urlAddressVo;
    }

}
