package com.biz.crm.business.common.auth.sdk.utils;

import cn.hutool.core.util.XmlUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.data.util.Pair;
import org.springframework.http.*;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Component
@Slf4j
public class SapUtil {



    private static final ThreadLocal<String> logMap = new ThreadLocal<>();

    private static final RestTemplate restTemplate = new RestTemplate();

    static {
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = new MappingJackson2HttpMessageConverter();
        mappingJackson2HttpMessageConverter.setSupportedMediaTypes(Arrays.asList(MediaType.TEXT_HTML));
        restTemplate.getMessageConverters().add(mappingJackson2HttpMessageConverter);
        restTemplate.getMessageConverters().set(1, new StringHttpMessageConverter(StandardCharsets.UTF_8));
    }
    /**
     * 请求参数占位符
     */
    public static final String REQUEST_PLACEHOLDER = "请求参数占位符";



    public static void setUpLog(String prefix) {
        logMap.set(prefix);
    }


    public static <T, R> R send(String url, T param,String username, String password) {
        long tid = Thread.currentThread().getId();
        log.info("tid:{},sap入参url:{}", tid, url);
        log.info("tid:{},sap入参param:{}", tid, param);

        HttpHeaders headers = new HttpHeaders();
        setToken(headers,username,password);
        HttpEntity<T> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<R> exchange = null;
        try {
            exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<R>() {
            });
        } catch (RestClientException e) {
            storeLog(url, String.valueOf(param),String.valueOf(e.getCause()));
            throw new RuntimeException(e);
        }
        log.info("tid:{},sap响应result:{}", tid, exchange);
        storeLog(url, String.valueOf(param),String.valueOf(exchange.getBody()));
        R body = exchange.getBody();
        log.info("tid:{},body:{}", tid, exchange);
        Assert.notNull(body, "缺失返回数据");
        return body;
    }

    public static <T, R> R sendConvertParam(String url, T param, Class<T> clazz,String username, String password) {
        long tid = Thread.currentThread().getId();
        log.info("tid:{},sap入参url:{}", tid, url);
        log.info("tid:{},sap入参param:{}", tid, param);
        String p = convertParam(param, clazz);

        HttpHeaders headers = new HttpHeaders();
        setToken(headers,username,password);
        HttpEntity<String> httpEntity = new HttpEntity<>(p, headers);
        ResponseEntity<R> exchange = null;
        try {
            exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<R>() {
            });
        } catch (RestClientException e) {
            storeLog(url, String.valueOf(param),String.valueOf(e.getCause()));
            throw new RuntimeException(e);
        }
        log.info("tid:{},sap响应result:{}", tid, exchange);
        storeLog(url, p,String.valueOf(exchange.getBody()));
        R body = exchange.getBody();
        log.info("tid:{},body:{}", tid, exchange);
        Assert.notNull(body, "缺失返回数据");
        return body;
    }

    public static String sendParam(String url, String param,String username, String password) {
        String prefix = logMap.get();
        prefix = !StringUtils.hasText(prefix) ? "" : prefix;
        ResponseEntity<String> exchange = null;
        try {
            log.info(prefix + "sap接口sap入参url:{}", url);
            log.info(prefix + "sap入参:{}", param);
            HttpHeaders headers = new HttpHeaders();
            setToken(headers,username,password);
            HttpEntity<String> httpEntity = new HttpEntity<>(param, headers);
            exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, String.class);
            log.info(prefix + "sap响应:{}", exchange);
            String body = exchange.getBody();
            Assert.hasText(body, "缺失返回数据");
            return body;
        } finally {
            logMap.remove();
            storeLog(url, param,String.valueOf(exchange));
        }
    }

    /**
     * 通过类与注解请求
     *
     * @param url
     * @param templateXml
     * @param t
     * @param tClazz
     * @param <T>
     * @return
     */
    public static <T> String sendParamByClass(String url, String templateXml, T t, Class<T> tClazz,String username, String password) {

        StringBuilder paramXml = new StringBuilder();
        generateXmlByClass(t, tClazz, paramXml, "");
        String requestXml = templateXml.replace(REQUEST_PLACEHOLDER, paramXml.toString());
        String response = sendParam(url, requestXml,username,password);
        return response;
    }

    /**
     * 通过类与注解请求 返回单个目标类
     *
     * @param url
     * @param templateXml
     * @param t
     * @param tClazz
     * @param <T>
     * @return
     */
    public static <T, R> R sendParamByClass(String url, String templateXml, T t, Class<T> tClazz, Class<R> rClazz, String tag,String username, String password) {
        Validate.notBlank(tag, "结果选取标签不能为空");
        StringBuilder paramXml = new StringBuilder();
        generateXmlByClass(t, tClazz, paramXml, "");
        String requestXml = templateXml.replace(REQUEST_PLACEHOLDER, paramXml.toString());
        String response = sendParam(url, requestXml,username,password);
        return string2Class(response, tag, rClazz, true);
    }

    /**
     * 通过类与注解请求 返回多个目标类
     *
     * @param url
     * @param templateXml
     * @param t
     * @param tClazz
     * @param <T>
     * @return
     */
    public static <T, R> List<R> sendParamByClassBatch(String url, String templateXml, T t, Class<T> tClazz, Class<R> rClazz, String tag,String username, String password) {
        Validate.notBlank(tag, "结果选取标签不能为空");
        StringBuilder paramXml = new StringBuilder();
        generateXmlByClass(t, tClazz, paramXml, "");
        String requestXml = templateXml.replace(REQUEST_PLACEHOLDER, paramXml.toString());
        String response = sendParam(url, requestXml,username,password);
        return string2ClassBatch(response, tag, rClazz);
    }

    /**
     * 批量转换数据  从List<Map>转换成List<Obj>
     *
     * @param clazz
     * @param <R>
     * @return
     */
    public static <R> List<R> string2ClassBatch(String response, String tag, Class<R> clazz) {
        Pattern pattern = Pattern.compile("<" + tag + ">" + "[\\s\\S]*?" + "</" + tag + ">");
        Matcher matcher = pattern.matcher(response);
        List<R> reList = new ArrayList<>();
        while (matcher.find()) {
            reList.add(string2Class(matcher.group(), tag, clazz, false));
        }
        return reList;
    }

    /**
     * 节点数据
     */
    private static class NodeData {
        /**
         * 单个数据
         */
        private String singleData;
        /**
         * 多条数据 存list item为key到data映射
         */
        private LinkedList<Map<String, String>> multiData;

        public NodeData(String singleData) {
            this.singleData = singleData;
        }

        public NodeData(LinkedList<Map<String, String>> multiData) {
            this.multiData = multiData;
        }

        public String getSingleData() {
            return singleData;
        }

        public void setSingleData(String singleData) {
            this.singleData = singleData;
        }

        public LinkedList<Map<String, String>> getMultiData() {
            return multiData;
        }

        public void setMultiData(LinkedList<Map<String, String>> multiData) {
            this.multiData = multiData;
        }
    }

    /**
     * 转换数据  从字符串转换成实例
     *
     * @param
     */
    public static <R> R string2Class(String response, String tag, Class<R> clazz, boolean useMatch) {
        String matchResponse = "";
        if (useMatch) {
            Pattern pattern = Pattern.compile("<" + tag + ">" + "[\\s\\S]*?" + "</" + tag + ">");
            Matcher matcher = pattern.matcher(response);
            while (matcher.find()) {
                matchResponse = matcher.group();
            }
        } else {
            matchResponse = response;
        }

        Document document = XmlUtil.parseXml(matchResponse);
        Element rootElement = XmlUtil.getRootElement(document);
        Map<String, NodeData> prefix2Data = new HashMap<>();

        //节点栈
        Stack<Node> nodeStack = new Stack<>();
        //获取tag的所有子节点进行压栈
        NodeList nodeList = rootElement.getChildNodes();
        //节点到路径映射
        Map<Node, String> node2Path = new HashMap<>();
        //压栈
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);
            node2Path.put(node, node.getLocalName());
            nodeStack.push(nodeList.item(i));
        }
        //进行处理直到栈为空
        while (!nodeStack.isEmpty()) {
            //出栈一个节点
            Node node = nodeStack.pop();
            //获取当前节点的路径
            String currentNodePath = node2Path.get(node);
            //如果不是最后一个节点则对所有子节点进行压栈
            if (node.hasChildNodes()) {
                NodeList children = node.getChildNodes();
                for (int i = 0; i < children.getLength(); i++) {
                    //孩子节点
                    Node childNode = children.item(i);
                    nodeStack.push(children.item(i));
                    if (childNode.hasChildNodes()) {
                        node2Path.put(childNode, currentNodePath + "." + childNode.getLocalName());
                    } else {
                        node2Path.put(childNode, currentNodePath);
                    }
                }
            } else {
                //获取父路径
                String parentPath = node2Path.get(node.getParentNode());
                //当父路径不为空 且存在父路径的数据时，表明当前节点存在多个数据，因此需要保存为list
                if (null != parentPath && prefix2Data.containsKey(parentPath)) {
                    //获取父路径存的数据
                    NodeData nodeData = prefix2Data.get(parentPath);
                    String[] currentPathSplit = currentNodePath.split(".");
                    //取最后一个路径
                    String currentLastPath = currentPathSplit[currentNodePath.length() - 1];
                    //对第一次添加的数据进行特殊处理
                    //当前路径存在另外的数据时，将其取出来放到nodeData中的multiData第一个元素的map中，并将当前元素移除
                    if (prefix2Data.containsKey(currentNodePath)) {
                        nodeData.getMultiData().get(0).put(currentLastPath, prefix2Data.get(currentNodePath).getSingleData());
                        prefix2Data.remove(currentNodePath);
                    }

                    //list最后一个节点包含当前key则表示已存在数据，因此multiData需要追加数据
                    if (nodeData.getMultiData().getLast().containsKey(currentLastPath)) {
                        Map<String, String> newMultiData = new HashMap<>();
                        newMultiData.put(currentLastPath, node.getTextContent());
                        nodeData.getMultiData().add(newMultiData);
                    } else {
                        nodeData.getMultiData().getLast().put(currentLastPath, node.getTextContent());
                    }
                } else {

                    //对第一次添加的数据进行特殊处理
                    //当前路径存在另外的数据时，将其取出来放到nodeData中的multiData第一个元素的map中，并将当前元素移除
                    if (prefix2Data.containsKey(currentNodePath)) {
                        String[] currentPathSplit = currentNodePath.split(".");
                        //取最后一个路径
                        String currentLastPath = currentPathSplit[currentNodePath.length() - 1];

                        LinkedList<Map<String, String>> multiData = new LinkedList<>();
                        //将第一条数据加进去
                        Map<String, String> multiDataItem1 = new HashMap<>();
                        multiDataItem1.put(currentLastPath, prefix2Data.get(currentNodePath).getSingleData());
                        multiData.add(multiDataItem1);
                        //移除原数据
                        prefix2Data.remove(currentNodePath);
                        //当前数据
                        Map<String, String> multiDataItem2 = new HashMap<>();
                        multiDataItem2.put(currentLastPath, node.getTextContent());
                        multiData.add(multiDataItem2);

                        NodeData nodeData = new NodeData(multiData);
                        //注意这里有问题 对于那种父没有的  出问题了再处理吧   不想动了  暂时能用就行
                        prefix2Data.put(parentPath, nodeData);
                    } else {
                        //需要判断当前节点是否存在数据了，如果存在数据需要放多个，并删除原数据
                        //直接往里面加数据即可
                        //最后一个节点进行数据保存
                        NodeData nodeData = new NodeData(node.getTextContent());
                        prefix2Data.put(currentNodePath, nodeData);
                    }
                }
            }
        }


        R obj = null;
        try {
            obj = clazz.newInstance();
        } catch (Exception e) {
            log.info("实例化失败");
            return null;
        }
        Field[] fields = clazz.getDeclaredFields();

        for (int i = 0; i < fields.length; i++) {
            Field field = fields[i];
            SapFormField sapFormField = field.getAnnotation(SapFormField.class);
            //未加注解的不处理
            if (null == sapFormField) {
                continue;
            }
            try {
                //对于列表数据
                if (sapFormField.list()) {
                    field.setAccessible(true);
                    List dataList = nodeData2DataList(prefix2Data, sapFormField);
                    field.set(obj, dataList);
                } else {
                    NodeData data = prefix2Data.get(sapFormField.path());
                    if (null != data) {
                        setData(field, data.getSingleData(), obj);
                    }
                }

            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }

        return obj;
    }

    /**
     * 设置数据
     *
     * @param field
     */
    private static void setData(Field field, String data, Object obj) throws IllegalAccessException {
        field.setAccessible(true);
        if (null == data) {
            return;
        }
        if (field.getType() == Integer.class) {
            field.set(obj, Integer.valueOf(data.trim()));
        }

        if (field.getType() == BigDecimal.class) {
            BigDecimal b = null;
            try{
                b = new BigDecimal(data.trim());
            }catch (Exception e){
                log.error("数据赋值异常:", e);
                b = BigDecimal.ZERO;
            }
            field.set(obj,b);
        }

        if (field.getType() == String.class) {
            field.set(obj, data.trim());
        }
    }

    /**
     * nodeData转换为class
     *
     * @return
     */
    private static List<Object> nodeData2DataList(Map<String, NodeData> prefix2Data, SapFormField parentSapFormField) {
        List<Object> dataList = new ArrayList<>();

        //对于列表数据先查看是否父路径存在数据
        NodeData nodeData = prefix2Data.get(parentSapFormField.path());
        //获取字段
        Field[] fields = parentSapFormField.itemClazz().getDeclaredFields();
        //建立字段到注解映射
        Map<Field, SapFormField> field2SapFormField = new HashMap<>();
        for (Field field : fields) {
            SapFormField sapFormField = field.getAnnotation(SapFormField.class);
            if (null == sapFormField) {
                continue;
            }
            field2SapFormField.put(field, sapFormField);
        }
        //父没有数据,表明list只有一条数据,直接从子进行取数据
        if (null == nodeData) {
            Object obj = null;
            try {
                obj = parentSapFormField.itemClazz().newInstance();
                for (Field field : fields) {
                    SapFormField sapFormField = field2SapFormField.get(field);
                    if (null == sapFormField) {
                        continue;
                    }
                    NodeData data = prefix2Data.get(parentSapFormField.path() + "." + sapFormField.path());
                    if (null != data) {
                        setData(field, data.getSingleData(), obj);
                    }
                }

                dataList.add(obj);
            } catch (Exception e) {
                log.error("sap工具类返回生成实例错误");
            }
        } else {
            //父有数据时，直接从父的多个数据拿生成
            for (Map<String, String> path2Data : nodeData.getMultiData()) {
                Object obj = null;
                try {
                    obj = parentSapFormField.itemClazz().newInstance();
                    for (Field field : fields) {
                        SapFormField sapFormField = field2SapFormField.get(field);
                        if (null == sapFormField) {
                            continue;
                        }
                        String data = path2Data.get(sapFormField.path());
                        if (null != data) {
                            setData(field, data, obj);
                        }
                    }
                    dataList.add(obj);
                } catch (Exception e) {
                    log.error("sap工具类返回生成实例错误");
                }
            }

        }

        return dataList;
    }

    public static String sendParamBody(String url, String methodName, String methodNameSpace, String param,String username, String password) {
        String p = "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:cass=\"http://cofco.com/mm/cass\">\n" +
                "   <soapenv:Header/>\n" +
                "   <soapenv:Body>\n" +
                "      <" + methodName + " " + methodNameSpace + ">\n" +
                "         <I_REQUEST>\n"
                + param +
                "         </I_REQUEST>\n" +
                "      </" + methodName + ">\n" +
                "   </soapenv:Body>\n" +
                "</soapenv:Envelope>";


        long tid = Thread.currentThread().getId();
        log.info("tid:{},sap入参url:{}", tid, url);
        log.info("tid:{},sap入参param:{}", tid, p);
        HttpHeaders headers = new HttpHeaders();
        setToken(headers,username, password);
        HttpEntity<String> httpEntity = new HttpEntity<>(p, headers);
        ResponseEntity<String> exchange = null;
        try {
            exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, String.class);
        } catch (RestClientException e) {
            storeLog(url, p,String.valueOf(e.getCause()));
            throw new RuntimeException(e);
        }
        log.info("tid:{},sap响应result:{}", tid, exchange);
        storeLog(url, p,String.valueOf(exchange.getBody()));
        String body = exchange.getBody();
        log.info("tid:{},body:{}", tid, exchange);
        Assert.hasText(body, "缺失返回数据");
        return body;
    }

    private static void setToken(HttpHeaders headers,String username, String password) {
        headers.set("Content-type", "text/xml; charset=utf-8");
        headers.set("Accept", "text/xml; charset=utf-8");
        headers.add("authorization", "Basic " + HttpHeaders.encodeBasicAuth(username, password, null));
    }

    public static List<Map<String, String>> sendMap(String url
            , String targetNameSpace
            , String methodName
            , String methodNameSpace
            , Map<String, Object> param
            , String receiveTag,String username, String password) {
        StringBuilder p = new StringBuilder();
        p.append("<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:cass=\"").append(targetNameSpace).append("\">");
        p.append("<soapenv:Header/>");
        p.append("<soapenv:Body>");
        p.append("<").append(methodName);
        if (StringUtils.hasText(methodNameSpace)) {
            p.append(" ").append(methodNameSpace);
        }
        p.append(">");

        StringBuilder sb = new StringBuilder();
        mapToXML(param, sb);
        p.append(sb);

        p.append("</").append(methodName).append(">");
        p.append("</soapenv:Body>");
        p.append("</soapenv:Envelope>");

        String result = sendParam(url, p.toString(), username, password);
        return resolveDocument(receiveTag, result);
    }


    public static <T> List<Map<String, String>> sendMap(String url, T t, Class<T> clazz, String tagName,String username, String password) {
        String result = sendParam(url, convertParam(t, clazz), username, password);
        return resolveDocument(tagName, result);
    }

    /**
     * sendMap另一层封装 批量的返回数组  注意 returnClass必须要做字段对应使用 {@link SapFormField}注解
     *
     * @param url
     * @param t
     * @param tagName
     * @param returnClass
     * @param <T>
     * @param <R>
     * @return
     */
    public static <T, R> List<R> sendMapBatch(String url, T t, String tagName, Class<R> returnClass, Class<T> tClass,String username, String password) {
        String result = sendParam(url, convertParam(t, tClass), username, password);
        List<Map<String, String>> key2DataList = resolveDocument(tagName, result);
        return setResponseDataBatch(key2DataList, returnClass);
    }

    /**
     * 另一层封装 只返回单个  注意 returnClass必须要做字段对应使用 {@link SapFormField}注解
     *
     * @param url
     * @param t
     * @param tagName
     * @param returnClass
     * @param <T>
     * @param <R>
     * @return
     */
    public static <T, R> R sendMap(String url, T t, String tagName, Class<R> returnClass, Class<T> tClass,String username, String password) {
        String result = sendParam(url, convertParam(t, tClass), username, password);
        List<Map<String, String>> key2DataList = resolveDocument(tagName, result);
        if (CollectionUtils.isEmpty(key2DataList)) {
            return null;
        }
        return setResponseData(key2DataList.get(0), returnClass);
    }

    public static List<Map<String, String>> resolveDocument(String tagName, String result) {
        List<Map<String, String>> list = new ArrayList<>();
        Document document = XmlUtil.parseXml(result);
        Element rootElement = XmlUtil.getRootElement(document);
        NodeList nodeList = rootElement.getElementsByTagName(tagName);

        for (int i = 0; i < nodeList.getLength(); i++) {
            Node item = nodeList.item(i);
            NodeList childNodes = item.getChildNodes();
            Map<String, String> tmpMap = new HashMap<>();
            for (int j = 0; j < childNodes.getLength(); j++) {
                Node tmp = childNodes.item(j);
                if (StringUtils.hasText(tmp.getLocalName())) {
                    tmpMap.put(tmp.getLocalName(), tmp.getTextContent().trim());
                }
            }
            if (!CollectionUtils.isEmpty(tmpMap)) {
                list.add(tmpMap);
            }
        }
        return list;
    }

    /**
     * 根据class动态生成xml
     *
     * @param t
     * @param clazz
     * @param p
     * @param basicName 基本数据类型的 属性名
     * @param <T>
     */
    public static <T> void generateXmlByClass(T t, Class<T> clazz, StringBuilder p, String basicName) {
        //没列举完 需要什么再加什么吧
        if (String.class == clazz || Integer.class == clazz || Long.class == clazz) {
            p.append("<").append(basicName).append(">");
            p.append(t);
            p.append("</" + basicName + ">");
            return;
        }
        Field[] fields = clazz.getDeclaredFields();
        //数据根据前缀分组  目前仅支持一层前缀
        Map<String, List<Pair<String, Object>>> dataGroupByPrefix = new HashMap<>();
        for (Field field : fields) {
            XmlFieldProperty anno = field.getAnnotation(XmlFieldProperty.class);
            if (anno == null) {
                continue;
            }
            String name = anno.name();
            Assert.hasText(name, field.getName() + "缺失属性名称");

            try {
                field.setAccessible(true);
                Object data = field.get(t);
                if (null != data) {
                    //是list
                    if (anno.list()) {
                        generateXmlBatchByClass((List) data, anno.itemClass(), p, anno.name());
                        continue;
                    }

                    String[] fieldArr = name.split("\\.");
                    Validate.isTrue(fieldArr.length <= 2, "目前仅支持2层前缀");
                    //一层的直接包起来  多层的同一个前缀的放一起  先取出来后面统一放一起
                    if (fieldArr.length == 1) {
                        p.append("<").append(fieldArr[0]).append(">");
                        p.append(data);
                        p.append("</" + fieldArr[0] + ">");
                    } else {
                        List<Pair<String, Object>> key2Data = dataGroupByPrefix.getOrDefault(fieldArr[0], new ArrayList<>());
                        key2Data.add(Pair.of(fieldArr[1], data));
                        dataGroupByPrefix.put(fieldArr[0], key2Data);
                    }
                }
            } catch (Exception e) {
                log.error("获取数据失败");
                e.printStackTrace();
            }
        }

        //同一个前缀的放一起
        if (!CollectionUtils.isEmpty(dataGroupByPrefix)) {
            dataGroupByPrefix.forEach((prefix, dataList) -> {
                //大的前缀
                p.append("<").append(prefix).append(">");

                //详细数据
                Set<String> repeatSet = new HashSet<>();
                dataList.forEach(e -> {
                    String first = e.getFirst();
                    //重复数据过滤
                    if (repeatSet.add(first)) {
                        p.append("<").append(first).append(">");
                        p.append(e.getSecond());
                        p.append("</" + first + ">");
                    }
                });

                p.append("</" + prefix + ">");
            });
        }
    }

    /**
     * 批量生成xml
     *
     * @param dataList
     * @param clazz
     * @param p
     * @param <T>
     */
    public static <T> void generateXmlBatchByClass(List<T> dataList, Class<T> clazz, StringBuilder p, String basicName) {
        if (CollectionUtils.isEmpty(dataList)) {
            return;
        }
        Validate.notNull(clazz, "明细Class不能为空");
        //循环生成
        for (T t : dataList) {
            generateXmlByClass(t, clazz, p, basicName);
        }
    }

    public static <T> String convertParam(T t, Class<T> clazz) {
        StringBuilder p = new StringBuilder();
        XmlRootProperty clazzAnno = clazz.getAnnotation(XmlRootProperty.class);
        Assert.notNull(clazzAnno, "类上缺失【XmlRootProperty】注解");
        p.append("<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:dms=\"").append(clazzAnno.targetNameSpace()).append("\">");
        p.append("<soapenv:Header/>");
        p.append("<soapenv:Body>");
        //原来的 注释掉
//        p.append("<").append(clazzAnno.methodName()).append(">");
        p.append("<").append(clazzAnno.methodName());
        if (StringUtils.hasText(clazzAnno.methodNameSpace())) {
            p.append(" ").append(clazzAnno.methodNameSpace());
        }
        p.append(">");
        p.append("<MESSAGE>");
        String[] tags = clazzAnno.tags();
        Deque<String> tagStack = new LinkedList<>();
        if (tags != null) {
            for (String tag : tags) {
                p.append("<").append(tag).append(">");
                tagStack.push("</" + tag + ">");
            }
        }
        //根据class生成xml
        generateXmlByClass(t, clazz, p, "");
        while (!tagStack.isEmpty()) {
            p.append(tagStack.pop());
        }

        p.append("</MESSAGE>");
        p.append("</").append(clazzAnno.methodName()).append(">");
        p.append("</soapenv:Body>");
        p.append("</soapenv:Envelope>");

        return p.toString();
    }

    /**
     * @param t
     * @param clazz
     * @return {@link String}
     */
    public static <T> String convertParam(T t, Class<T> clazz, String targetNameSpace) {
        StringBuilder p = new StringBuilder();
        XmlRootProperty clazzAnno = clazz.getAnnotation(XmlRootProperty.class);
        Assert.notNull(clazzAnno, "类上缺失【XmlRootProperty】注解");
        p.append("<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" ").append(targetNameSpace).append(">");
        p.append("<soapenv:Header/>");
        p.append("<soapenv:Body>");
        //原来的 注释掉
        p.append("<").append(clazzAnno.methodName());
        if (StringUtils.hasText(clazzAnno.methodNameSpace())) {
            p.append(" ").append(clazzAnno.methodNameSpace());
        }
        p.append(">");

        p.append("<I_REQUEST>");
        p.append("<MESSAGE>");

        String[] tags = clazzAnno.tags();
        Deque<String> tagStack = new LinkedList<>();
        if (tags != null) {
            for (String tag : tags) {
                p.append("<").append(tag).append(">");
                tagStack.push("</" + tag + ">");
            }
        }

        //根据class生成xml
        generateXmlByClass(t, clazz, p, "");

        while (!tagStack.isEmpty()) {
            p.append(tagStack.pop());
        }

        p.append("</MESSAGE>");
        p.append("</I_REQUEST>");
        p.append("</").append(clazzAnno.methodName()).append(">");
        p.append("</soapenv:Body>");
        p.append("</soapenv:Envelope>");

        return p.toString();
    }

    public static void mapToXML(Map<?, ?> map, StringBuilder sb) {
        Set<?> set = map.keySet();
        for (Iterator<?> it = set.iterator(); it.hasNext(); ) {
            String key = (String) it.next();
            Object value = map.get(key);
            if (value instanceof Map) {
                sb.append("<").append(key).append(">\n");
                mapToXML((Map<?, ?>) value, sb);
                sb.append("</").append(key).append(">\n");
            } else if (value instanceof List) {
                List<?> list = (List<?>) map.get(key);
                for (int i = 0; i < list.size(); i++) {
                    sb.append("<").append(key).append(">\n");
                    Map<?, ?> hm = (Map<?, ?>) list.get(i);
                    mapToXML(hm, sb);
                    sb.append("</").append(key).append(">\n");
                }
            } else {
                sb.append("<").append(key).append(">").append(value).append("</").append(key).append(">\n");
            }
        }
    }

    /**
     * 转换数据  从map转换成实例
     *
     * @param key2Data
     */
    public static <R> R setResponseData(Map<String, String> key2Data, Class<R> clazz) {
        R obj = null;
        try {
            obj = clazz.newInstance();
        } catch (Exception e) {
            log.info("实例化失败");
            return null;
        }
        Field[] fields = clazz.getDeclaredFields();

        for (int i = 0; i < fields.length; i++) {
            Field field = fields[i];
            SapFormField sapFormField = field.getAnnotation(SapFormField.class);
            //未加注解的不处理
            if (null == sapFormField) {
                continue;
            }
            try {
                field.setAccessible(true);
                field.set(obj, key2Data.get(sapFormField.key()));
            } catch (IllegalAccessException e) {
                e.printStackTrace();
            }
        }

        return obj;
    }

    /**
     * 批量转换数据  从List<Map>转换成List<Obj>
     *
     * @param key2DataList
     * @param clazz
     * @param <R>
     * @return
     */
    public static <R> List<R> setResponseDataBatch(List<Map<String, String>> key2DataList, Class<R> clazz) {
        if (CollectionUtils.isEmpty(key2DataList)) {
            return Lists.newArrayList();
        }
        List<R> reList = new ArrayList<>(key2DataList.size());
        key2DataList.forEach(key2Data -> reList.add(SapUtil.setResponseData(key2Data, clazz)));
        return reList;
    }

    private static void storeLog(String url, String p,String r) {
        p = "接口地址:" + url + "\n参数:" + p;
//        ApiLogUtil.record("SAP",p,r);
    }

    public static String sendParamBody(String url, String param,String username, String password) {
        String p = "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:cass=\"http://cofco.com/mm/cass\">\n" +
            "   <soapenv:Header/>\n" +
            "   <soapenv:Body>\n" +
            param +
            "   </soapenv:Body>\n" +
            "</soapenv:Envelope>";


        long tid = Thread.currentThread().getId();
        log.info("tid:{},sap入参url:{}", tid, url);
        log.info("tid:{},sap入参param:{}", tid, p);
        HttpHeaders headers = new HttpHeaders();
        setToken(headers, username, password);
        HttpEntity<String> httpEntity = new HttpEntity<>(p, headers);
        ResponseEntity<String> exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, String.class);
        log.info("tid:{},sap响应result:{}", tid, exchange);
        storeLog(url, p,String.valueOf(exchange.getBody()));
        String body = exchange.getBody();
        log.info("tid:{},body:{}", tid, exchange);
        Assert.hasText(body, "缺失返回数据");
        return body;
    }


    /**
     * cass专用
     * @param dataList
     * @param clazz
     * @return
     * @param <T>
     */
    public static <T> String cassConvertParam(List<T> dataList, Class<T> clazz) {
        StringBuilder p = new StringBuilder();
        XmlRootProperty clazzAnno = clazz.getAnnotation(XmlRootProperty.class);
        Assert.notNull(clazzAnno, "类上缺失【XmlRootProperty】注解");
        p.append("<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\" xmlns:tem=\"").append(clazzAnno.targetNameSpace()).append("\">");
        p.append("<soapenv:Header/>");
        p.append("<soapenv:Body>");
        //原来的 注释掉
        p.append("<").append(clazzAnno.methodName());
        if (StringUtils.hasText(clazzAnno.methodNameSpace())) {
            p.append(" ").append(clazzAnno.methodNameSpace());
        }
        p.append(">");
        p.append("<tem:ReturnInfolist>");
        String[] tags = clazzAnno.tags();
        Deque<String> tagStack = new LinkedList<>();
        if (tags != null) {
            for (String tag : tags) {
                p.append("<").append(tag).append(">");
                tagStack.push("</" + tag + ">");
            }
        }
        //根据class生成xml
        for (T t : dataList) {
            p.append("<tem:ReturnInfo>");
            generateXmlByClass(t, clazz, p, "");
            p.append("</tem:ReturnInfo>");
        }
        while (!tagStack.isEmpty()) {
            p.append(tagStack.pop());
        }

        p.append("</tem:ReturnInfolist>");
        p.append("</").append(clazzAnno.methodName()).append(">");
        p.append("</soapenv:Body>");
        p.append("</soapenv:Envelope>");

        return p.toString();
    }


    public static <T, R> R sendNoAuth(String url, T param) {
        long tid = Thread.currentThread().getId();
        log.info("tid:{},sap入参url:{}", tid, url);
        log.info("tid:{},sap入参param:{}", tid, param);

        HttpHeaders headers = new HttpHeaders();
        headers.set("Content-type", "text/xml; charset=utf-8");
        headers.set("Accept", "text/xml; charset=utf-8");
        HttpEntity<T> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<R> exchange = null;
        try {
            exchange = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<R>() {
            });
        } catch (RestClientException e) {
            storeLog(url, String.valueOf(param),String.valueOf(e.getCause()));
            throw new RuntimeException(e);
        }
        log.info("tid:{},sap响应result:{}", tid, exchange);
        storeLog(url, String.valueOf(param),String.valueOf(exchange.getBody()));
        R body = exchange.getBody();
        log.info("tid:{},body:{}", tid, exchange);
        Assert.notNull(body, "缺失返回数据");
        return body;
    }
}
