package com.biz.crm.business.common.base.util;

import cn.hutool.core.collection.CollectionUtil;
import com.biz.crm.business.common.base.vo.RuleSplitVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.beans.PropertyEditorSupport;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.time.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author huxmld
 * @version v1.0.0
 * @describe 时间方法
 * @date 2022.10.29 14:42
 */
@Slf4j
public abstract class DateUtil extends PropertyEditorSupport {

    /**
     * 各种时间格式
     */
    public static final SimpleDateFormat date_yyyy = new SimpleDateFormat("yyyy");
    public static final SimpleDateFormat date_yyyy_MM = new SimpleDateFormat("yyyy-MM");
    public static final SimpleDateFormat date_yy_MM = new SimpleDateFormat("yy-MM");
    public static final SimpleDateFormat date_yyyyMM = new SimpleDateFormat("yyyyMM");
    public static final SimpleDateFormat date_yyyy_MM_dd = new SimpleDateFormat("yyyy-MM-dd");
    public static final SimpleDateFormat date_yyyyMMdd = new SimpleDateFormat("yyyyMMdd");

    public static final SimpleDateFormat date_HH_mm = new SimpleDateFormat("HH:mm");
    public static final SimpleDateFormat date_HH_mm_ss = new SimpleDateFormat("HH:mm:ss");
    public static final SimpleDateFormat date_HH_mm_ss_SSS = new SimpleDateFormat("HH:mm:ss SSS");

    public static final SimpleDateFormat date_yyyy_MM_dd_HH = new SimpleDateFormat("yyyy-MM-dd HH");
    public static final SimpleDateFormat date_yyyy_MM_dd_HH_mm = new SimpleDateFormat("yyyy-MM-dd HH:mm");
    public static final SimpleDateFormat date_yyyy_MM_dd_HH_mm_ss = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    public static final SimpleDateFormat date_yyyyMMddHHmmss = new SimpleDateFormat("yyyyMMddHHmmss");
    public static final SimpleDateFormat date_yyyy_MM_dd_HH_mm_ss_SSS = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss SSS");

    /**
     * 各种时间格式
     */
    public static final SimpleDateFormat date_str_yyyy = new SimpleDateFormat("yyyy年");
    public static final SimpleDateFormat date_str_yyyy_MM = new SimpleDateFormat("yyyy年MM月");
    public static final SimpleDateFormat date_str_yyyy_MM_dd = new SimpleDateFormat("yyyy年MM月dd日");
    public static final SimpleDateFormat date_str_yyyy_MM_dd_HH_mm_ss_SSS = new SimpleDateFormat("yyyy年MM月dd日 HH:mm:ss");
    /**
     * 以秒表示的时间
     */
    public static final long MINUTE_IN_SECOND = 60;
    public static final long HOUR_IN_SECOND = 60 * MINUTE_IN_SECOND;
    public static final long DAY_IN_SECOND = 24 * HOUR_IN_SECOND;


    public static final int MINIMAL_DAYS_IN_FIRST_WEEK = 4;
    public static final int FIRST_DAY_OF_WEEK = 2;

    /**
     * 以毫秒表示的时间
     */
    public static final long SECOND_IN_MILLIS = 1000;
    public static final long MINUTE_IN_MILLIS = MINUTE_IN_SECOND * SECOND_IN_MILLIS;
    public static final long HOUR_IN_MILLIS = HOUR_IN_SECOND * SECOND_IN_MILLIS;
    public static final long DAY_IN_MILLIS = DAY_IN_SECOND * SECOND_IN_MILLIS;

    /**
     * 年月
     */
    public static final String DEFAULT_YEAR_MONTH = "yyyy-MM";
    public static final String DEFAULT_YEAR_MONTH_NO_CH = "yyyyMM";
    public static final String DEFAULT_YEAR_MONTH_PATTERN = "yyyy年MM月";
    /**
     * 年月日
     */
    public static final String DEFAULT_YEAR_MONTH_DAY = "yyyy-MM-dd";
    public static final String DEFAULT_YEAR_MONTH_DAY_NO_CH = "yyyyMMdd";

    /**
     * 日
     */
    public static final String DEFAULT_DAY = "dd";

    /**
     * 年月日时
     */
    public static final String DEFAULT_DATE_PATTERN = "yyyy-MM-dd HH:mm";
    /**
     * 年月日时分
     */
    public static final String DEFAULT_TIME_PATTERN = "HH:mm";
    /**
     * 年月日时分秒
     */
    public static final String DEFAULT_TIME_ALL_PATTERN = "HH:mm:ss";
    public static final String DEFAULT_DATE_TIME_PATTERN = "yyyyMMddHHmmss";
    public static final String DEFAULT_DATE_ALL_PATTERN = "yyyy-MM-dd HH:mm:ss";
    /**
     * 最大结束时间
     */
    public static final String MAX_END_TIME = "9999-12-31" + DateUtil.DAY_LATEST_TIME;

    /**
     * 每天最早的时刻
     */
    public static final String DAY_EARLIEST_TIME = " 00:00:00";

    /**
     * 每天最晚的时刻
     */
    public static final String DAY_LATEST_TIME = " 23:59:59";

    // ////////////////////////////////////////////////////////////////////////////
    // date
    // 各种方式获取的时间
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 得到当前时间 yyyy-MM-dd
     */
    public static String dateStrNowYYYYMMDD() {
        return new SimpleDateFormat(DEFAULT_YEAR_MONTH_DAY).format(new Date());
    }

    /**
     * 得到时间格式为:yyyy-MM-dd的Str
     */
    public static String dateStrYYYYMMDD(Date date) {
        return new SimpleDateFormat(DEFAULT_YEAR_MONTH_DAY).format(date);
    }

    /**
     * 得到时间格式为:yyyy-MM-dd
     */
    public static Date dateYYYYMMDD(String date) {
        try {
            return new SimpleDateFormat(DEFAULT_YEAR_MONTH_DAY).parse(date);
        } catch (ParseException e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 得到当前时间 yyyy-MM-dd HH:mm:ss
     */
    public static String dateStrNowAll() {
        return dateToStr(new SimpleDateFormat(DEFAULT_DATE_ALL_PATTERN));
    }

    /**
     * 得到当前时间 HH:mm:ss
     */
    public static String dateStrNowHms() {
        return dateToStr(date_HH_mm_ss);
    }


    /**
     * 日期转换为字符串
     *
     * @param dateSdf 日期格式
     * @return 字符串
     */
    public static String dateToStr(SimpleDateFormat dateSdf) {
        return dateToStr(dateSdf, new Date());
    }

    /**
     * 日期转换为字符串
     *
     * @param dateSdf 日期格式
     * @return 字符串
     */
    public static String dateToStr(SimpleDateFormat dateSdf, Date date) {
        return dateSdf.format(date);
    }

    // 指定模式的时间格式
    public static SimpleDateFormat getSimpleDateFormat(String pattern) {
        return new SimpleDateFormat(pattern);
    }

    public static String getFormatDateStr(Date date, String pattern) {
        if (date == null) {
            return null;
        }
        DateFormat dtFmt = new SimpleDateFormat(pattern, Locale.US);
        return dtFmt.format(date);
    }

    /**
     * 当前日历，这里用中国时间表示
     *
     * @return 以当地时区表示的系统当前日历
     */
    public static Calendar getCalendar() {
        return Calendar.getInstance();
    }

    /**
     * 指定毫秒数表示的日历
     *
     * @param millis 毫秒数
     * @return 指定毫秒数表示的日历
     */
    public static Calendar getCalendar(long millis) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date(millis));
        return cal;
    }


    // ////////////////////////////////////////////////////////////////////////////
    // Timestamp
    // 各种方式获取的时间
    // ////////////////////////////////////////////////////////////////////////////


    /**
     * 时间戳转换为字符串
     *
     * @param time
     * @return
     */
    public static String timestampToStr(Timestamp time) {
        Date date = null;
        if (null != time) {
            date = new Date(time.getTime());
        }
        return dateToStr(date_yyyy_MM_dd, date);
    }

    /**
     * 字符串转换时间戳
     *
     * @param str
     * @return
     */
    public static Timestamp strToTimestamp(String str) {
        Date date = strToDate(str, date_yyyy_MM_dd);
        return new Timestamp(date.getTime());
    }

    /**
     * 字符串转换成日期
     *
     * @param dateStr
     * @param sdf
     * @return
     */
    public static Date strToDate(String dateStr, SimpleDateFormat sdf) {
        if (null == sdf ||
                null == dateStr ||
                "".equals(dateStr)) {
            return null;
        }
        Date date = null;
        try {
            date = sdf.parse(dateStr);
            return date;
        } catch (ParseException e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 计算两个时间的中间的月数
     *
     * @param date1
     * @param date2
     * @return
     */
    public static int getMonthSpace(String date1, String date2, String format) {
        int result = 0;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            Calendar c1 = Calendar.getInstance();
            Calendar c2 = Calendar.getInstance();
            c1.setTime(sdf.parse(date1));
            c2.setTime(sdf.parse(date2));
            result = (c1.get(Calendar.YEAR) - c2.get(Calendar.YEAR)) * 12 + c1.get(Calendar.MONTH)
                    - c2.get(Calendar.MONTH);
        } catch (Exception e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);

        }
        return result == 0 ? 1 : Math.abs(result);
    }


    /**
     * 格式化时间
     *
     * @param date
     * @param format
     * @return
     */
    public static String dateStrFormat(String date, String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        Date _date = null;
        try {
            _date = sdf.parse(date);
        } catch (ParseException e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);
        }
        return sdf.format(_date);
    }

    /**
     * 日期转换为字符串
     *
     * @param date     日期
     * @param date_sdf 日期格式
     * @return 字符串
     */
    public static String dateToStr(Date date, SimpleDateFormat date_sdf) {
        if (null == date) {
            return null;
        }
        return date_sdf.format(date);
    }

    /**
     * 日期转换为字符串
     *
     * @param format 日期格式
     * @return 字符串
     */
    public static String getDate(String format) {
        return new SimpleDateFormat(format).format(new Date());
    }

    /**
     * 日期转换为字符串
     *
     * @param format 日期格式
     * @return 字符串
     */
    public static Date getDate(SimpleDateFormat format) {
        try {
            String nowDate = format.format(new Date());
            return format.parse(nowDate);
        } catch (ParseException e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 指定毫秒数的时间戳
     *
     * @param millis 毫秒数
     * @return 指定毫秒数的时间戳
     */
    public static Timestamp getTimestamp(long millis) {
        return new Timestamp(millis);
    }

    /**
     * 以字符形式表示的时间戳
     *
     * @param time 毫秒数
     * @return 以字符形式表示的时间戳
     */
    public static Timestamp getTimestamp(String time) {
        if (StringUtils.isEmpty(time)) {
            return null;
        }
        return new Timestamp(Long.parseLong(time));
    }

    /**
     * 系统当前的时间戳
     *
     * @return 系统当前的时间戳
     */
    public static Timestamp getTimestamp() {
        return new Timestamp(new Date().getTime());
    }

    /**
     * 指定日期的时间戳
     *
     * @param date 指定日期
     * @return 指定日期的时间戳
     */
    public static Timestamp getTimestamp(Date date) {
        return new Timestamp(date.getTime());
    }

    /**
     * 指定日历的时间戳
     *
     * @param cal 指定日历
     * @return 指定日历的时间戳
     */
    public static Timestamp getCalendarTimestamp(Calendar cal) {
        return new Timestamp(cal.getTime().getTime());
    }


    // ////////////////////////////////////////////////////////////////////////////
    // formatDate
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：年-月-日
     *
     * @return 默认日期按“年-月-日“格式显示
     */
    public static String formatDate() {
        return new SimpleDateFormat("yyyy-MM-dd").format(getCalendar().getTime());
    }

    /**
     * 获取时间字符串
     */
    public static String getDataString(SimpleDateFormat formatstr) {
        return formatstr.format(getCalendar().getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日
     *
     * @param cal 指定的日期
     * @return 指定日期按“年-月-日“格式显示
     */
    public static String formatDate(Calendar cal) {
        return date_yyyy_MM_dd.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日
     *
     * @param date 指定的日期
     * @return 指定日期按“年-月-日“格式显示
     */
    public static String formatDate(Date date) {
        return date_yyyy_MM_dd.format(date);
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：年-月-日
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“年-月-日“格式显示
     */
    public static String formatDate(long millis) {
        return date_yyyy_MM_dd.format(new Date(millis));
    }

    /**
     * 默认日期按指定格式显示
     *
     * @param pattern 指定的格式
     * @return 默认日期按指定格式显示
     */
    public static String formatDate(String pattern) {
        return getSimpleDateFormat(pattern).format(getCalendar().getTime());
    }

    /**
     * 指定日期按指定格式显示
     *
     * @param cal     指定的日期
     * @param pattern 指定的格式
     * @return 指定日期按指定格式显示
     */
    public static String formatDate(Calendar cal, String pattern) {
        return getSimpleDateFormat(pattern).format(cal.getTime());
    }

    /**
     * 指定日期按指定格式显示
     *
     * @param date    指定的日期
     * @param pattern 指定的格式
     * @return 指定日期按指定格式显示
     */
    public static String formatDate(Date date, String pattern) {
        return getSimpleDateFormat(pattern).format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // formatTime
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：年-月-日 时：分
     *
     * @return 默认日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime() {
        return date_yyyy_MM_dd_HH_mm.format(getCalendar().getTime());
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(long millis) {
        return date_yyyy_MM_dd_HH_mm.format(new Date(millis));
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param cal 指定的日期
     * @return 指定日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(Calendar cal) {
        return date_yyyy_MM_dd_HH_mm.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：年-月-日 时：分
     *
     * @param date 指定的日期
     * @return 指定日期按“年-月-日 时：分“格式显示
     */
    public static String formatTime(Date date) {
        return date_yyyy_MM_dd_HH_mm.format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // formatShortTime
    // 将日期按照一定的格式转化为字符串
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 默认方式表示的系统当前日期，具体格式：时：分
     *
     * @return 默认日期按“时：分“格式显示
     */
    public static String formatShortTime() {
        return new SimpleDateFormat(DEFAULT_TIME_ALL_PATTERN).format(getCalendar().getTime());
    }

    /**
     * 指定毫秒数表示日期的默认显示，具体格式：时：分
     *
     * @param millis 指定的毫秒数
     * @return 指定毫秒数表示日期按“时：分“格式显示
     */
    public static String formatShortTime(long millis) {
        return date_HH_mm_ss.format(new Date(millis));
    }

    /**
     * 指定日期的默认显示，具体格式：时：分
     *
     * @param cal 指定的日期
     * @return 指定日期按“时：分“格式显示
     */
    public static String formatShortTime(Calendar cal) {
        return date_HH_mm_ss.format(cal.getTime());
    }

    /**
     * 指定日期的默认显示，具体格式：时：分
     *
     * @param date 指定的日期
     * @return 指定日期按“时：分“格式显示
     */
    public static String formatShortTime(Date date) {
        return date_HH_mm_ss.format(date);
    }

    // ////////////////////////////////////////////////////////////////////////////
    // parseDate
    // parseCalendar
    // parseTimestamp
    // 将字符串按照一定的格式转化为日期或时间
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的日期
     * @
     * @
     */
    public static Date parseDate(String src, String pattern) {
        try {
            return getSimpleDateFormat(pattern).parse(src);
        } catch (Exception e) {
            log.error("日期[{}],格式[{}]", src, pattern);
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的日期
     * @
     * @
     */
    public static Calendar parseCalendar(String src, String pattern) {

        Date date = parseDate(src, pattern);
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal;
    }

    public static String formatAddDate(String src, String pattern, int amount) {
        Calendar cal;
        cal = parseCalendar(src, pattern);
        cal.add(Calendar.DATE, amount);
        return formatDate(cal);
    }

    public static String formatCurrAddDate(int amount) {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.DATE, amount);
        return formatDate(cal);
    }

    public static String formatAddMonth(String src, String pattern, int month) {
        Calendar cal;
        cal = parseCalendar(src, pattern);
        cal.add(Calendar.MONTH, month);
        return formatDate(cal);
    }

    public static String formatAddMonthYearMonth(String src, int month) {
        Calendar cal;
        cal = parseCalendar(src, DEFAULT_YEAR_MONTH_DAY);
        cal.add(Calendar.MONTH, month);
        return date_yyyy_MM.format(cal.getTime());
    }

    public static Date formatAddMonth(Date src, int month) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(src);
        calendar.add(Calendar.MONTH, month);
        return calendar.getTime();
    }

    public static Date formatAddYear(Date src, int year) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(src);
        calendar.add(Calendar.YEAR, year);
        return calendar.getTime();
    }

    /**
     * 根据指定的格式将字符串转换成Date 如输入：2003-11-19 11:20:20将按照这个转成时间
     *
     * @param src     将要转换的原始字符窜
     * @param pattern 转换的匹配格式
     * @return 如果转换成功则返回转换后的时间戳
     * @
     * @
     */
    public static Timestamp parseTimestamp(String src, String pattern) {
        Date date = parseDate(src, pattern);
        return new Timestamp(date.getTime());
    }

    // ////////////////////////////////////////////////////////////////////////////
    // dateDiff
    // 计算两个日期之间的差值
    // ////////////////////////////////////////////////////////////////////////////

    /**
     * 计算两个时间之间的差值，根据标志的不同而不同
     *
     * @param flag   计算标志，表示按照年/月/日/时/分/秒等计算
     * @param calSrc 减数
     * @param calDes 被减数
     * @return 两个日期之间的差值
     */
    public static int dateDiff(char flag, Calendar calSrc, Calendar calDes) {

        long millisDiff = calSrc.getTimeInMillis() - calDes.getTimeInMillis();

        if (flag == 'y') {
            return (calSrc.get(Calendar.YEAR) - calDes.get(Calendar.YEAR));
        }

        if (flag == 'd') {
            return (int) (millisDiff / DAY_IN_MILLIS);
        }

        if (flag == 'h') {
            return (int) (millisDiff / HOUR_IN_MILLIS);
        }

        if (flag == 'm') {
            return (int) (millisDiff / MINUTE_IN_MILLIS);
        }

        if (flag == 's') {
            return (int) (millisDiff / SECOND_IN_MILLIS);
        }

        return 0;
    }

    /**
     * String类型 转换为Date, 如果参数长度为10 转换格式”yyyy-MM-dd“ 如果参数长度为19 转换格式”yyyy-MM-dd HH:mm:ss“ * @param text String类型的时间值
     */
    @Override
    public void setAsText(String text) throws IllegalArgumentException {
        if (text != null
                && !"".equals(text)) {
            try {
                if (!text.contains(":") && text.length() == 10) {
                    setValue(DateUtil.date_yyyy_MM_dd.parse(text));
                } else if (text.indexOf(":") > 0 && text.length() == 19) {
                    setValue(DateUtil.date_yyyy_MM_dd_HH_mm_ss.parse(text));
                } else {
                    throw new IllegalArgumentException("Could not parse date, date format is error ");
                }
            } catch (ParseException ex) {
                throw new IllegalArgumentException("Could not parse date: " + ex.getMessage(), ex);
            }
        } else {
            setValue(null);
        }
    }

    /**
     * 获取当前年份
     *
     * @return
     */
    public static int getCurrentYear() {
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(new Date());
        return calendar.get(Calendar.YEAR);
    }

    /**
     * 当前月份 1-12
     *
     * @return
     */
    public static int getCurrentMonth() {
        return getPostCurrentMonth() + 1;
    }


    /**
     * 当前月份  0-11
     *
     * @return
     */
    public static int getPostCurrentMonth() {
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.setTime(new Date());
        return calendar.get(Calendar.MONTH);
    }

    /**
     * 两个时间间隔多少天
     */
    public static long subtractDay(Date endDate, Date startDate) {
        return (endDate.getTime() - startDate.getTime()) / (24 * 60 * 60 * 1000);
    }

    /**
     * 返回Date参数的 前后N天日期 取决于第二个参数
     * 返回date
     *
     * @param date
     * @param i
     * @return
     */
    public static Date ReturnTheDay(Date date, int i) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DAY_OF_MONTH, i);
        return calendar.getTime();
    }

    public static String format(Date date, String pattern) {
        if (date != null) {
            try {
                return new SimpleDateFormat(pattern).format(date);
            } catch (Exception e) {
                log.error("日期[{}],格式[{}]", date, pattern);
                log.error(e.getMessage(), e);
            }
        }
        return "";
    }

    public static String format(Object value, String pattern) {
        if (value != null) {
            try {
                Date date = null;
//				if(value instanceof oracle.sql.TIMESTAMP) {
//					date = new Date(((oracle.sql.TIMESTAMP)value).dateValue().getTime());
//				} else {
//					date = (Date)value;
//				}
                return new SimpleDateFormat(pattern).format(date);
            } catch (Exception e) {
                log.error("日期[{}],格式[{}]", value, pattern);
                log.error(e.getMessage(), e);
            }
        }
        return "";
    }

    public static Date getDateTime(String dateTime) {
        Date date = new Date();
        String datePart = DateUtil.format(date, "yyyy-MM-dd");
        String dateTimePart = datePart + " " + dateTime;
        Date newDate = DateUtil.parse(dateTimePart, "yyyy-MM-dd HH:mm");
        return newDate;
    }

    public static Date parse(String date, String pattern) {
        Date newDate = null;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setLenient(false);
            newDate = sdf.parse(date);
        } catch (Exception e) {
            log.error("日期[{}],格式[{}]", date, pattern);
            log.error(e.getMessage(), e);
            throw new RuntimeException("日期转换错误：" + date + "，转换为：" + pattern);
        }
        return newDate;
    }

    /**
     * @描述: 得到date的昨天
     */
    public static Date getYesterdayByDate(String date, String pattern) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(pattern);
            sdf.setLenient(false);
            Date temp = sdf.parse(date);
            Calendar cal = Calendar.getInstance();
            cal.setTime(temp);
            cal.add(Calendar.DAY_OF_MONTH, -1);
            return cal.getTime();
        } catch (Exception e) {
            log.error("日期[{}],格式[{}]", date, pattern);
            log.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * @描述: 得到某一月的天数
     */
    public static int getDaysOfMonth(int year, int month) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, 1);
        return cal.getActualMaximum(Calendar.DAY_OF_MONTH);
    }

    /**
     * @描述: 得到某一月的天数
     */
    public static int getDaysOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.getActualMaximum(Calendar.DAY_OF_MONTH);
    }

    /**
     * @描述: 得到某一季的天数
     */
    public static int getDaysOfQuarter(int year, int quarter) {
        Calendar cal = Calendar.getInstance();
        int days = 0;
        int firstMonth = (quarter - 1) * 3;
        for (int i = firstMonth; i < firstMonth + 3; i++) {
            cal.set(year, i, 1);
            days += cal.getActualMaximum(Calendar.DAY_OF_MONTH);
        }
        return days;
    }

    /**
     * @描述: 得到某一年的天数
     */
    public static int getDaysOfYear(int year) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, 0, 1);
        return cal.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    /**
     * @描述: 得到某一年的天数
     */
    public static int getDaysOfYear(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.getActualMaximum(Calendar.DAY_OF_YEAR);
    }

    /**
     * @描述: 得到某一周的星期一的时间
     */
    public static Date getFirstDayOfWeek(int year, int week) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRST_WEEK);
        cal.set(year, Calendar.JANUARY, 1);
        int addDays = week * 7;
        if (cal.get(Calendar.WEEK_OF_YEAR) == 1) {
            addDays = (week - 1) * 7;
        }
        cal.add(Calendar.DATE, addDays);
        cal.set(Calendar.DAY_OF_WEEK, cal.getFirstDayOfWeek());
        return cal.getTime();

    }

    /**
     * @描述: 得到某一周的第一天
     */
    public static Date getFirstDayOfWeek(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRST_WEEK);
        cal.setTime(date);
        int dayOfWeek = cal.get(Calendar.DAY_OF_WEEK);
        if (dayOfWeek == 1) {
            dayOfWeek = 8;
        }

        cal.setTimeInMillis((cal.getTimeInMillis() + (FIRST_DAY_OF_WEEK - dayOfWeek) * 86400000L));
        return cal.getTime();
    }

    /**
     * @描述: 得到某一月的第一天
     */
    public static Date getFirstDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * @描述: 得到某一月的最后一天
     */
    public static Date getLastDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的第一天
     */
    public static Date getFirstDayOfLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的最后一天
     */
    public static Date getLastDayOfLastMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到上月的最后一天
     */
    public static Date getLastDayOfLastMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.setTimeInMillis(cal.getTimeInMillis() - cal.get(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的第一天
     */
    public static Date getFirstDayOfNextMonth() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的第一天
     */
    public static Date getFirstDayOfNextMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        return cal.getTime();
    }

    /**
     * @描述: 得到下月的最后一天
     */
    public static Date getLastDayOfNextMonth() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(cal.getTimeInMillis() + cal.getActualMaximum(Calendar.DAY_OF_MONTH) * 86400000L);
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.setTimeInMillis(cal.getTimeInMillis() + (cal.getActualMaximum(Calendar.DAY_OF_MONTH) - 1) * 86400000L);
        return cal.getTime();
    }


    /**
     * @描述: 得到某一日期的周数
     */
    public static int getWeekOfDate(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.setFirstDayOfWeek(FIRST_DAY_OF_WEEK);
        cal.setMinimalDaysInFirstWeek(MINIMAL_DAYS_IN_FIRST_WEEK);
        cal.set(year, month - 1, day, 0, 0, 0);
        return cal.get(Calendar.WEEK_OF_YEAR);

    }

    /**
     * @描述: 得到某一日期的天数
     */
    public static int getDayOfDate(int year, int month, int day) {
        Calendar cal = Calendar.getInstance();
        cal.set(year, month - 1, day, 0, 0, 0);
        return cal.get(Calendar.DAY_OF_YEAR);
    }

    public static Date getDate2() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date getDate(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date yesterday() {
        return new Date(System.currentTimeMillis() - 86400000L);
    }

    public static Date tomorrow() {
        return new Date(System.currentTimeMillis() + 86400000L);
    }

    public static Date getEndDate() {
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.YEAR, 2099);
        cal.set(Calendar.MONTH, 12 - 1);
        cal.set(Calendar.DATE, 31);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal.getTime();
    }

    public static Date parseToDate(Object value) {
        if (value instanceof Timestamp) {
            return (Date) value;
        }
        if (value instanceof java.sql.Date) {
            return (Date) value;
        }
        return (Date) value;

    }

    /**
     * 获取指定日期时间
     *
     * @param calendarConstant 指定时间类型
     * @param appointNum       指定时间
     * @return
     */
    public static Date getAppointDate(int calendarConstant, int appointNum) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        calendar.add(calendarConstant, appointNum);
        return calendar.getTime();
    }

    public static String calSpendTimes(String startTime, String endTime) {
        Date startTimeDateType = DateUtil.parse(startTime, "yyyy-MM-dd HH:mm:ss");

        Date endTimeDateType = null;
        if (endTime != null) {
            endTimeDateType = DateUtil.parse(endTime, "yyyy-MM-dd HH:mm:ss");
        }

        long totalTimes = (endTime == null) ?
                (Calendar.getInstance().getTimeInMillis() - startTimeDateType.getTime())
                :
                (endTimeDateType.getTime() - startTimeDateType.getTime());

        long dayCount = totalTimes / (1000 * 60 * 60 * 24);//计算天
        long restTimes = totalTimes % (1000 * 60 * 60 * 24);//剩下的时间用于计于小时
        long hourCount = restTimes / (1000 * 60 * 60);//小时
        restTimes = restTimes % (1000 * 60 * 60);
        long minuteCount = restTimes / (1000 * 60);

        String spendTimes = dayCount + "天" + hourCount + "小时" + minuteCount + "分钟";

        return spendTimes;
    }

    /**
     * 拆分两个日期获取月份
     *
     * @param minDate
     * @param maxDate
     * @return
     * @
     */
    public static List<String> getMonthBetween(String minDate, String maxDate) {
        ArrayList<String> result = new ArrayList<String>();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");// 格式化为年月

        Calendar min = Calendar.getInstance();
        Calendar max = Calendar.getInstance();

        try {
            min.setTime(sdf.parse(minDate));
            min.set(min.get(Calendar.YEAR), min.get(Calendar.MONTH), 1);

            max.setTime(sdf.parse(maxDate));
            max.set(max.get(Calendar.YEAR), max.get(Calendar.MONTH), 2);

            Calendar curr = min;
            while (curr.before(max)) {
                result.add(sdf.format(curr.getTime()));
                curr.add(Calendar.MONTH, 1);
            }
        } catch (ParseException e) {
            log.error("日期[{}],格式[{}]", minDate, maxDate);
            log.error(e.getMessage(), e);
        }
        return result;
    }

    /**
     * 拆分两个日期获取年对应的月份map
     *
     * @param minDate
     * @param maxDate
     * @return
     * @
     */
    public static Map<Integer, List<Integer>> getDateBetweenMap(String minDate, String maxDate) {
        Map<Integer, List<Integer>> map = Maps.newHashMap();
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM");// 格式化为年月

        Calendar min = Calendar.getInstance();
        Calendar max = Calendar.getInstance();

        try {
            min.setTime(sdf.parse(minDate));
            min.set(min.get(Calendar.YEAR), min.get(Calendar.MONTH), 1);

            max.setTime(sdf.parse(maxDate));
            max.set(max.get(Calendar.YEAR), max.get(Calendar.MONTH), 2);

            Calendar curr = min;
            while (curr.before(max)) {
                int year = curr.get(Calendar.YEAR);
                List<Integer> months = map.get(year);
                if (CollectionUtil.isEmpty(months)) {
                    months = Lists.newArrayList();
                }
                int month = curr.get(Calendar.MONTH);
                months.add(month + 1);
                map.put(year, months);
                curr.add(Calendar.MONTH, 1);
            }
        } catch (ParseException e) {
            log.error("日期[{}],格式[{}]", minDate, maxDate);
            log.error(e.getMessage(), e);
        }
        return map;
    }

    static int[] DAYS = {0, 31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

    /**
     * @param date yyyy-MM-dd HH:mm:ss
     * @return
     */
    public static boolean isValidDateTime(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0) {
                return false;
            }
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12) {
                return false;
            }
            int day = Integer.parseInt(date.substring(8, 10));
            if (day <= 0 || day > DAYS[month]) {
                return false;
            }
            if (month == 2 && day == 29 && isGregorianLeapYear(year)) {
                return false;
            }
            int hour = Integer.parseInt(date.substring(11, 13));
            if (hour < 0 || hour > 23) {
                return false;
            }
            int minute = Integer.parseInt(date.substring(14, 16));
            if (minute < 0 || minute > 59) {
                return false;
            }
            int second = Integer.parseInt(date.substring(17, 19));
            if (second < 0 || second > 59) {
                return false;
            }

        } catch (Exception e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * @param date yyyy-MM-dd
     * @return
     */
    public static boolean isValidMoth(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0) {
                return false;
            }
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12) {
                return false;
            }
        } catch (Exception e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * @param date yyyyMMdd
     * @return
     */
    public static boolean isValidDateNotLine(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0) {
                return false;
            }
            int month = Integer.parseInt(date.substring(4, 6));
            if (month <= 0 || month > 12) {
                return false;
            }
            int day = Integer.parseInt(date.substring(6, 8));
            if (day <= 0 || day > DAYS[month]) {
                return false;
            }
            if (month == 2 && day == 29 && isGregorianLeapYear(year)) {
                return false;
            }
        } catch (Exception e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * @param date yyyy-MM-dd
     * @return
     */
    public static boolean isValidDate(String date) {
        try {
            int year = Integer.parseInt(date.substring(0, 4));
            if (year <= 0) {
                return false;
            }
            int month = Integer.parseInt(date.substring(5, 7));
            if (month <= 0 || month > 12) {
                return false;
            }
            int day = Integer.parseInt(date.substring(8, 10));
            if (day <= 0 || day > DAYS[month]) {
                return false;
            }
            if (month == 2 && day == 29 && isGregorianLeapYear(year)) {
                return false;
            }
        } catch (Exception e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    public static boolean isGregorianLeapYear(int year) {
        return year % 4 != 0 || (year % 100 == 0 && year % 400 != 0);
    }

    /**
     * 时间转换器 月末与月初
     *
     * @param fromDate
     * @param type     0表示当前月第一天 1当前月最后一天
     */
    public static Date switchDate(Date fromDate, int type) {
        if (fromDate != null) {
            if (type == 0) {
                GregorianCalendar gcLast = (GregorianCalendar) Calendar
                        .getInstance();
                gcLast.setTime(fromDate);
                gcLast.set(Calendar.HOUR_OF_DAY, 0);
                gcLast.set(Calendar.MINUTE, 0);
                gcLast.set(Calendar.SECOND, 0);
                // 设置为第一天
                gcLast.set(Calendar.DAY_OF_MONTH, 1);
                return gcLast.getTime();
            } else {
                // 获取Calendar
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(fromDate);
                calendar.set(Calendar.HOUR_OF_DAY, 23);
                calendar.set(Calendar.MINUTE, 59);
                calendar.set(Calendar.SECOND, 59);
                // 设置日期为本月最大日期
                calendar.set(Calendar.DATE,
                        calendar.getActualMaximum(Calendar.DATE));
                // 设置日期格式
                return calendar.getTime();
            }
        }
        return null;
    }

    /**
     * 获取指定月的前一月（年）或后一月（年）
     *
     * @param dateStr
     * @param addYear
     * @param addMonth 月份为 实际月份减一
     * @param addDate
     * @param ymd      1月 2 年 3月最后一天 4年最后一天
     * @return
     * @throws Exception 方法的简述.
     * @throws Exception
     */
    public static String getAnyDay(String dateStr, int addYear, int addMonth,
                                   int addDate, int ymd) throws Exception {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yyyy");
            Date sourceDate = sdf.parse(dateStr);
            SimpleDateFormat sdf_ = new SimpleDateFormat(
                    "yyyy-MM");
            Date sourceDate_ = sdf_.parse(dateStr);
            Calendar cal = Calendar.getInstance();
            cal.clear();
            cal.setTime(sourceDate);
            cal.add(Calendar.YEAR, addYear);
            cal.add(Calendar.MONTH, sourceDate_.getMonth() + addMonth);
            if (ymd == 1) {// 月
                cal.add(Calendar.DAY_OF_MONTH, 0);
            } else if (ymd == 2) {
                cal.set(Calendar.DAY_OF_YEAR, 1);
            } else if (ymd == 3) {
                cal.set(Calendar.DAY_OF_MONTH,
                        cal.getActualMaximum(Calendar.DAY_OF_MONTH));
                System.out.println(cal.getTime().getMonth());
            } else {
                cal.set(Calendar.YEAR, cal.getTime().getYear() + 1900);
                cal.set(Calendar.DAY_OF_YEAR,
                        cal.getActualMaximum(Calendar.DAY_OF_YEAR));
            }
            cal.add(Calendar.DATE, addDate);

            SimpleDateFormat returnSdf = new SimpleDateFormat(
                    "yyyy-MM-dd");

            String dateTmp = returnSdf.format(cal.getTime());
            return dateTmp;
        } catch (Exception e) {
            log.error("日期[{}]", dateStr);
            log.error(e.getMessage(), e);
            throw new Exception(e.getMessage());
        }
    }

    /**
     * 获取指定月的第一天
     * <p>指定月的第一天<br>
     * 方法的详细说明第二行
     *
     * @param date String格式yyyy-MM
     * @return
     * @author Warren Wang
     */
    public static String getFirstDayOfMonth(String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");

        Calendar calendar = Calendar.getInstance();// 获取当前日期
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            throw new IllegalArgumentException("参数错误");
        }
        calendar.set(Calendar.DATE, calendar.getMinimum(Calendar.DATE));
        SimpleDateFormat format2 = new SimpleDateFormat("yyyy-MM-dd");
        return format2.format(calendar.getTime());
    }

    /**
     * 获取指定月的最后一天
     * <p>指定月的最后一天<br>
     *
     * @param date String格式yyyy-MM
     * @return
     * @author Warren Wang
     */
    public static String getFinalDayOfMonth(String date) {
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM");

        Calendar calendar = Calendar.getInstance();
        try {
            calendar.setTime(format.parse(date));
        } catch (ParseException e) {
            log.error("日期[{}]", date);
            log.error(e.getMessage(), e);
            throw new IllegalArgumentException("参数错误");
        }
        calendar.set(Calendar.DATE, calendar.getActualMaximum(Calendar.DATE));
        SimpleDateFormat format2 = new SimpleDateFormat("yyyy-MM-dd");
        return format2.format(calendar.getTime());
    }

    public static boolean isDate(String aDate) {
        return isDate(aDate, DEFAULT_YEAR_MONTH_DAY);
    }

    public static boolean isDate(String aDate, String fmt) {
        if (aDate == null || "".equals(aDate.trim())) {
            return false;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(fmt);
        sdf.setLenient(false);
        try {
            sdf.parse(aDate);
        } catch (ParseException e) {
            log.error("日期[{}],格式[{}]", aDate, fmt);
            log.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    /**
     * 获取指定月的前一月（年）或后一月（年）
     *
     * @param dateStr
     * @param addYear
     * @param addMonth 月份为 实际月份减一
     * @param addDay
     * @param pattern  例如yyyy-MM-dd
     * @return
     * @throws Exception 方法的简述.
     * @throws Exception
     */
    public static String getAnyDay(String dateStr, String pattern, int addYear, int addMonth,
                                   int addDay) {
        try {
            Date date = DateUtil.parseDate(dateStr, pattern);
            Calendar cal = Calendar.getInstance();
            cal.setTime(date);
            if (addYear != 0) {
                cal.add(Calendar.YEAR, addYear);
            }
            if (addMonth != 0) {
                cal.add(Calendar.MONTH, addMonth);
            }
            if (addDay != 0) {
                cal.add(Calendar.DATE, addDay);
            }
            return DateUtil.formatDate(cal);
        } catch (Exception e) {
            log.error("日期处理异常", e);
            log.error(e.getMessage(), e);
            throw new RuntimeException(e.getMessage());
        }
    }

    /**
     * 过滤特殊字符
     */
    public static String filterChar(String date) {
        Pattern p = Pattern.compile("\\s*|\t|\r|\n");
        Matcher m = p.matcher(date);
        return m.replaceAll("");
    }

    public static int getDayOfMonth(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        return cal.get(Calendar.DATE);
    }

    public static String convertDateStr(long millSec) {
        DateFormat fmtDateTime = new SimpleDateFormat("yyyy-MM-dd");
        return fmtDateTime.format(new Date(millSec * 1000));
    }

    public static Date extractionDate(long millSec) {
        Date in = new Date(millSec * 1000);
        DateFormat fmtDateTime = new SimpleDateFormat("yyyy-MM-dd");
        String text = fmtDateTime.format(in);
        ParsePosition pos = new ParsePosition(0);
        return fmtDateTime.parse(text, pos);
    }

    public static int[] monthToYear(int month) {
        int[] ret = new int[]{0, 0};
        ret[0] = month / 12;
        if (month % 12 != 0) {
            ret[1] = month % 12;
        }
        return ret;
    }

    public static String monthToYearStr(int month) {
        int[] ret = monthToYear(month);
        String str = ret[0] + "岁";
        if (ret[0] <= 5) {
            if (ret[1] != 0) {
                str += ret[1] + "个月";
            }
        }
        return str;
    }

    /**
     * ** 根据传入的时间和格式获得字符串 *****
     */
    public static String getDateStrByFormat(Date date, String format) {
        if (date == null) {
            return "";
        }
        DateFormat dtFmt = new SimpleDateFormat(format);
        return dtFmt.format(date);
    }

    public static boolean isSameDate(Date date1, Date date2) {
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);

        Calendar cal2 = Calendar.getInstance();
        cal2.setTime(date2);

        boolean isSameYear = cal1.get(Calendar.YEAR) == cal2
                .get(Calendar.YEAR);
        boolean isSameMonth = isSameYear
                && cal1.get(Calendar.MONTH) == cal2.get(Calendar.MONTH);
        boolean isSameDate = isSameMonth
                && cal1.get(Calendar.DAY_OF_MONTH) == cal2
                .get(Calendar.DAY_OF_MONTH);

        return isSameDate;
    }

    public static boolean isSameDay(Date date1, Date date2) {
        long between = Math.abs(date2.getTime() - date1.getTime());

        if (between > DAY_IN_MILLIS) {
            return false;
        } else if (between == DAY_IN_MILLIS) {
            Calendar cal1 = Calendar.getInstance();
            cal1.setTime(date1);

            Calendar cal2 = Calendar.getInstance();
            cal2.setTime(date2);

            if (cal1.get(Calendar.HOUR_OF_DAY) == 0 && cal2.get(Calendar.HOUR_OF_DAY) == 0) {
                return true;
            }
            if (cal1.get(Calendar.HOUR_OF_DAY) == 24) {
                cal2.get(Calendar.HOUR_OF_DAY);
            }
            return false;
        }
        return true;
    }

    /**
     * ** 根据传入的时间字符串和格式获得Date对象 *****
     */
    public static Date getDateByFormat(String date, String format) {
        if (date == null) {
            return null;
        }
        DateFormat dtFmt = new SimpleDateFormat(format);
        Date dateTime = null;
        try {
            dateTime = dtFmt.parse(date);
        } catch (ParseException e) {
            log.error("日期处理异常");
            log.error(e.getMessage(), e);
        }
        return dateTime;
    }

    /**
     * *** 传入时间戳(秒),计算2个时间差多少天(自然天，不是24小时，如今天凌晨0点1分和昨天23点59分相差为1天) *******
     */
    public static long getDistanceDay(long date1, long date2) {
        String dateStr1 = getDateStrByFormat(new Date(date1 * 1000), "yyyy-MM-dd");
        String dateStr2 = getDateStrByFormat(new Date(date2 * 1000), "yyyy-MM-dd");

        Date date3 = getDateByFormat(dateStr1, "yyyy-MM-dd");
        Date date4 = getDateByFormat(dateStr2, "yyyy-MM-dd");

        long newDate1 = convertTimeStamp(date3);
        long newDate2 = convertTimeStamp(date4);
        long diff;
        if (newDate1 >= newDate2) {
            diff = newDate1 - newDate2;
        } else {
            diff = newDate2 - newDate1;
        }
        return diff / (60 * 60 * 24);
    }

    /**
     * 传入时间返回时间戳(秒) **
     */
    public static long convertTimeStamp(Date date) {
        if (date == null) {
            return 0;
        }
        return date.getTime() / 1000;
    }

    /**
     * *** 传入时间戳,计算2个时间差多少秒 *******
     */
    public static int getDistanceSecond(long date1, long date2) {
        long diff = date1 - date2;
        double diffDouble = Math.ceil(1.0 * diff / 1000);
        return (int) diffDouble;
    }


    /**
     * *** 根据生日算年龄 *******
     */
    public static Integer getAge(Date birthday) {
        if (birthday == null) {
            return null;
        }
        Calendar calendar = Calendar.getInstance();
        int yearNow = calendar.get(Calendar.YEAR);
        calendar.setTime(birthday);
        int yearBirth = calendar.get(Calendar.YEAR);
        if (yearNow - yearBirth < 0) {
            return null;
        }
        return yearNow - yearBirth;
    }

    /**
     * *** 传人两个时间求之间天数 *******
     */
    public static int getDayCount(Date startDate, Date endDate) {
        if (!startDate.before(endDate)) {
            return 0;
        }
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(startDate);

        Calendar calendar1 = Calendar.getInstance();
        calendar1.setTime(endDate);

        return (int) ((calendar1.getTimeInMillis() - calendar.getTimeInMillis()) / 86400000L);
    }

    /**
     * *** 计算2个时间差多少月
     */
    public static int getDistanceMonth(Date start, Date end) {
        if (start.after(end)) {
            Date t = start;
            start = end;
            end = t;
        }
        Calendar startCalendar = Calendar.getInstance();
        startCalendar.setTime(start);
        Calendar endCalendar = Calendar.getInstance();
        endCalendar.setTime(end);
        Calendar temp = Calendar.getInstance();
        temp.setTime(end);
        temp.add(Calendar.DATE, 1);

        int year = endCalendar.get(Calendar.YEAR)
                - startCalendar.get(Calendar.YEAR);
        int month = endCalendar.get(Calendar.MONTH)
                - startCalendar.get(Calendar.MONTH);

        if ((startCalendar.get(Calendar.DATE) == 1)
                && (temp.get(Calendar.DATE) == 1)) {
            return year * 12 + month + 1;
        } else if ((startCalendar.get(Calendar.DATE) != 1)
                && (temp.get(Calendar.DATE) == 1)) {
            return year * 12 + month;
        } else if ((startCalendar.get(Calendar.DATE) == 1)
                && (temp.get(Calendar.DATE) != 1)) {
            return year * 12 + month;
        } else {
            return (year * 12 + month - 1) < 0 ? 0 : (year * 12 + month);
        }
    }

    /**
     * 获得周一到周七的文本 **
     */
    public static String getWeekDayStr(Integer weekDay) {
        String str = "周";
        switch (weekDay) {
            case 1:
                str += "一";
                break;
            case 2:
                str += "二";
                break;
            case 3:
                str += "三";
                break;
            case 4:
                str += "四";
                break;
            case 5:
                str += "五";
                break;
            case 6:
                str += "六";
                break;
            case 7:
                str += "日";
                break;
        }
        return str;
    }

    /**
     * 获得周上午下午晚上的文本 **
     */
    public static String getDayTypeStr(Integer dayType) {
        String str = "";
        switch (dayType) {
            case 1:
                str += "上午";
                break;
            case 2:
                str += "下午";
                break;
            case 3:
                str += "晚上";
                break;
        }
        return str;
    }

    /**
     * *** 根据传入时间判断是否为当天 *******
     */
    public static boolean isToday(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(new Date());
        int startDayCount = (int) calendar.getTimeInMillis() / 86400000;
        calendar.setTime(date);
        int endDayCount = (int) calendar.getTimeInMillis() / 86400000;
        return endDayCount == startDayCount;
    }

    /**
     * *** 根据传入时间和相差天数计算结果Date *******
     */
    public static Date getDayByDateDistance(Date date, int distanceDay) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.DATE, distanceDay); // 日期减1
        Date lastDay = calendar.getTime(); // 结果
        return lastDay;
    }

    /**
     * *** 根据传入时间和相差秒数计算结果Date *******
     */
    public static Date getTimeByTimeAndDiffSecond(Date date, int second) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        calendar.add(Calendar.SECOND, second);
        return calendar.getTime();
    }


    public static boolean isOverlap(Date dt1Start, Date dt1End, Date dt2Start, Date dt2End) {
        if (dt1Start == null
                || dt1End == null
                || dt2Start == null
                || dt2End == null) {
            throw new IllegalArgumentException("时间不能为空!");
        }
        long dt1StartLong = dt1Start.getTime();
        long dt1EndLong = dt1End.getTime();
        long dt2StartLong = dt2Start.getTime();
        long dt2EndLong = dt2End.getTime();

        if (dt1StartLong >= dt1EndLong) {
            throw new IllegalArgumentException("param error,param1 must before param2");
        }
        if (dt2StartLong >= dt2EndLong) {
            throw new IllegalArgumentException("param error,param3 must before param4");
        }
        if (dt1EndLong <= dt2StartLong) {
            return false;
        } else {
            return dt1StartLong < dt2EndLong;
        }
    }

    /**
     * 判断时间是否在两个时间中间
     *
     * @param start
     * @param end
     * @param paramDate
     * @return
     */
    public static boolean isDateBetween(Date start, Date end, Date paramDate) {
        if (start == null || end == null || paramDate == null) {
            throw new IllegalArgumentException("时间不能为空!");
        }
        long startLong = start.getTime();
        long endLong = end.getTime();
        long paramLong = paramDate.getTime();
        return startLong <= paramLong
                && paramLong <= endLong;

    }

    /**
     * 获取当天的开始时间
     *
     * @param date
     * @return
     */
    public static Date getTodayStartTime(Date date) {
        if (date == null) {
            date = new Date();
        }

        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        return c.getTime();
    }

    /**
     * 获取当天的结束时间
     *
     * @param date
     * @return
     */
    public static Date getTodayEndTime(Date date) {
        if (date == null) {
            date = new Date();
        }

        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.set(Calendar.HOUR_OF_DAY, 23);
        c.set(Calendar.MINUTE, 59);
        c.set(Calendar.SECOND, 59);
        return c.getTime();
    }

    /**
     * 日期格式字符串转换成时间戳
     *
     * @param dateStr
     * @param format
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.30 19:22
     */
    public static long dateToTimeStamp(String dateStr, String format) {
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(format);
            return sdf.parse(dateStr).getTime() / 1000;
        } catch (Exception e) {
            log.error("日期[{}],格式[{}]", dateStr, format);
            log.error(e.getMessage(), e);
        }
        return 0;
    }

    /**
     * 根据月份获取对应的季度
     *
     * @param month
     * @return
     */
    public static int getQuarterByMonth(int month) {
        int quarter = 4;
        if (month >= 1 && month <= 3) {
            quarter = 1;
        } else if (month >= 4 && month <= 6) {
            quarter = 2;
        } else if (month >= 7 && month <= 9) {
            quarter = 3;
        }
        return quarter;
    }

    //将java8 的 java.time.LocalDateTime 转换为 java.util.Date，默认时区为东8区
    public static Date localDateTimeConvertToDate(LocalDateTime localDateTime) {
        return Date.from(localDateTime.toInstant(ZoneOffset.of("+8")));
    }

    //将java.util.Date 转换为java8 的java.time.LocalDateTime,默认时区为东8区
    public static LocalDateTime dateConvertToLocalDateTime(Date date) {
        return date.toInstant().atOffset(ZoneOffset.of("+8")).toLocalDateTime();
    }

    /**
     * 得到日期为星期几
     *
     * @param date
     * @return
     */
    public static String getWeek(String date) {
        return getWeek(date, DEFAULT_YEAR_MONTH_DAY);
    }

    /**
     * 得到日期为星期几
     *
     * @param dateStr
     * @return
     */
    public static String getWeek(String dateStr, String format) {
        if (StringUtils.isEmpty(dateStr)
                || StringUtils.isEmpty(format)) {
            return "";
        }
        try {
            Calendar cal = Calendar.getInstance();
            Date date = new SimpleDateFormat(format).parse(dateStr);
            cal.setTime(date);
            int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
            if (w == 0) {
                w = 7;
            }
            return String.valueOf(w);
        } catch (Exception e) {
            log.error("dates[{}]", dateStr);
            log.error(e.getMessage(), e);
        }
        return "";
    }

    /**
     * 通过传入时间和数字  正整增加秒  负数减少秒
     *
     * @param date
     * @param second
     * @return
     */
    public static Date dateAddSecond(Date date, int second) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.SECOND, second);
        return cal.getTime();
    }

    /**
     * 通过传入时间和数字  正整增加分钟  负数减少钟
     *
     * @param date
     * @param num
     * @return
     */
    public static Date dateAddMinute(Date date, Integer num) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(Calendar.MINUTE, num);
        return c.getTime();
    }

    /**
     * 通过传入时间和数字  正整增加分钟  负数减少钟
     *
     * @param date
     * @param num
     * @return
     */
    public static Date dateAddHour(Date date, Integer num) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(Calendar.HOUR_OF_DAY, num);
        return c.getTime();
    }

    /**
     * 通过传入时间和数字  正整增加天  负数减少天
     *
     * @param date
     * @param num
     * @return
     */
    public static Date dateAddDay(Date date, Integer num) {
        Calendar c = Calendar.getInstance();
        c.setTime(date);
        c.add(Calendar.DAY_OF_MONTH, num);
        return c.getTime();
    }

    /**
     * 通过传入时间和数字  正整增加月  负数减少月
     *
     * @param date 日期
     * @return 2017-05
     * @
     */
    public static Date dateAddMonth(Date date, Integer month) {
        Calendar rightNow = Calendar.getInstance();
        rightNow.setTime(date);
        rightNow.add(Calendar.MONTH, month);
        return rightNow.getTime();
    }

    /**
     * @param dateStr   2020-11
     * @param beginDate 2020-11-01
     * @param endDate   2020-12-31
     * @return 判断时间是否在范围内
     */
    public static Boolean judgeDateStrBetween(String dateStr, String beginDate, String endDate) {
        String substring = dateStr.substring(0, 7);
        String substring1 = beginDate.substring(0, 7);
        String substring2 = endDate.substring(0, 7);
        return substring.compareTo(substring1) >= 0 && substring.compareTo(substring2) <= 0;
    }


    /**
     * 秒转  XX天xx时XX分
     *
     * @param seconds 秒
     * @return
     */
    public static String secondsToStr(Long seconds) {
        if (seconds == null) {
            return "";
        }
        return millisecondToStr(seconds * 1000);
    }

    /**
     * 毫秒转  天小时分秒
     *
     * @param millisecond
     * @return
     */
    public static String millisecondToStr(long millisecond) {
        long days = millisecond / 86400000;
        long hours = (millisecond % 86400000) / 3600000;
        long minute = (millisecond % 86400000) % 3600000 / 60000;
        long second = (millisecond % 86400000) % 3600000 % 60000 / 1000;
        long millisecondStr = (millisecond % 86400000) % 3600000 % 60000 % 1000;
        String spendTimes = "";
        if (days > 0) {
            spendTimes = spendTimes + days + "天";
        }
        if (hours > 0) {
            spendTimes = spendTimes + hours + "小时";
        }
        if (minute > 0) {
            spendTimes = spendTimes + minute + "分钟";
        }
        if (second > 0) {
            spendTimes = spendTimes + second + "秒";
        }
        if (millisecondStr > 0) {
            spendTimes = spendTimes + millisecondStr + "毫秒";
        }
        return spendTimes;
    }

    /**
     * 秒转  XX天xx时XX分 模糊提示
     *
     * @param seconds 秒
     * @return
     */
    public static String secondsToStrVague(Long seconds) {
        if (seconds == null) {
            return "";
        }
        return millisecondToStrVague(seconds * 1000);
    }

    /**
     * 毫秒转  天小时分秒 模糊提示
     *
     * @param millisecond
     * @return
     */
    public static String millisecondToStrVague(long millisecond) {
        long days = millisecond / 86400000;
        long hours = (millisecond % 86400000) / 3600000;
        long minute = (millisecond % 86400000) % 3600000 / 60000;
        long second = (millisecond % 86400000) % 3600000 % 60000 / 1000;
        long millisecondStr = (millisecond % 86400000) % 3600000 % 60000 % 1000;
        String spendTimes = "";
        if (days > 0) {
            spendTimes = spendTimes + days + "天";
        }
        if (hours > 0) {
            spendTimes = spendTimes + hours + "小时";
        }
        if (minute > 0) {
            spendTimes = spendTimes + minute + "分钟";
        }
        if (StringUtils.isEmpty(spendTimes)
                && second > 0) {
            spendTimes = spendTimes + second + "秒";
        }
        if (StringUtils.isEmpty(spendTimes)
                && millisecondStr > 0) {
            spendTimes = spendTimes + millisecondStr + "毫秒";
        }
        return spendTimes;
    }

    /**
     * 获取两个日期之间的所有日期（包含开始结束）
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static List<String> getBetweenDays(Date startTime, Date endTime) {
        return getBetweenDays(startTime, endTime, DEFAULT_YEAR_MONTH_DAY);
    }

    /**
     * 获取两个日期之间的所有日期（包含开始结束）
     *
     * @param startTime
     * @param endTime
     * @return
     */
    public static List<String> getBetweenDays(Date startTime, Date endTime, String format) {

        // 返回的日期集合
        List<String> days = new LinkedList<>();

        DateFormat dateFormat = new SimpleDateFormat(format);
        Calendar tempStart = Calendar.getInstance();
        tempStart.setTime(startTime);

        Calendar tempEnd = Calendar.getInstance();
        tempEnd.setTime(endTime);
        tempEnd.add(Calendar.DATE, +1);// 日期加1(包含结束)
        while (tempStart.before(tempEnd)) {
            days.add(dateFormat.format(tempStart.getTime()));
            tempStart.add(Calendar.DAY_OF_YEAR, 1);
        }
        return days;
    }

    /**
     * 获取两个日期之间的所有年月（包含开始结束）
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return List<String>
     */
    public static List<String> getBetweenYearMonths(Date startTime, Date endTime, String format) {

        // 返回的日期集合
        List<String> days = new LinkedList<>();

        DateFormat dateFormat = new SimpleDateFormat(format);
        Calendar tempStart = Calendar.getInstance();
        tempStart.setTime(startTime);

        Calendar tempEnd = Calendar.getInstance();
        tempEnd.setTime(endTime);
//        tempEnd.add(Calendar.MONTH, +1);// 日期加1(包含结束)
        while (tempStart.before(tempEnd)) {
            days.add(dateFormat.format(tempStart.getTime()));
            tempStart.add(Calendar.MONTH, 1);
            tempStart.set(Calendar.DAY_OF_MONTH, 1);
        }
        return days;
    }

    /***
     * 入参只支持 【yyyyMMdd】 【yyyy-MM-dd】 【yyyy/MM/dd】 【yyyy/MM/dd HH:mm:ss】  转换成你需要的格式format
     * @param date,format
     * @return java.util.Date
     **/
    public static String transToDateString(String date, String format) {
        //如果日期格式为yyyyMMdd
        if (date.length() == 8) {
            return DateUtil.getFormatDateStr(DateUtil.getDateByFormat(date, DEFAULT_YEAR_MONTH_DAY_NO_CH), format);
        } else if (date.length() == 10) {
            //如果日期格式为yyyy-MM-dd
            if (date.contains("-")) {
                return DateUtil.getFormatDateStr(DateUtil.getDateByFormat(date, DEFAULT_YEAR_MONTH_DAY), format);
            } //如果日期格式为yyyy/MM/dd
            else if (date.contains("/")) {
                return DateUtil.getFormatDateStr(DateUtil.getDateByFormat(date, "yyyy/MM/dd"), format);
            }
        }
        return null;
    }

    public static Date localDate2Date(LocalDate localDate) {
        ZonedDateTime zonedDateTime = localDate.atStartOfDay(ZoneId.systemDefault());
        Instant instant1 = zonedDateTime.toInstant();
        Date from = Date.from(instant1);
        return from;
    }

    public static void main(String[] args) {
        System.out.println(date_yyyy_MM_dd_HH_mm_ss.format(dateAddSecond(new Date(), 120)));
        System.out.println(date_yyyy_MM_dd_HH_mm_ss.format(dateAddMinute(new Date(), 5)));
        System.out.println(date_yyyy_MM_dd_HH_mm_ss.format(dateAddHour(new Date(), 1)));
        System.out.println(date_yyyy_MM_dd_HH_mm_ss.format(dateAddDay(new Date(), 1)));
        System.out.println(date_yyyy_MM_dd_HH_mm_ss.format(dateAddMonth(new Date(), 1)));
    }

    /**
     * 拆分日期
     *
     * @param begin
     * @param end
     * @return
     */
    public static List<RuleSplitVo> splitForTable02(String begin, String end) {
        List<RuleSplitVo> list = new ArrayList<>();
        dealSplit(list, begin, end); //递归拆分处理日期
        return list;
    }

    private static void dealSplit(List<RuleSplitVo> list, String begin, String end) {
        begin = dealDate00(begin);
        end = dealDate00(end);
        RuleSplitVo temp = new RuleSplitVo();
        String begin_end = getMaxMonthDate(begin, 0, 2);//获取开始时间月的最后一天
        temp.setBegin(begin);
        if (begin_end.compareTo(end) >= 0) {//如果 最后一天大于等于 结束时间 那么说明  时间段在本月
            temp.setEnd(end);
            list.add(temp);
        } else {  //本次结束时间为  当月最后一天
            temp.setEnd(begin_end);
            list.add(temp);
            //获取下月第一天  并 递归重新计算 开始结束
            String nextFirst = getMaxMonthDate(begin, 1, 1);//获取开始时间 下月第一天
            dealSplit(list, nextFirst, end);
        }
    }


    //处理日期错误 为00开着的数据
    private static String dealDate00(String datestr) {
        if (StringUtils.isNotBlank(datestr)) {
            if (datestr.substring(0, 2).equals("00")) {
                datestr = "20" + datestr.substring(2);
            }
        }
        return datestr;
    }

    /**
     * 获取当前月份最后一天
     *
     * @param dateStr 当前日期 yyyyMMdd
     * @param month   加减月数
     * @param type    1月初  2  月末
     * @return
     */
    public static String getMaxMonthDate(String dateStr, int month, int type) {
        dateStr = dealDate00(dateStr);
        SimpleDateFormat dft = null;
        Calendar calendar = null;
        dft = new SimpleDateFormat(DEFAULT_YEAR_MONTH_DAY_NO_CH);

        Date date = parseDate(dateStr, DEFAULT_YEAR_MONTH_DAY_NO_CH);
        calendar = Calendar.getInstance();
        calendar.setTime(date);
        // calendar.add(Calendar.MONTH, -1);
        calendar.add(Calendar.MONTH, month);
        if (type == 1) { //第一天
            calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMinimum(Calendar.DAY_OF_MONTH));
        } else { //最后一天
            calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
        }

        return dft.format(calendar.getTime());
    }
}