package com.biz.crm.business.common.base.util;

import com.biz.crm.business.common.sdk.model.Result;
import jodd.bean.BeanTool;
import jodd.bean.BeanUtil;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.util.Assert;

import javax.persistence.Column;
import javax.persistence.JoinColumn;
import javax.persistence.Table;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.util.*;

/**
 * 提供基本的Bean转换与映射功能。
 *
 */
public class OwnBeanUtils {

	private static final String ONE_STR = "1";

	/**
	 * 复制所有与目标对象的属性相匹配的源对象中的属性值到目标对象中。（源对象中的null值也同样会被复制）
	 *
	 * @param srcBean
	 * @param dstBean
	 */
	public static void copy(Object srcBean, Object dstBean) {
		copy(srcBean, dstBean, null, (Collection<String>) null);
	}

	/**
	 * 考虑被忽略属性列表与属性名称转换表,将复制所有与目标对象的属性相匹配的源对象中的属性值到目标对象中。 （源对象中的null值也同样会被复制）.
	 *
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 *            由目标属性名称到源属性名称的映射表
	 * @param ignoredProps
	 *            被忽略的属性集合
	 */
	public static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps) {
		copy(srcBean, dstBean, revertMap, ignoredProps, null, true);
	}

	/**
	 * 考虑被忽略属性列表与属性名称转换表,将复制所有与目标对象的属性相匹配的源对象中的属性值到目标对象中。(包含null值）
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 */
	public static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, String[] ignoredProps) {
		copy(srcBean, dstBean, revertMap, ignoredProps, null, true);
	}

	/**
	 * 考虑属性名称转换表,将复制所有与目标对象的属性相匹配的源对象中的属性值到目标对象中。（包含null值）
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 */
	public static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap) {
		copy(srcBean, dstBean, revertMap, (Collection<String>) null, null, true);
	}

	/**
	 * 考虑属性名称转换表,将复制所有与指定的目标对象的属性相匹配的源对象中的属性值到目标对象中。（包含null值）
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 * @param propsToCopy
	 */
	public static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps,
			Collection<String> propsToCopy) {

		copy(srcBean, dstBean, revertMap, ignoredProps, propsToCopy, true);
	}

	/**
	 * 考虑属性名称转换表,将复制所有与指定的目标对象的属性相匹配的源对象中的属性值到目标对象中。（包含null值）
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 */
	public static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, String[] ignoredProps,
			String[] props) {
		copy(srcBean, dstBean, revertMap, ignoredProps, props, true);
	}

	/**
	 * 应用所有与目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 * 
	 * @param srcBean
	 * @param dstBean
	 */
	public static void apply(Object srcBean, Object dstBean) {
		copy(srcBean, dstBean, null, (Collection<String>) null, null, false);
	}

	/**
	 * 应用所有与目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param copyEmptyStr
	 *            是否复制空字符串，只能处理"" 不能处理"  "
	 */
	public static void apply(Object srcBean, Object dstBean, boolean copyEmptyStr) {
		copy(srcBean, dstBean, null, (Collection<String>) null, null, false, copyEmptyStr);
	}

	/**
	 * 考虑被忽略属性列表与属性名称,应用所有与目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 *
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 */
	public static void apply(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps) {
		copy(srcBean, dstBean, revertMap, ignoredProps, null, false);
	}

	/**
	 * 考虑被忽略属性列表与属性名称映射表,应用所有与目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 */
	public static void apply(Object srcBean, Object dstBean, Map<String, String> revertMap, String[] ignoredProps) {
		copy(srcBean, dstBean, revertMap, ignoredProps, null, false);
	}

	/**
	 * 考虑被忽略属性列表与属性名称,应用所有与限定的目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 *
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 * @param propsToApply
	 */
	public static void apply(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps,
			Collection<String> propsToApply) {
		copy(srcBean, dstBean, revertMap, ignoredProps, propsToApply, false);
	}

	/**
	 * 考虑被忽略属性列表与属性名称映射表,应用所有与限定的目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 * @param ignoredProps
	 * @param propsToApply
	 */
	public static void apply(Object srcBean, Object dstBean, Map<String, String> revertMap, String[] ignoredProps,
			String[] propsToApply) {
		copy(srcBean, dstBean, revertMap, ignoredProps, propsToApply, false);
	}

	/**
	 * 考虑与属性名称映射表,应用所有与目标对象的属性相匹配的源对象中的非null属性值到目标对象中。
	 * 
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 */
	public static void apply(Object srcBean, Object dstBean, Map<String, String> revertMap) {
		copy(srcBean, dstBean, revertMap, (Collection<String>) null, null, false);
	}

	/**
	 * 克隆一份源Bean（所有属性均浅拷贝）到新建的Bean对象中。
	 * 
	 * @param srcBean
	 * @return
	 */
	public static Object clone(Object srcBean) {
		return clone(srcBean, srcBean.getClass(), null);
	}

	/**
	 * 考虑属性名称转换表将源对象克隆到一个指定类型的新建目标对象中并返回新创建的Bean
	 * 
	 * @param srcBean
	 * @param dstBeanClass
	 * @param revertMap
	 * @return
	 */
	public static <T> T clone(Object srcBean, Class<T> dstBeanClass, Map<String, String> revertMap) {
		T dstBean = null;
		try {
			dstBean = dstBeanClass.newInstance();
		} catch (Throwable t) {
			throw new RuntimeException("Exception Raised During Clone bean", t);
		}
		copy(srcBean, dstBean, null, (Collection<String>) null, null, false);
		return dstBean;
	}

	/**
	 * 考虑被忽略属性列表与属性名称,应用所有与目标对象的属性相匹配的源对象中值到目标对象中。
	 */
	private static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, String[] ignoredProps, String[] props,
			boolean copyNull) {
		Set<String> propsToCopy = null;
		if (props != null && props.length > 0) {
			propsToCopy = new HashSet<String>();
			for (String prop : props) {
				propsToCopy.add(prop);
			}
		}
		Set<String> propsToIgnore = null;
		if (ignoredProps != null && ignoredProps.length > 0) {
			propsToIgnore = new HashSet<String>();
			for (String prop : ignoredProps) {
				propsToIgnore.add(prop);
			}
		}
		copy(srcBean, dstBean, revertMap, propsToIgnore, propsToCopy, copyNull);
	}

	/**
	 * 提交基本的Bean操作能力。
	 * 
	 * 所有的Bean操作的规范如下： 按目标Bean中的所有属性(如果指定了待复制属性列表，则仅限于待复制属性列表)，
	 * 如果该属性不在被忽略属性中，且有源Bean中的对应属性， 则以源Bean中的对应属性设置目标Bean中的属性。 源与目标间的属性对应方式为：
	 * 如果存在映射表， 以则目标对象中的属性为key从映射表中查到的字符串为源对象中的属性名称。 如果不存在映射表，或映射表中不包含指定的属性，
	 * 则源对象中的属性名称与目标对象的属性名称相同。
	 *
	 * @param srcBean
	 * @param dstBean
	 * @param revertMap
	 *            属性转换表， 为空则不转换。 属性转换表中不存在的项目也不做转换。
	 * @param ignoredProps
	 *            被忽略的属性列表
	 * @param props
	 *            希望被复制的属性
	 * @param copyNull
	 *            是否复制空值。
	 */
	private static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps,
			Collection<String> props, boolean copyNull) {
		copy(srcBean, dstBean, revertMap, ignoredProps, props, copyNull, true);
	}

	private static void copy(Object srcBean, Object dstBean, Map<String, String> revertMap, Collection<String> ignoredProps,
			Collection<String> props, boolean copyNull, boolean copyEmptyStr) {
		String[] properties = BeanTool.resolveProperties(dstBean, true);
		String srcPropName;
		Object value;

		for (String property : properties) {
			// 过滤掉不在指定属性列表中的属性
			if (props != null && !props.contains(property)) {
				continue;
			}
			// 过滤掉在忽略属性列表中的属性
			if (ignoredProps != null && ignoredProps.contains(property)) {
				continue;
			}
			if (revertMap != null) {
				srcPropName = revertMap.get(property);
				if (srcPropName == null) {
					srcPropName = property;
				}
			} else {
				srcPropName = property;
			}
			// 使用Silently版本， 这样将允许不存在的目标字段存在而继续工作，但这样也将掩盖一些问题。
			value = BeanUtil.silent.getProperty(srcBean, srcPropName);
			if (value == null && !copyNull) {
				continue;
			}
			// 只处理"" 不处理 " "
			if (value != null && value.toString().isEmpty() && !copyEmptyStr) {
				continue;
			}
			BeanUtil.silent.setProperty(dstBean, property, value);
		}
	}


	/**
	 * 比较编辑前和编辑后的两个的内容是否相同
	 * 
	 * @return
	 */
	public static <T> boolean toCompare(T oldObj, T newObj) {
		ReflectHelper reflectHelper_old = new ReflectHelper(oldObj);
		ReflectHelper reflectHelper_new = new ReflectHelper(newObj);

		try {
			Field[] fields = oldObj.getClass().getDeclaredFields();
			for (Field field : fields) {
				String fieldName = field.getName();
				Object oldO = reflectHelper_old.getMethodValue(fieldName);
				Object newO = reflectHelper_new.getMethodValue(fieldName);
				if (oldO instanceof String) {
					if (!oldO.equals(newO)) {
						return false;
					}
				} else if (oldO instanceof Integer) {
					Integer oldI = (Integer) oldO;
					Integer newI = (Integer) newO;
					if (oldI != newI) {
						return false;
					}
				} else if (oldO instanceof Double) {
					Double oldB = (Double) oldO;
					Double newB = (Double) newO;
					if (oldB != newB) {
						return false;
					}
				} else if (oldO instanceof BigDecimal) {
					BigDecimal oldB = (BigDecimal) oldO;
					BigDecimal newB = (BigDecimal) newO;
					if (oldB.compareTo(newB) != 0) {
						return false;
					}
				} else if (oldO instanceof Date) {
					Date oldD = (Date) oldO;
					Date newD = (Date) newO;
					if (oldD.getTime() != newD.getTime()) {
						return false;
					}
				}
			}
			return true;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return false;
	}

	/**
	 * 检查实体内的值是否为空
	 * 
	 * @param obj
	 *            该实体的对象
	 * @return true 该实体为空或实体参数的值为空，false 该实体中至少有一个以上的参数不为空
	 */
	public static <T> boolean checkEntityValuesIsNull(T obj) {
		boolean flag = true;
		if (obj != null) {
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			Field[] fields = obj.getClass().getDeclaredFields();
			for (Field field : fields) {
				String fieldName = field.getName();
				Object object = reflectHelper.getMethodValue(fieldName);
				if (StringUtil.isNotEmpty(object)) {
					flag = false;
					break;
				}
			}
		}
		return flag;
	}

	/**
	 * 打包返回ajax的数据并装到Result的obj属性中
	 * 
	 * @param j
	 *            Result对象
	 * @param objs
	 *            list对象
	 * @param fields
	 *            要打包的属性
	 */
	public static <T> void encapsulateDataAjax(Result j, List<T> objs, String... fields) {
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		for (T obj : objs) {
			Map<String, Object> map = new HashMap<String, Object>();
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			for (String field : fields) {
				Object object = reflectHelper.getMethodValue(field);
				map.put(field, object);
			}
			list.add(map);
		}
		j.setResult(list);
	}
	/**
	 * 打包返回List<map>的数据
	 * @param objs
	 * @param fields
	 */
	public static <T> List<Map<String, Object>> encapsulateDataAjax(List<T> objs, String... fields) {
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		for (T obj : objs) {
			Map<String, Object> map = new HashMap<String, Object>();
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			for (String field : fields) {
				Object object = reflectHelper.getMethodValue(field);
				map.put(field, object);
			}
			list.add(map);
		}
		return list;
	}
	/**
	 * 打包返回map的数据--并连接传入fields参数对应的实体中的属性的值作为key，
	 * @param objs
	 * @param redundant  -- 冗余所有
	 * @param valueField 不传则默认1,传入则为对应实体的属性的值
	 * @param fields
	 * @return
	 */
	public static <T> Map<String, Object> encapsulateDataAndLinkFieldsValue(List<T> objs,boolean redundant,String valueField, String... fields) {
		Map<String, Object> map = new HashMap<String, Object>();
		for (T obj : objs) {
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			String keyStr = "";
			Object valueStr = ONE_STR;
			//如果不为空则赋值
			if (StringUtil.isNotEmpty(valueField)) {
				Object valueStrTemp = reflectHelper.getMethodValue(valueField);
				//找出不为空则赋值否则不赋值
				if (StringUtil.isNotEmpty(valueStrTemp)) {
					valueStr = valueStrTemp;
				}
			}
			int i = 0;
			for (String field : fields) {
				//map的Key值
				Object object = reflectHelper.getMethodValue(field);
				if (object instanceof Date) {
					throw new IllegalArgumentException("类型转换错误，不能转换将‘" + Date.class.getName() + "’类型转换为‘java.lang.String’类型");
				}else{
					String valueTmp = String.valueOf(object);
					if (redundant) {
						map.put(valueTmp, valueStr);
					}
					keyStr += valueTmp;
				}
				if (redundant) {
					i ++;
				}
			}
			if (i != 1 ) {
				map.put(keyStr, valueStr);
			}
		}
		return map;
	}
	/**
	 * 打包返回map的数据--并对传入fields参数对应的实体中的属性的值分别连接并将传入的filed值作为key
	 * @param objs
	 * @param isInStr 是否使用in方式
	 * @param fields
	 * @return 
	 */
	public static <T> Map<String, Object> encapsulateDataFieldsToKeyAndRespectivelyLinkFieldsValue(List<T> objs,boolean isInStr,String... fields) {
		Map<String, Object> map = new HashMap<String, Object>();
//		if (CollectionUtils.isNotEmpty(objs)) {
//			reflectHelper_obj = new ReflectHelper(objs.get(0));
		for (T obj : objs) {
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			String keyStr = "";
			int i = 0;
			thsTwo : for (String field : fields) {
				//map的Key值
				Object object = reflectHelper.getMethodValue(field);
//				reflectHelper.getMethodValue(obj,field)
				if (object instanceof Date) {
					throw new IllegalArgumentException("类型转换错误，不能转换将‘" + Date.class.getName() + "’类型转换为‘java.lang.String’类型");
				}else{
					String valueOldTemp = returnNotNullWordValue(String.valueOf(map.get(field)));
					String valueTmp = String.valueOf(object);
					if (StringUtil.isNotEmpty(valueOldTemp)) {
						if(valueOldTemp.contains(valueTmp)){
							continue thsTwo;
						}
						valueOldTemp += ",";
					}

					valueOldTemp += valueTmp;
					map.put(field, valueOldTemp);
				}
			}
		}
		if (isInStr) {
			//转换为sql中in函数使用的str
			for (String str : map.keySet()) {
				Object object = map.get(str);
				map.put(str, changeStringToSqlInString(String.valueOf(object)));
			}
		}
		return map;
	}
	/**
	 * 打包返回ajax的数据并装到Result的obj属性中以map形式key和value 两个
	 * 
	 * @param j
	 *            Result对象
	 * @param objs
	 *            list对象
	 * @param idName
	 *            要打包的属性
	 */
	public static <T> void encapsulateDataAjaxByMap(Result j, List<T> objs, String idName, String valueName) {
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		for (T obj : objs) {
			Map<String, Object> map = new HashMap<String, Object>();
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			Object object = reflectHelper.getMethodValue(idName);
			map.put("key", object);
			object = reflectHelper.getMethodValue(valueName);
			map.put("value", object);
			list.add(map);
		}
		j.setResult(list);
	}
	/**
	 * 打包返回List<map>的数据以map形式key和value 两个
	 * @param objs
	 * @param idName
	 * @param valueName
	 * @return
	 */
	public static <T> List<Map<String, Object>> encapsulateDataAjaxByMap(List<T> objs, String idName, String valueName) {
		List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
		if (CollectionUtils.isNotEmpty(objs)) {
			for (T obj : objs) {
				Map<String, Object> map = new HashMap<String, Object>();
				ReflectHelper reflectHelper = new ReflectHelper(obj);
				Object object = reflectHelper.getMethodValue(idName);
				map.put("key", object);
				object = reflectHelper.getMethodValue(valueName);
				map.put("value", object);
				list.add(map);
			}
		}
		return list;
	}
	/**
	 * 打包返回map的数据以map形式key和value 两个
	 * @param objs
	 * @param key (属性值)
	 * @param value (属性值)
	 * @例：user.id = 1,user.name = aa,放入本方法（user.id，user.name）(（key,value）),map.put(user.id,user.name);return map;
	 * @return
	 */
	public static <T>Map<String, Object> encapsulateDataByMapKeyToKayValueToValue(List<T> objs, String key, String value) {
		if (!StringUtil.isNotEmpty(key)) {
			throw new IllegalArgumentException("key is null ");
		}
		if (!StringUtil.isNotEmpty(value)) {
			throw new IllegalArgumentException("value is null ");
		}
		Map<String, Object> map = new HashMap<String, Object>();
		for (T obj : objs) {
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			Object keyObj = reflectHelper.getMethodValue(key);
			if (keyObj instanceof String) {
				Object valueObj = reflectHelper.getMethodValue(value);
				map.put(keyObj.toString(), valueObj);
			}else{
				throw new IllegalArgumentException("map的键必须为String");
			}
		}
		return map;
	}
	/**
	 * 打包返回map的数据以map形式key和value(T) 两个
	 * @param objs
	 * @param key  (属性值)
	 * @param <T>
	 * @return
	 */
	public static <T>Map<String, T> encapsulateDataByMapKeyToKayMapValueToEntity(List<T> objs, String key) {
		if (!StringUtil.isNotEmpty(key)) {
			throw new IllegalArgumentException("key is null ");
		}
		Map<String, T> map = new HashMap<String, T>();
		if (CollectionUtils.isNotEmpty(objs)) {
			for (T obj : objs) {
				ReflectHelper reflectHelper = new ReflectHelper(obj);
				Object keyObj = reflectHelper.getMethodValue(key);
				if (keyObj instanceof String) {
					map.put(keyObj.toString(), obj);
				}else{
					throw new IllegalArgumentException("map的键必须为String");
				}
			}
		}
		return map;
	}
	/**
	 * 打包返回map的数据以map形式key（可链接多个字段值）和value(T) 两个
	 * @param objs
	 * @param connect  连接字段
	 * @param keys	Map key
	 * @param <T>
	 * @return
	 */
	public static <T>Map<String, T> encapsulateDataByMapKeyToKayMapValueToEntity(List<T> objs,String connect, String... keys) {
		if (!StringUtil.isNotEmpty(keys)) {
			throw new IllegalArgumentException("key is null ");
		}
		Map<String, T> map = new HashMap<String, T>();
		if (CollectionUtils.isNotEmpty(objs)) {
			for (T obj : objs) {
				ReflectHelper reflectHelper = new ReflectHelper(obj);
				String keyTemp = "";
				for (String key : keys) {
					Object keyObj = reflectHelper.getMethodValue(key);
					if (keyObj instanceof String) {
						if (StringUtil.isNotEmpty(keyTemp)) {
							keyTemp += connect;
						}
						keyTemp += keyObj.toString();
					}else{
						throw new IllegalArgumentException("map的键必须为String");
					}
				}
				if (StringUtil.isNotEmpty(keyTemp)) {
					map.put(keyTemp, obj);
				}
			}
		}
		return map;
	}

	/**
	 * 检查金额字段--或数字字段
	 * @param obj  检查数值
	 * @param objChineseName 该数值对应的中文名称
	 * @param messages	错误提示信息集合
	 * @param expres  正则表达式 （
	 *                   不传则为: isAmount == true :‘([0])|([1-9]+[0-9]{0,}\\.{0,1}[0-9]{0,2})|([0]\\.[0-9]{0,1}[1-9]{1})|([0]\\.[1-9]{1}[0-9]{0,1})’）
	 *							  isAmount == false : ‘([0-9]+)’
	 * @param isAmount 是否为金额字段--标准金额字段有两位小数  true : 是金额 ，false : 不为金额字段
	 * @param resultMap 参数值返回用map集合需传入 不传则不返回 对应的key也可不要
	 * @param mapKey 返回参数值resultMap 的key
	 * @return
	 */
	public static boolean checkTheAmountIsOk(Object obj,String objChineseName,List<String> messages,String expres,boolean isAmount,Map<String,Object> resultMap,String mapKey) {
		boolean falg = true;
		BigDecimal tempBigDecimal = BigDecimal.ZERO;
		//检查正则表达式
		if (!StringUtil.isNotEmpty(expres)) {
			if(isAmount){
				expres = "([0])|([1-9]+[0-9]{0,}\\.{0,1}[0-9]{0,2})|([0]\\.[0-9]{0,1}[1-9]{1})|([0]\\.[1-9]{1}[0-9]{0,1})";
			}else{
				expres = "([0-9]+)";
			}
		}
		boolean isCouldZero = false;
		if ("0".matches(expres)) {// 正则表达式是否可为0
			isCouldZero = true;
		}

		if (StringUtil.isNotEmpty(obj)) {
			String srcValue = (String) obj;
			//独立出来不影响目标数据
			String srcValueTemp = srcValue;
			BigDecimal decimal = BigDecimal.ZERO;

			String[] srcValues = srcValueTemp.replace(".", ",").split(",");
			//有小数
			if (isAmount && srcValues.length > 1) {
				//小数值大于4位，例：123.12345
				if (srcValues[1].length() > 5) {
					String expresTemp = "(0)|([0-9]{1,}\\.{0,1}[0-9]+)";
					if (srcValue.matches(expresTemp)) {
						//强制转换为两位小数（四舍五入）
						decimal = new BigDecimal(srcValue).setScale(2, BigDecimal.ROUND_HALF_UP);
						srcValue = String.valueOf(decimal);
					}else{
						//错误信息
						falg = toFromErrorMessageUsedToTheAmountIsOk(messages,isCouldZero,objChineseName,isAmount);
					}
				}
			}
			if (srcValue.matches(expres)) {
				//如果前面已经执行了则这里就不用执行了
				if (decimal.compareTo(BigDecimal.ZERO) == 0) {
					decimal = new BigDecimal(srcValue).setScale(2, BigDecimal.ROUND_HALF_UP);
				}
				if(resultMap != null){
					resultMap.put(mapKey,decimal);
				}
			} else {
				//错误信息
				falg = toFromErrorMessageUsedToTheAmountIsOk(messages,isCouldZero,objChineseName,isAmount);
			}
		} else {
			messages.add(objChineseName + "不能为空，请检查!");
			falg = false;
		}
		return falg;
	}

	/**
	 * 错误提示信息组装
	 * @param messages
	 * @param isCouldZero
	 * @param objChineseName
	 * @param isAmount
	 * @return
	 */
	private static boolean toFromErrorMessageUsedToTheAmountIsOk(List<String> messages,boolean isCouldZero, String objChineseName,boolean isAmount) {
		String messageTemp = objChineseName + "必须是大于";
		if (isCouldZero) {
			messageTemp += "等于";
		}
		messageTemp += "0的整数";
		if(isAmount){
			messageTemp += "或两位小数!";
		}
		messages.add(messageTemp);
		return false;
	}

	/**
	 * 改变值
	 * 
	 * @param valueObj
	 * @param format
	 * @return
	 * @throws Exception
	 */
	private static String changeValueToString(Object valueObj, String format) {
		String tempStr = "";
		if (valueObj instanceof Date) {
			tempStr = DateUtil.format((Date) valueObj, format);
		} else {
			tempStr = String.valueOf(valueObj);
		}
		return tempStr;
	}

	/**
	 * 将list实体转换为字符串 '1','2','3'
	 * @param objs  list<T>实体
	 * @param propertie 要转换成字符串的实体中的属性
	 * @param otherValue 额外的属性值
	 * @param isNoTRepeat  是否去除重复 -- true : 去除 ；false ： 不去除，可重复
	 * @param <T>
	 * @return
	 */
	public static <T> String changeListEntityOnePropertyToString(List<T> objs, String propertie, String otherValue,boolean isNoTRepeat) {
		List<Object> list = new ArrayList<Object>();
		for (T obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			Object object = reflectHelper_obj.getMethodValue(propertie);
			if (StringUtil.isNotEmpty(object)) {
				list.add("'" + object + "'");
			}
		}
		if (StringUtil.isNotEmpty(otherValue)) {
			for (String otherValueTemp : otherValue.split(",")) {
				list.add("'" + otherValueTemp.trim() + "'");
			}
		}
		if(isNoTRepeat){
			if (CollectionUtils.isNotEmpty(list)){
				list = new ArrayList<Object>(new HashSet<Object>(list));
			}
		}
		return list.toString().replace("[", "").replace("]", "");
	}
	/**
	 * 将list选择的属性转换为字符串
	 * @param objs list<T>实体
	 * @param connect 连接符 （不能是‘,（逗号）’或‘ ’（空格））
	 * @param properties 要转换成字符串的实体中的属性s
	 * @return string 以逗号隔开
	 */
	public static <T> String changeListEntityMorePropertyToString(List<T> objs,String connect,String... properties) {
		//检查连接符是否为逗号
		boolean falg = false;
		if (",".equals(connect)) {
			falg = true;
		}else if(" ".equals(connect)){
			falg = true;
		}
		if (falg) {
			throw new IllegalArgumentException("连接符不能是‘" + connect + "’");
		}

		List<Object> list = new ArrayList<Object>();
		for (T obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			String valueTemp = "";
			for (String propertie : properties) {
				Object object = reflectHelper_obj.getMethodValue(propertie);
				if (StringUtil.isNotEmpty(object)) {
					if (StringUtil.isNotEmpty(valueTemp)) {
						valueTemp += connect;
					}
					valueTemp += "'" + object + "'" ;
				}
			}
			list.add(valueTemp);
		}
		return list.toString().replace("[", "").replace("]", "").replace(" ", "");
	}

	/**
	 * 将list选择的属性转换为字符串--分别
	 * @param objs list<T>实体
	 * @param properties 要转换成字符串的实体中的属性s（‘s’非多余）
	 * @param isNoTRepeat  是否去除重复 -- true : 去除 ；false ： 不去除，可重复
	 * @return Map<String,List<String>></String,List<String>>
	 */
	public static <T> Map<String,List<String>> changeListEntityMorePropertyPartToPartListString(List<T> objs,boolean isNoTRepeat,String... properties) {
		//返回的参数对应值集合
		Map<String,List<String>> map = new HashMap<String,List<String>>();
		//初始化集合参数
		List<List<String>> objectLists = new ArrayList<List<String>>();
		for(String proper : properties){
			objectLists.add(new ArrayList<String>());
		}
		for (T obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			String valueTemp = "";
			for (int i = 0; i < properties.length; i++) {
				String fild = properties[i];
				Object object = reflectHelper_obj.getMethodValue(fild);
				if (StringUtil.isNotEmpty(object)) {
					objectLists.get(i).add(String.valueOf(object));
				}
			}
		}
		for (int i = 0; i < properties.length; i++) {
			String fild = properties[i];
			List<String> list = objectLists.get(i);
			if(isNoTRepeat){
				if (CollectionUtils.isNotEmpty(list)){
					list = new ArrayList<String>(new HashSet<String>(list));
				}
			}
			map.put(fild,list);
		}
		return map;
	}

	/**
	 * 组装关键值key
	 * @param obj
	 * @param connect
	 * @param filds
	 * @return
	 */
	public static String connectStrAsKey(Object obj,String connect, String filds ) {
		//用StringBuffer来链接字段
		StringBuffer buffer = new StringBuffer();
		ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
		for (String fild : filds.split(",")){
			Object object = reflectHelper_obj.getMethodValue(fild);
			if (buffer.length() != 0) {
				buffer.append(connect);
			}
			buffer.append(object);
		}
		return buffer.toString();
	}



	/**
	 * 将list选择的属性转换为字符串
	 * @param objs list<T>实体
	 * @param connect 连接符
	 * @param properties 要转换成字符串的实体中的属性
	 * @param valueFilds 参数值
	 * @return string 以逗号隔开
	 */
	/*public static <T> Map<String, Object> changeListEntityMorePropertyToKeyAndValueToValueToMap(List<T> objs,String connect,boolean isCheckExPres ,String[] properties,String... valueFilds) {
		Map<String, Object> map = new HashMap<String, Object>();
		ReflectHelper reflectHelper_obj = null;
		//检查正则表达式
		String expres = "([0])|([1-9]+[0-9]{0,}\\.{0,1}[0-9]{0,2})|([0]\\.[0-9]{0,1}[1-9]{1})|([0]\\.[1-9]{1}[0-9]{0,1})";

		if (CollectionUtils.isNotEmpty(objs)) {
			reflectHelper_obj = new ReflectHelper(objs.get(0));
			for (T obj : objs) {
				String ksyTemp = "";
				for (String propertie : properties) {
					if (StringUtil.isNotEmpty(ksyTemp)) {
						ksyTemp += connect;
					}
					Object object = reflectHelper_obj.getMethodValue(obj,propertie);
					ksyTemp += object ;
				}
				String valueTemp = "";
				for (String valueFild : valueFilds) {
					if (StringUtil.isNotEmpty(valueTemp)) {
						valueTemp += connect;
					}
					Object object = reflectHelper_obj.getMethodValue(obj,valueFild);

					if (isCheckExPres) {
						if (String.valueOf(object).matches(expres)) {
							valueTemp += object ;
						}
					}else{
						valueTemp += object ;
					}
				}
				if (StringUtil.isNotEmpty(valueTemp)) {
					map.put(ksyTemp, valueTemp);
				}
			}
		}
		return map;
	}*/

	/**
	 * 将字符串集合转换为"obj1,obj2,obj3,obj4"  obj 可以是  1    '1'    .....
	 * @param list
	 * @param isToInStr 是否为inString
	 * @return
	 */
	public static String changeListStringToString(List<String> list,boolean isToInStr) {
		String resultStr = list.toString().replace("[", "").replace("]", "").replace(" ", "");
		if(isToInStr){
			resultStr = changeStringToSqlInString(resultStr);
		}
		return resultStr;
	}

	/**
	 * @将格式为"1,2,3,4" 的字符串转化为 "'1','2','3','4'"
	 * @param objs
	 */
	public static String changeStringToSqlInString(String objs) {
		if (StringUtil.isNotEmpty(objs)) {
			objs = objs.replace("'", "");
			objs = "'" + objs.replace(",", "','") + "'";
			objs = objs.replace(",''", "").replace("'',", "").replace("''", "");
		}
		return objs;
	}
	/**
	 * @转化为一般字符串"1,2,3,4"
	 * @param objs 数据集合
	 * @param propertie 要转换的属性
	 * @param otherValue 额外值
	 * @param isNoTRepeat 是否去除重复 -- true : 去除 ；false ： 不去除，可重复
	 * @return
	 */
	public static <T> String changeListEntityOnePropertyCommonString(List<T> objs, String propertie, String otherValue,boolean isNoTRepeat) {
		List<Object> list = new ArrayList<Object>();
		for (T obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			Object object = reflectHelper_obj.getMethodValue(propertie);
			if (StringUtil.isNotEmpty(object)) {
				list.add(object);
			}
		}
		if (StringUtil.isNotEmpty(otherValue)) {
			for (String otherValueTemp : otherValue.replace("'", " ").split(",")) {
				list.add(otherValueTemp.trim());
			}
		}
		if(isNoTRepeat){
			if (CollectionUtils.isNotEmpty(list)){
				list = new ArrayList<Object>(new HashSet<Object>(list));
			}
		}
		return list.toString().replace("[", "").replace("]", "").replace(" ", "");
	}
	/**
	 * 比较两个相同实体指定的参数的值
	 *
	 * @param sourceEntity
	 * @param targetEntity
	 * @return true:完全相同 false:至少有一个不同
	 */
	public static <T> boolean toCompareSameEntity(T sourceEntity, T targetEntity, String filds) {
		ReflectHelper reflectHelper_one = new ReflectHelper(sourceEntity);
		ReflectHelper reflectHelper_other = new ReflectHelper(targetEntity);
		try {
			Class<?> clazz = sourceEntity.getClass();
			for (String field : filds.split(",")) {
				Field fieldName = clazz.getDeclaredField(field);
				Object oneO = reflectHelper_one.getMethodValue(fieldName.getName());
				fieldName = clazz.getDeclaredField(field);
				Object otherO = reflectHelper_other.getMethodValue(fieldName.getName());
				if (oneO instanceof String) {
					if (!oneO.equals(otherO)) {
						return false;
					}
				} else if (oneO instanceof Integer) {
					Integer oldI = (Integer) oneO;
					Integer newI = (Integer) otherO;
					if (oldI != newI) {
						return false;
					}
				} else if (oneO instanceof Double) {
					Double oldB = (Double) oneO;
					Double newB = (Double) otherO;
					if (oldB != newB) {
						return false;
					}
				} else if (oneO instanceof BigDecimal) {
					BigDecimal oldB = (BigDecimal) oneO;
					BigDecimal newB = (BigDecimal) otherO;
					if (oldB.compareTo(newB) != 0) {
						return false;
					}
				} else if (oneO instanceof Date) {
					Date oldD = (Date) oneO;
					Date newD = (Date) otherO;
					if (oldD.getTime() != newD.getTime()) {
						return false;
					}
				}
			}
			return true;
		} catch (Exception e) {
			throw new IllegalArgumentException(e);
		}
	}

	/**
	 * 检查年是否合格(yyyy)例：1991
	 *
	 * @param year
	 * @param messages
	 * @param i
	 */
	public static boolean checkYearIsOk(String year, List<String> messages, int i) {
		boolean falg = true;
		if (StringUtil.isNotEmpty(year)) {
			String yearStr = "[1-9][0-9]{3}";
			if (!year.matches(yearStr)) {
				messages.add("第" + i + "行年只能是大于0的4位数的正整数!");
				falg = false;
			}
		} else {
			messages.add("第" + i + "行年不能为空!");
			falg = false;
		}
		return falg;
	}
	/**
	 * 检查年是否合格(yyyy)例：1991
	 *
	 * @param year
	 * @param messages
	 */
	public static boolean checkYearIsOkNoLineNum(String year, List<String> messages) {
		boolean falg = true;
		if (StringUtil.isNotEmpty(year)) {
			String yearStr = "[1-9][0-9]{3}";
			if (!year.matches(yearStr)) {
				messages.add("年（" + year + "）只能是大于0的4位数的正整数!");
				falg = false;
			}
		} else {
			messages.add("年不能为空!");
			falg = false;
		}
		return falg;
	}
	/**
	 * 检查月是否合格(1-12)
	 *
	 * @param month
	 * @param messages
	 * @param i
	 */
	public static boolean checkMonthIsOk(String month, List<String> messages, int i) {
		boolean falg = true;
		if (StringUtil.isNotEmpty(month)) {
			String monthstr = "([1-9])|([1][0-2])";
			if (!month.matches(monthstr)) {
				messages.add("第" + i + "行月只能是1-12的正整数!");
				falg = false;
			}
		} else {
			messages.add("第" + i + "行月不能为空!");
			falg = false;
		}
		return falg;
	}
	/**
	 * 检查月是否合格(1-12)
	 *
	 * @param month
	 * @param messages
	 */
	public static boolean checkMonthIsOkNoLineNum(String month, List<String> messages) {
		boolean falg = true;
		if (StringUtil.isNotEmpty(month)) {
			String monthstr = "([1-9])|([0][1-9])|([1][0-2])";
			if (!month.matches(monthstr)) {
				messages.add("月（" + month + "）只能是1-12或01-12的正整数或整数串!");
				falg = false;
			}
		} else {
			messages.add("行月不能为空!");
			falg = false;
		}
		return falg;
	}
	/**
	 * 检查实体中的Sting型的值是否为“null”及“undefined”,是将其替换为"",否则不管
	 *
	 * @param obj
	 */
	public static void changeNullValueToNull(Object obj) {
		if (obj != null) {
			ReflectHelper reflectHelper = new ReflectHelper(obj);
			Field[] fields = obj.getClass().getDeclaredFields();
			for (Field field : fields) {
				String fildName = field.getName();
				Object object = reflectHelper.getMethodValue(fildName);
				if (object instanceof String) {
					if (object != null && !object.equals("") && (object.equals("undefined") && object.equals("null"))) {
						reflectHelper.setMethodValue(fildName, null);
					}
				}
			}
		}
	}
	/**
	 * 置换null字符
	 * @param obj
	 * @return
	 */
	public static String returnNotNullWordValue(String obj) {
		if (obj != null && !obj.equals("") && !obj.equals("undefined") && obj.equals("null")) {
			return "";
		}else{
			return obj;
		}
	}
	/**
	 * 将java实体转为Map--或输入一部分属性，如果不输入则表示转换整个实体
	 * @param obj
	 * @param fieldNames
	 * @return
	 */
	public static Map<String, Object> entityToMap(Object obj,String... fieldNames) {
		Map<String, Object> map = new HashMap<String, Object>();
		ReflectHelper reflectHelper = new ReflectHelper(obj);
		Class<?> clazz = obj.getClass();
		//判定是否有指定字段
		if (fieldNames != null && fieldNames.length > 0 ) {
			for (String fieldName : fieldNames) {
				Object value = reflectHelper.getMethodValue(fieldName);
				map.put(fieldName, value);
			}
			return map;
		}
		//没有则获取全部字段
		Field[] fields = clazz.getDeclaredFields();
		for (Field field : fields) {
			String name = field.getName();
			Object value = reflectHelper.getMethodValue(name);
			map.put(name, value);
		}
		return map;
	}

	/**
	 * 按FiledName获得Field的类型.
	 */
	public static Class getPropertyType(Class type, String name) throws NoSuchFieldException {
		return getDeclaredField(type, name).getType();
	}

	/**
	 * 循环向上转型,获取对象的DeclaredField.
	 *
	 * @throws NoSuchFieldException
	 *             如果没有该Field时抛异常.
	 */
	public static Field getDeclaredField(Class clazz, String propertyName) throws NoSuchFieldException {
		Assert.notNull(clazz);
		Assert.hasText(propertyName);
		for (Class superClass = clazz; superClass != Object.class; superClass = superClass.getSuperclass()) {
			try {
				return superClass.getDeclaredField(propertyName);
			} catch (NoSuchFieldException e) {
				// Field不在当前类定�?,继续向上转型
			}
		}
		throw new NoSuchFieldException("No such field: " + clazz.getName() + '.' + propertyName);
	}

	/**
	 * 通过class获取
	 * @param clazzSrc
	 * @return
	 */
	public static String getTableNameByClassName(String clazzSrc) {
		if (!StringUtil.isNotEmpty(clazzSrc)) {
			throw new IllegalArgumentException("class名称不能为空");
		}
		try {
			Class<?> clazz = Class.forName(clazzSrc);
			Table table = clazz.getAnnotation(Table.class);
			if (table == null) {
				throw new IllegalArgumentException("未找到与该实体" + clazz.getName() + "对应的表(@tabel)");
			}
			return table.name();
		} catch (ClassNotFoundException e) {
			throw new IllegalArgumentException(e);
		}
	}
	/**
	 * 检查是否为空
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static Object checkIsNullAndTip(Object objt,Map<String, Object> fildsAndErrorInfomap) {
		Set<String> filds = fildsAndErrorInfomap.keySet();
		List<Object> objs = null ;
		if (objt instanceof List) {//能转吗？？？？？
			objs = (List<Object>)objt;
		}else{
			objs = new ArrayList<Object>();
			objs.add(objt);
		}
		for (Object obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			for (String fild : filds) {
				Object object = reflectHelper_obj.getMethodValue(fild);
				//检查是否为字符串
				if (object == null) {
					return fildsAndErrorInfomap.get(fild);
				}else{
					if (!StringUtil.isNotEmpty(object)) {
						return fildsAndErrorInfomap.get(fild);
					}
				}
			}
		}
		return null;
	}

	/**
	 * 比较同一个对象中的两个字段的值是否相同
	 * @param objt 实体
	 * @param fileMap 字段Map集合
	 * @param style 添加样式 样式： 例:"<span style="color:red">?</span>" 参数位置用?代替
	 */
	public static void compareTheTwoFildsInOneObject(Object objt,Map <String, String> fileMap, String style) {
		if(!StringUtil.isNotEmpty(style)){
			throw new IllegalArgumentException("无样式，无处理意义");//无样式，无处理意义
		}
		Set<String> filds = fileMap.keySet();
		List<Object> objs = null ;
		if (objt instanceof List) {//能转吗？？？？？
			objs = (List<Object>)objt;
		}else{
			objs = new ArrayList<Object>();
			objs.add(objt);
		}
		for (Object obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			for (String fild : filds) {
				Object oldObjValue = reflectHelper_obj.getMethodValue(fild);
				//检查是否为字符串
				if (!StringUtil.isNotEmpty(oldObjValue)) {
					continue;
				}
				String fildTemp = fileMap.get(fild);
				Object objValue = reflectHelper_obj.getMethodValue(fildTemp);
				if (!StringUtil.isNotEmpty(objValue)) {
					continue;
				}
				if(objValue instanceof String){
					String objValueStr = objValue.toString();
					if(!oldObjValue.toString().equals(objValueStr)){
						//样式替换
						objValueStr = style.replace("?",objValueStr);
						reflectHelper_obj.setMethodValue(fildTemp,objValueStr);
					}
				}else{
					throw new IllegalArgumentException( fildTemp + "的类型必须为:" + String.class.getName() );
				}
			}
		}
		return;
	}

	/**
	 * 按标志移除对象
	 * @param objs
	 * @param fild
	 * @param <T>
	 * @return
	 */
	public static <T>List<T> removeObjInObjListBySomeProperSign(List<T> objs,String fild){
		List<T> objsTemp = new ArrayList<T>();
		for (T obj : objs) {
			ReflectHelper reflectHelper_obj = new ReflectHelper(obj);
			Object object = reflectHelper_obj.getMethodValue(fild);
			//检查是否为字符串
			if (object != null) {
				if (object instanceof String) {
					if (StringUtil.isNotEmpty(object)) {
						objsTemp.add(obj);
					}
				}else{
					throw new IllegalArgumentException("只检查字符串类型");
				}
			}
		}
		return objsTemp;

	}

	/**
	 * 判断目标字符串中是否含有对应的字段值
	 * @param srcObj  源字符串
	 * @param targObj 目标字符串
	 * @return
	 */
	public static boolean isContainsNotInSysConfig(String srcObj,String targObj){
		if(!StringUtil.isNotEmpty(srcObj)){
			throw  new IllegalArgumentException("源字符串不能为空");
		}
		if(!StringUtil.isNotEmpty(targObj)){
			throw  new IllegalArgumentException("目标字符串不能为空");
		}
		for (String obj : targObj.split(",")) {
			if(srcObj.contains(obj)){
				return true;
			}
		}
		return false;
	}

	/**
	 * 有一个等于就ok
	 * @param srcObj
	 * @param targObj
	 * @return
	 */
	public static boolean isContainsNotInSysConfigUseTheEquals(String srcObj,String targObj) {
		if(!StringUtil.isNotEmpty(srcObj)){
			throw  new IllegalArgumentException("源字符串不能为空");
		}
		if(!StringUtil.isNotEmpty(targObj)){
			throw  new IllegalArgumentException("目标字符串不能为空");
		}
		//转换为map不用每次循环
		Map<String,String>  srcMap = changeStringArrayToMap(srcObj.split(","));
		for (String targ : targObj.split(",")) {
			//为避免包含----例：nowRoles=abc;realRose=abcd,abcde...;realRose.contains(nowRoles) -> 'true'--!!  >_<!
			String str = srcMap.get(targ);//读取数据
			if(ONE_STR.equals(str)){
				return true;
			}
			/*for (String src : srcObj.split(",")) {
				if (src.equals(targ)) {
					return true;
				}
			}*/
		}
		return false;
	}


	/**
	 * 目标全部被包含
	 * @param srcObj 源数据
	 * @param targObj 目标数据--被检查的数据
	 * @param correctObjValue --正确的返回数据（List）
	 * @param errorObjValue -- 错误返回数据（List）
	 * @return true : 全匹配 ，false ： 不是全匹配
	 */
	public static boolean isContainsNotInSysConfigAllEqualsUseTheEquals(String srcObj,String targObj,List<String> correctObjValue,List<String> errorObjValue) {
		if(!StringUtil.isNotEmpty(srcObj)){
			throw  new IllegalArgumentException("源字符串不能为空");
		}
		if(!StringUtil.isNotEmpty(targObj)){
			throw  new IllegalArgumentException("目标字符串不能为空");
		}

		Map<String,String>  srcMap = changeStringArrayToMap(srcObj.split(","));
		boolean falg = true;
		for (String targ : targObj.split(",")) {
			boolean falgTemp = false;
			String str = srcMap.get(targ);
			//为避免包含----例：nowRoles=abc;realRose=abcd,abcde...;realRose.contains(nowRoles) -> 'true'--!!  >_<!
			if(ONE_STR.equals(str)){
				//封装数据
				enStringListData(correctObjValue,targ);
				continue ;
			}else{
				//封装数据
				enStringListData(errorObjValue,targ);
			}
			if(!falgTemp)  // 目前来看只可能为false但加上聊胜于无
			falg = falgTemp;//强制break后将不能成功封装数据了
		}
		return falg;
	}

	/**
	 * 封装字符串list集合
	 */
	private static void enStringListData(List<String> list,String value){
		if(list != null) list.add(value);
	}

	/**
	 *
	 * @param strs
	 * @return
	 * 将数组转换为Map值恒定为“1”
	 */
	public static Map<String,String> changeStringArrayToMap(String[] strs){
		Map<String,String> map = new HashMap<String, String>();
		for (String string : strs) {
			map.put(string, ONE_STR);
		}
		return map;
	}

	/**
	 * 按正则表达式替换
	 * @param obj
	 * @return
	 */
	public static String changeNumerStrTonumber(String obj){
		return obj.replaceAll("'([\\d]+)'","$1");
	}

	/**
	 * 获取列表列名
	 * @param obj
	 * @param fild
	 * @return
	 */
	public static String getOneColmnName(Object obj,String fild) {
		if (obj == null) {
			throw new IllegalArgumentException("Object(对象)不能为空");
		}
		Class<?> clazz = obj.getClass();
		ReflectHelper reflectHelper = new ReflectHelper(obj);
		Method method = reflectHelper.returnGetMethod(fild);
		String clomnName = "";
		//找列
		Column column = method.getAnnotation(Column.class);
		if (column != null) {
			clomnName = column.name();
		}else{//没有-找关系列
			//找关系列
			JoinColumn joinColumn = method.getAnnotation(JoinColumn.class);
			if (joinColumn == null) {//都没有抛出异常
				throw new IllegalArgumentException("未找到与该实体" + clazz.getName() + ":" + method + "对应的表关系列");
			}
			clomnName = joinColumn.name();
		}
		if (!StringUtil.isNotEmpty(clomnName)) {//都没有抛出异常
			throw new IllegalArgumentException("未找到与该实体" + clazz.getName() + ":" + method + "对应的表列");
		}
		return clomnName;
	}

}
