package com.biz.crm.business.common.base.util;


import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.http.HttpUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.sdk.constant.CommonConstant;
import com.biz.crm.business.common.sdk.model.Result;
import jodd.net.MimeTypes;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.util.Map;
import java.util.Objects;

/**
 * @describe HTTP请求方法
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.29 14:40
 */
@Slf4j
public class BusinessHttpUtil {


    /**
     * http请求异常时错误编码
     */
    public final static int ERROR_CODE = 20001;

    /**
     * 发送post请求;contentType默认[application/json]
     *
     * @param url  请求地址
     * @param json 请求参数
     * @return
     */
    public static Result<String> post(String url, String json) {
        return post(url, json, null, MimeTypes.MIME_APPLICATION_JSON, 5, 120, null);
    }

    /**
     * 发送post请求;contentType默认[application/json]
     *
     * @param url     请求地址
     * @param json    请求参数
     * @param headMap head参数
     * @return
     */
    public static Result<String> post(String url, String json, Map<String, String> headMap) {
        return post(url, json, headMap, MimeTypes.MIME_APPLICATION_JSON, 5, 120, null);
    }

    /**
     * 发送post请求;contentType默认[application/json]
     *
     * @param url     请求地址
     * @param headMap head参数
     * @param formMap 请求参数
     * @return
     */
    public static Result<String> post(String url, Map<String, String> headMap, Map<String, Object> formMap) {
        return post(url, null, headMap, MimeTypes.MIME_APPLICATION_JSON, 5, 120, formMap);
    }

    /**
     * 发送post请求
     *
     * @param url               请求地址
     * @param json              请求参数
     * @param headMap           head参数
     * @param contentType       请求类型,为空时,将无请求类型
     * @param connectionTimeout 连接超时时间;单位:秒
     * @param readTimeout       读取超时时间;单位:秒
     * @param formMap           form表单参数
     * @return
     */
    public static Result<String> post(String url, String json, Map<String, String> headMap, String contentType,
                                      int connectionTimeout, int readTimeout, Map<String, Object> formMap) {
        Result<String> result = new Result<String>();
        HttpResponse httpResponse = null;
        try {
            log.info("http|https post request information,targetMain is [{}],contentType is [{}],head is [{}],json is [{}]",
                    url, contentType, headMap == null ? null : JSON.toJSONString(headMap), json);
            HttpRequest httpRequest = createRequest(url, json, Method.POST,
                    headMap, contentType, connectionTimeout, readTimeout, formMap);
            httpResponse = httpRequest.execute();
            result.setCode(httpResponse.getStatus());
            result.setSuccess(CommonConstant.SC_OK_200.equals(result.getCode()));
            result.setResult(httpResponse.body());
            if (!result.isSuccess()) {
                result.setMessage(httpResponse.body());
            }
            log.info("http|https post request return information [{}]", JSON.toJSONString(result));
        } catch (Exception e) {
            log.error("http|https post request  error ,contentType is [{}] seconds," +
                            "connectionTimeout is [{}],readTimeout is [{}] seconds",
                    contentType, connectionTimeout, readTimeout);
            log.error("http|https post request  error ,targetMain is [{}],head is [{}],body is [{}]",
                    url, headMap == null ? null : JSON.toJSONString(headMap), json);
            log.error(e.getMessage(), e);
            result.setSuccess(false);
            result.setCode(ERROR_CODE);
            result.setMessage(e.getMessage());
            log.error("http|https post request return information [{}]", JSON.toJSONString(result));
        } finally {
            if (Objects.nonNull(httpResponse)) {
                httpResponse.close();
            }
        }
        return result;
    }

    /**
     * 发送get请求;contentType默认[application/json]
     *
     * @param url  请求地址(不包含参数)
     * @param json 请求参数
     * @return
     */
    public static Result<String> get(String url, String json) {
        return get(url, json, null, MimeTypes.MIME_APPLICATION_JSON, 5, 120);
    }

    /**
     * 发送get请求;contentType默认[application/json]
     *
     * @param url     请求地址(不包含参数)
     * @param json    json参数
     * @param headMap head参数
     * @return
     */
    public static Result<String> get(String url, String json, Map<String, String> headMap) {
        return get(url, json, headMap, MimeTypes.MIME_APPLICATION_JSON, 5, 120);
    }

    /**
     * 发送get请求
     *
     * @param url               请求地址(不包含参数)
     * @param json              json参数
     * @param headMap           head参数
     * @param contentType       请求类型,为空时,将无请求类型
     * @param connectionTimeout 连接超时时间;单位:秒
     * @param readTimeout       读取超时时间;单位:秒
     * @return
     */
    public static Result<String> get(String url, String json, Map<String, String> headMap, String contentType,
                                     int connectionTimeout, int readTimeout) {
        Result<String> result = new Result<>();
        HttpResponse httpResponse = null;
        try {
            log.info("http|https get request information,targetMain is [{}],contentType is [{}],head is [{}],json is [{}]",
                    url, contentType, headMap == null ? null : JSON.toJSONString(headMap), json);
            HttpRequest httpRequest = createRequest(url, json, Method.GET,
                    headMap, contentType, connectionTimeout, readTimeout, null);
            httpResponse = httpRequest.execute();
            result.setCode(httpResponse.getStatus());
            result.setSuccess(CommonConstant.SC_OK_200.equals(result.getCode()));
            result.setResult(httpResponse.body());
            if (!result.isSuccess()) {
                result.setMessage(httpResponse.body());
            }
            log.info("http|https get request return information [{}]", JSON.toJSONString(result));
        } catch (Exception e) {
            log.error("http|https get request  error ,contentType is [{}] seconds," +
                            "connectionTimeout is [{}],readTimeout is [{}] seconds",
                    contentType, connectionTimeout, readTimeout);
            log.error("http|https get request  error ,targetMain is [{}],head is [{}],json is [{}]",
                    url, headMap == null ? null : JSON.toJSONString(headMap), json);
            log.error(e.getMessage(), e);
            result.setSuccess(false);
            result.setCode(ERROR_CODE);
            result.setMessage(e.getMessage());
            log.error("http|https get request return information [{}]", JSON.toJSONString(result));
        } finally {
            if (Objects.nonNull(httpResponse)) {
                httpResponse.close();
            }
        }
        return result;
    }


    /**
     * 目前只支持 post和get请求,其他未测试
     *
     * @param url               请求地址(不包含参数)
     * @param json              json参数
     * @param method            请求方式(目前只支持 post和get请求,其他未测试)
     * @param headMap           head参数
     * @param contentType       请求类型
     * @param connectionTimeout 连接超时时间;单位:秒
     * @param readTimeout       读取超时时间;单位:秒
     * @return
     */
    public static HttpRequest createRequest(String url, String json, Method method,
                                            Map<String, String> headMap, String contentType,
                                            int connectionTimeout, int readTimeout, Map<String, Object> formMap) {
        String realUrl = url;
        if (Method.GET.equals(method)) {
            if (!url.endsWith("?")) {
                realUrl = realUrl + "?";
            }
            realUrl = realUrl + json;
        }
        HttpRequest httpRequest = HttpUtil.createRequest(method, realUrl);
        httpRequest.setConnectionTimeout(connectionTimeout * 1000);
        httpRequest.setReadTimeout(readTimeout * 1000);
        if (!CollectionUtils.isEmpty(headMap)) {
            httpRequest.addHeaders(headMap);
        }
        if (StringUtils.isNotBlank(contentType)) {
            httpRequest.contentType(contentType);
        }
        if (Method.POST.equals(method)) {
            if (MimeTypes.MIME_APPLICATION_JSON.equals(contentType)) {
                httpRequest.body(json);
            } else {
                if (Objects.nonNull(formMap)) {
                    httpRequest.form(formMap);
                }
            }
        }
        return httpRequest;
    }
}

