package com.biz.crm.business.common.base.util;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;

/**
 * 数字字符串处理工具类
 *
 * @author wanghaojia
 * @date 2022/11/12 14:04
 */
@Slf4j
public class DateStringDealUtil {

    /**
     * 时间格式化工具
     */
    protected static final ThreadLocal<ConcurrentHashMap<String, SimpleDateFormat>> DATE_FORMAT_MAP_THREAD_LOCAL = new ThreadLocal<>();

    /**
     * 如果值不为空则调用set方法
     *
     * @param obj       要设置的值
     * @param setMethod set方法
     */
    public static void setStringIfNotNull(Date obj, String pattern, Consumer<String> setMethod) {
        if (null != obj) {
            String value = getDateFormat(pattern).format(obj);
            setMethod.accept(value);
        }
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值-默认做必填校验
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param setMethod 赋值方法
     * @param <T>       转换类型
     */
    public static <T> void validateDateStrAndSet(String str, String desc, String pattern, Consumer<Date> setMethod) {
        validateDateStrAndSet(str, desc, true, pattern, setMethod);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param required  是否校验必填
     * @param setMethod 赋值方法
     */
    public static void validateDateStrAndSet(String str, String desc, boolean required, String pattern, Consumer<Date> setMethod) {
        String errorMsg = validateDateStrAndSetErrorMsg(str, desc, required, pattern, setMethod);
        Assert.isTrue(StringUtils.isEmpty(errorMsg), errorMsg);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值-默认做必填校验
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param setMethod 赋值方法
     */
    public static String validateDateStrAndSetErrorMsg(String str, String desc, String pattern, Consumer<Date> setMethod) {
        return validateDateStrAndSetErrorMsg(str, desc, true, pattern, setMethod);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param required  是否校验必填
     * @param setMethod 赋值方法
     */
    public static String validateDateStrAndSetErrorMsg(String str, String desc, boolean required, String pattern, Consumer<Date> setMethod) {
        if (StringUtils.isBlank(str)) {
            if (required) {
                return desc + "不能为空";
            }
            return "";
        }
        if (StringUtils.isBlank(pattern)) {
            return desc + "的时间格式[pattern]不能为空";
        }
        try {
            Date value = getDateFormat(pattern).parse(str);
            if (!str.equals(DateUtil.format(value, pattern))) {
                return desc + "的时间不合法";
            }
            setMethod.accept(value);
            return "";
        } catch (ParseException e) {
            return desc + "[" + str + "]格式有误";
        }
    }

    public static SimpleDateFormat getDateFormat(String pattern) {
        ConcurrentHashMap<String, SimpleDateFormat> dateFormatMap = DATE_FORMAT_MAP_THREAD_LOCAL.get();
        if (dateFormatMap == null) {
            dateFormatMap = new ConcurrentHashMap<>(16);
        } else if (dateFormatMap.containsKey(pattern)) {
            return dateFormatMap.get(pattern);
        }
        SimpleDateFormat dateFormat = new SimpleDateFormat(pattern);
        dateFormatMap.put(pattern, dateFormat);
        DATE_FORMAT_MAP_THREAD_LOCAL.set(dateFormatMap);
        return dateFormat;
    }

}
