package com.biz.crm.business.common.base.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.connection.DataType;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;


/**
 * redis hash 类型 基础方法
 *
 * @author huxmld
 * @version v1.0.0
 * @date 2022.12.13 19:27
 */
@Service
@Slf4j
public class RedisCrmService {

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    public Long incr(String key, long delta) {
        return this.redisTemplate.opsForValue().increment(key, delta);
    }

    public Long incr(String key, long delta, TimeUnit timeUnit, long expireTime) {
        Long count = null;
        try {
            count = this.redisTemplate.opsForValue().increment(key, delta);
            this.expire(key, timeUnit, expireTime);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            throw e;
        }
        return count;
    }

    public Long decr(String key, long delta) {
        return this.redisTemplate.opsForValue().increment(key, -delta);
    }

    public Long decr(String key, long delta, TimeUnit timeUnit, long expireTime) {
        Long count = null;
        try {
            count = this.redisTemplate.opsForValue().increment(key, -delta);
            this.expire(key, timeUnit, expireTime);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            throw e;
        }
        return count;
    }

    /**
     * 设置过期时间
     *
     * @param key      key
     * @param timeUnit 时间类型
     * @param time     时间
     * @return
     */
    public Boolean expire(String key, TimeUnit timeUnit, long time) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        return this.redisTemplate.expire(key, time, timeUnit);
    }

    public Object hget(String key, String item) {
        return this.redisTemplate.opsForHash().get(key, item);
    }

    public List<Object> hmget(String key, Set<Object> item) {
        return this.redisTemplate.opsForHash().multiGet(key, item);
    }

    public Object hmget(String key, Object item) {
        return this.redisTemplate.opsForHash().get(key, item);
    }

    public Boolean hmHasKey(String hash, Object key) {
        return this.redisTemplate.opsForHash().hasKey(hash, key);
    }

    public Map<?, ?> hmget(String key) {
        return this.redisTemplate.opsForHash().entries(key);
    }

    public boolean hmset(String key, Map<?, ?> map, TimeUnit timeUnit, long time) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        try {
            this.redisTemplate.opsForHash().putAll(key, map);
            this.expire(key, timeUnit, time);
            return true;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return false;
        }
    }

    public boolean hset(String key, String item, Object value, TimeUnit timeUnit, long time) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        try {
            this.redisTemplate.opsForHash().put(key, item, value);
            this.expire(key, timeUnit, time);
            return true;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return false;
        }
    }

    public void hdel(String key, Object... item) {
        this.redisTemplate.opsForHash().delete(key, item);
    }

    public boolean hHasKey(String key, String item) {
        return this.redisTemplate.opsForHash().hasKey(key, item);
    }

    public double hincr(String key, String item, double by) {
        return this.redisTemplate.opsForHash().increment(key, item, by);
    }

    public double hdecr(String key, String item, double by) {
        return this.redisTemplate.opsForHash().increment(key, item, -by);
    }

    public Set<Object> sGet(String key) {
        try {
            return this.redisTemplate.opsForSet().members(key);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return null;
        }
    }

    public boolean sHasKey(String key, Object value) {
        try {
            return this.redisTemplate.opsForSet().isMember(key, value);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return false;
        }
    }


    public long sSet(String key, TimeUnit timeUnit, long time, Object... values) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        try {
            Long count = this.redisTemplate.opsForSet().add(key, values);
            this.expire(key, timeUnit, time);
            return count;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return 0L;
        }
    }

    public List<Object> lGet(String key, long start, long end) {
        try {
            return this.redisTemplate.opsForList().range(key, start, end);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return null;
        }
    }

    public long lGetListSize(String key) {
        try {
            return this.redisTemplate.opsForList().size(key);
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return 0L;
        }
    }

    public Object lGetIndex(String key, long index, long time) {
        try {
            Object obj = this.redisTemplate.opsForList().index(key, index);
            return obj;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return null;
        }
    }


    public boolean lSet(String key, Object value, TimeUnit timeUnit, long time) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        try {
            this.redisTemplate.opsForList().rightPush(key, value);
            this.expire(key, timeUnit, time);
            return true;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return false;
        }
    }

    public boolean lSet(String key, Collection value, TimeUnit timeUnit, long time) {
        Assert.notNull(timeUnit, "时间类型不能为空!");
        Assert.isTrue(time >= 0, "时间不能小于0!");
        try {
            this.redisTemplate.opsForList().rightPushAll(key, value);
            this.expire(key, timeUnit, time);
            return true;
        } catch (Exception e) {
            log.error("redis异常");
            log.error(e.getMessage(), e);
            return false;
        }
    }


    public Set<String> keys(String pattern) {
        return this.redisTemplate.keys(pattern);
    }

    public DataType type(String key) {
        return this.redisTemplate.type(key);
    }


}
