package com.biz.crm.business.common.base.util;

import org.apache.commons.lang3.StringUtils;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.function.Consumer;

/**
 * 数字处理工具类
 *
 * @author wanghaojia
 * @date 2022/12/12 20:37
 */
public class NumberDealUtil {

    public static BigDecimal decimalSum(BigDecimal... decimals) {
        BigDecimal result = BigDecimal.ZERO;
        for (BigDecimal decimal : decimals) {
            if (Objects.nonNull(decimal)) {
                result = result.add(decimal);
            }
        }
        return result;
    }

    /**
     * 如果值不为空则调用set方法
     *
     * @param obj       要设置的值
     * @param setMethod set方法
     */
    public static void setStringIfNotNull(Object obj, Consumer<String> setMethod) {
        if (null != obj) {
            setMethod.accept(obj.toString());
        }
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值-默认做必填校验
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param setMethod 赋值方法
     * @param cls       转换类型
     * @param <T>       转换类型
     */
    public static <T> void validateNumberStrAndSet(String str, String desc, Consumer<T> setMethod, Class<T> cls) {
        validateNumberStrAndSet(str, desc, true, setMethod, cls);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param required  是否校验必填
     * @param setMethod 赋值方法
     * @param cls       转换类型
     * @param <T>       转换类型
     */
    public static <T> void validateNumberStrAndSet(String str, String desc, boolean required, Consumer<T> setMethod, Class<T> cls) {
        String errorMsg = validateNumberStrAndSetErrorMsg(str, desc, required, setMethod, cls);
        Assert.isTrue(StringUtils.isEmpty(errorMsg), errorMsg);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值-默认做必填校验
     * 返回错误信息
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param setMethod 赋值方法
     * @param cls       转换类型
     * @param <T>       转换类型
     */
    public static <T> String validateNumberStrAndSetErrorMsg(String str, String desc, Consumer<T> setMethod, Class<T> cls) {
        return validateNumberStrAndSetErrorMsg(str, desc, true, setMethod, cls);
    }

    /**
     * 校验Integer,BigDecimal 字符串转对应类型并赋值
     * 返回错误信息
     *
     * @param str       字符串
     * @param desc      报错描述
     * @param required  是否校验必填
     * @param setMethod 赋值方法
     * @param cls       转换类型
     * @param <T>       转换类型
     */
    public static <T> String validateNumberStrAndSetErrorMsg(String str, String desc, boolean required, Consumer<T> setMethod, Class<T> cls) {
        if (StringUtils.isBlank(str)) {
            if (required) {
                return desc + "不能为空||";
            }
            return "";
        }
        try {
            T value = null;
            if (cls == Integer.class) {
                value = (T) Integer.valueOf(str.trim());
            } else if (cls == BigDecimal.class) {
                value = (T) new BigDecimal(str.trim());
            }
            setMethod.accept(value);
            return "";
        } catch (NumberFormatException e) {
            return desc + "[" + str + "]格式有误||";
        }
    }

}
