package com.biz.crm.business.common.base.aspect;

import cn.hutool.json.JSONUtil;
import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.base.vo.WebLogVo;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.*;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import static jodd.net.MimeTypes.MIME_APPLICATION_JSON;

/**
 * @author huxmld
 * @version v1.0.0
 * @describe 进参出参控制台打印
 * @date 2022.10.13 11:52
 */
@Aspect
@Component
@Slf4j
public class WebLogAspect {

    protected static final ThreadLocal<ConcurrentHashMap<String, WebLogVo>> THREAD_LOCAL_REQ = new ThreadLocal<>();


    @Autowired(required = false)
    private LoginUserService loginUserService;

    private final static String TIP_MSG = "未获取到入参信息";

    @Pointcut(value = "execution(public * com..*.controller..*.*(..))")
    public void openApiLog() {
    }

    /**
     * 前置通知：
     * 1. 在执行目标方法之前执行，比如请求接口之前的登录验证;
     * 2. 在前置通知中设置请求日志信息，如开始时间，请求参数，注解内容等
     */
    @Before("openApiLog()")
    public void doBefore(JoinPoint joinPoint) {
        try {
            ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            HttpServletRequest request = null;
            String uuid = UUID.randomUUID().toString().replaceAll("-", "");
            if (requestAttributes != null) {
                request = requestAttributes.getRequest();
            }
            WebLogVo logVo = new WebLogVo();
            logVo.setUuid(uuid);
            logVo.setTime(System.currentTimeMillis());
            addLogVo(logVo);
            AbstractCrmUserIdentity loginDetails = Objects.nonNull(loginUserService) ? loginUserService.getAbstractLoginUser() : null;
            log.info("[AOP请求日志]:\n 请求唯一标记:[{}] \n 请求用户:[{}]", uuid, Objects.isNull(loginDetails) ? "未获取到用户信息" : JSONUtil.toJsonStr(loginDetails));
            if (request != null) {
                String contentType = request.getContentType();
                Class<?> clazz = joinPoint.getTarget().getClass();
                Method method = ((MethodSignature) joinPoint.getSignature()).getMethod();
                ApiOperation apiOperation = method.getAnnotation(ApiOperation.class);
                Map<String, String> map = new HashMap<>(32);
                Enumeration headerNames = request.getHeaderNames();
                while (headerNames.hasMoreElements()) {
                    String key = (String) headerNames.nextElement();
                    String value = request.getHeader(key);
                    map.put(key, value);
                }
                String bodyData = TIP_MSG;
                if (Objects.nonNull(joinPoint.getArgs())) {
                    Object[] args = joinPoint.getArgs();
                    Object[] arguments = new Object[args.length];
                    if (args.length == 1) {
                        bodyData = JSONUtil.toJsonStr(args[0]);
                    } else {
                        for (int i = 0; i < args.length; i++) {
                            if (args[i] instanceof ServletRequest
                                    || args[i] instanceof ServletResponse
                                    || args[i] instanceof MultipartFile) {
                                //ServletRequest不能序列化，从入参里排除，否则报异常：java.lang.IllegalStateException: It is illegal to call this method if the current request is not in asynchronous mode (i.e. isAsyncStarted() returns false)
                                //ServletResponse不能序列化 从入参里排除，否则报异常：java.lang.IllegalStateException: getOutputStream() has already been called for this response
                                continue;
                            }
                            arguments[i] = args[i];
                        }
                        bodyData = JSONUtil.toJsonStr(arguments);
                    }
                } else {
                    Map<String, Object> dataMap = new HashMap<>(16);
                    Enumeration enu = request.getParameterNames();
                    while (enu.hasMoreElements()) {
                        String paraName = (String) enu.nextElement();
                        String paraValue = request.getParameter(paraName);
                        dataMap.put(paraName, paraValue);
                    }
                    bodyData = JSONUtil.toJsonStr(dataMap);
                }

                log.info("[AOP请求日志]:\n 请求唯一标记:[{}] \n apiOperation注释:[{}] " +
                                "\n 请求方式:[{}] \n contentType:[{}] \n uri:[{}]  " +
                                "\n 方法:[{}.{}] " +
                                "\n header入参:{} \n body入参:{}",
                        uuid, Objects.isNull(apiOperation) ? "无注释" : apiOperation.value(),
                        request.getMethod(), contentType, clazz.getName(),
                        request.getRequestURI(), method.getName(), map, bodyData);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

    }

    /**
     * 返回通知：
     * 1. 在执行目标方法之后执行，比如请求接口之后的日志保存;
     * 2. 在返回通知设置返回信息，如返回时间，返回参数等
     */
    @AfterReturning(value = "openApiLog()",
            returning = "result")
    public void AfterReturning(Object result) {
        try {
            WebLogVo logVo = this.getLogVo();
            String uuid = Objects.isNull(logVo) ? TIP_MSG : logVo.getUuid();
            String msg = Objects.isNull(logVo) ? TIP_MSG : millisecondToStr(System.currentTimeMillis() - logVo.getTime());

            String resultJson = buildSafeJson(result, 2000); // 限制JSON最大长度，防止OOM

            if (result != null) {
                if (result instanceof Result) {
                    log.info("[AOP返回日志]: 请求唯一标记[{}] 处理耗时[{}] \n Result类型[{}]", uuid, msg, resultJson);
                } else {
                    ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
                    if (requestAttributes != null) {
                        HttpServletResponse response = requestAttributes.getResponse();
                        if (response != null && MIME_APPLICATION_JSON.equals(response.getContentType())) {
                            log.info("[AOP返回日志]: 请求唯一标记[{}] 处理耗时[{}] \n 非Result类型[{}]", uuid, msg, resultJson);
                        }
                    }
                }
            } else {
                log.info("[AOP返回日志]: 请求唯一标记[{}] 处理耗时[{}] \n 返回数据为 null", uuid, msg);
            }
        } catch (Exception e) {
            log.error("记录AOP返回日志异常: {}", e.getMessage(), e);
        }
    }

    /**
     * 异常通知：
     * 1. 在执行目标方法抛出异常之后;
     * 2. 在异常通知设置返回的异常信息
     */
    @AfterThrowing(value = "openApiLog()", throwing = "e")
    public void AfterThrowing(Exception e) {
        try {
            WebLogVo logVo = this.getLogVo();
            String uuid = Objects.isNull(logVo) ? TIP_MSG : logVo.getUuid();
            String msg = Objects.isNull(logVo) ? TIP_MSG : millisecondToStr(System.currentTimeMillis() - logVo.getTime());
            log.info("[AOP返回日志]:请求唯一标记[{}] 处理耗时[{}]  \n 异常信息:{}", uuid, msg, e.getMessage());
        } catch (Exception ex) {
            log.error(ex.getMessage(), ex);

        }
    }

    /**
     * 添加日志对象
     *
     * @param respVo
     * @return
     * @describe 日志
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/13 12:33
     */
    private void addLogVo(WebLogVo respVo) {
        Thread thread = Thread.currentThread();
        ConcurrentHashMap<String, WebLogVo> logMap = THREAD_LOCAL_REQ.get();
        if (logMap == null) {
            logMap = new ConcurrentHashMap<>(16);
        }
        logMap.put(thread.getName() + "_" + thread.getId(), respVo);
        THREAD_LOCAL_REQ.set(logMap);
    }

    /**
     * 获取日志对象
     *
     * @return
     * @describe 日志
     * @author huxmld
     * @version v1.0.0
     * @date 2021/8/13 12:33
     */
    private WebLogVo getLogVo() {
        Thread thread = Thread.currentThread();
        ConcurrentHashMap<String, WebLogVo> logMap = THREAD_LOCAL_REQ.get();
        if (logMap == null) {
            logMap = new ConcurrentHashMap<>(16);
        }
        String key = thread.getName() + "_" + thread.getId();
        WebLogVo respVo = logMap.get(key);
        logMap.remove(key);
        THREAD_LOCAL_REQ.set(logMap);
        return respVo;
    }

    /**
     * 毫秒转  天小时分秒
     *
     * @param millisecond 毫秒
     * @return
     */
    private static String millisecondToStr(long millisecond) {
        long days = millisecond / 86400000;
        long hours = (millisecond % 86400000) / 3600000;
        long minute = (millisecond % 86400000) % 3600000 / 60000;
        long second = (millisecond % 86400000) % 3600000 % 60000 / 1000;
        long millisecondStr = (millisecond % 86400000) % 3600000 % 60000 % 1000;
        String spendTimes = "";
        if (days > 0) {
            spendTimes = spendTimes + days + "天";
        }
        if (hours > 0) {
            spendTimes = spendTimes + hours + "小时";
        }
        if (minute > 0) {
            spendTimes = spendTimes + minute + "分钟";
        }
        if (second > 0) {
            spendTimes = spendTimes + second + "秒";
        }
        if (millisecondStr > 0
                || StringUtils.isEmpty(spendTimes)) {
            spendTimes = spendTimes + millisecondStr + "毫秒";
        }
        return spendTimes;
    }

    /**
     * 安全构建 JSON 字符串，避免内存溢出
     */
    private String buildSafeJson(Object obj, int maxLength) {
        if (obj == null) return "null";

        try {
            String json = JSONUtil.toJsonStr(obj);
            if (json.length() > maxLength) {
                return json.substring(0, maxLength) + "...（已截断）";
            }
            return json;
        } catch (Throwable e) {
            return "JSON序列化失败：" + e.getMessage();
        }
    }
}
