package com.biz.crm.business.common.base.util;

import com.biz.crm.business.common.sdk.utils.HttpServletRequestUtil;
import lombok.extern.slf4j.Slf4j;

import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;

/**
 * 获取IP
 *
 * @author huxmld
 * @version v1.0.0
 * @date 2023-06-26 16:09
 */
@Slf4j
public class IpUtils {

    /**
     * 识别异常常量
     */
    private final static String UNKNOWN = "unknown";
    /**
     * 查询本机公网IP
     */
    private final static String[] CMDS = {"curl", "icanhazip.com"};

    /**
     * 获取请求服务的IP地址
     *
     * @return
     */
    public static String findRequestIp() {
        return findRequestIp(HttpServletRequestUtil.getRequest());
    }

    /**
     * 获取请求服务的IP地址
     *
     * @return
     */
    public static String findRequestIp(HttpServletRequest request) {
        if (request == null) {
            try {
                return findLocalPublicIp();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            return "127.0.0.1";
        }
        String ip = request.getHeader("X-Real-IP");

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("X-Forwarded-For");
        }

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("x-forwarded-for");
        }

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }

        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }


        if (ip == null || ip.length() == 0 || UNKNOWN.equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }

        return ip;
    }


    /**
     * 查询本机访问公网时,出口IP
     *
     * @return
     */
    public static String findPublicIp() {
        ProcessBuilder process = new ProcessBuilder(CMDS);
        Process p;
        try {
            p = process.start();
            BufferedReader reader = new BufferedReader(new InputStreamReader(p.getInputStream()));
            StringBuilder builder = new StringBuilder();
            String line = null;
            while ((line = reader.readLine()) != null) {
                builder.append(line);
                builder.append(System.getProperty("line.separator"));
            }
            return builder.toString();
        } catch (IOException e) {
            log.error("查询本机公网[{}]IP失败!", CMDS);
            log.error(e.getMessage(), e);
        }
        return "127.0.0.1";
    }

    /**
     * 查询本机公网IP
     * 当是内网时,获取到的是内网IP
     *
     * @return
     */
    public static String findLocalPublicIp() {
        try {
            InetAddress inetAddress = InetAddress.getLocalHost();
            return inetAddress.getHostAddress();
        } catch (Exception e) {
            log.error("查询本机公网IP失败!");
            log.error(e.getMessage(), e);
        }
        return findLocalInternalIp();
    }

    /**
     * 查询本机内网IP
     *
     * @return
     */
    public static String findLocalInternalIp() {
        try {

            return com.bizunited.nebula.common.util.IpUtils.findInternalIp();
        } catch (Exception e) {
            log.error("查询本机内网IP失败!");
            log.error(e.getMessage(), e);

        }
        return "127.0.0.1";
    }
}