package com.biz.crm.business.common.base.util;

import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.BeansException;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.util.CollectionUtils;

import java.beans.PropertyDescriptor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author wanghaojia
 * @date 2023/1/29 07:24
 */
@Slf4j
public class ObjectConvertStringUtil {

    private static final String END_STR = "end";
    private static final String END_STR_UP = "End";

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void fillObjectListStrProperties(List<T> list, Class<T> cls) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(cls);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors)
                .collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        Map<String, Field> dateFieldMap = getPropertyFieldMap(cls);
        for (T t : list) {
            fillObjectStrProperties(t, dateFieldMap, propertyMap);
        }
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void fillObjectStrProperties(T t, Class<T> cls) {
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(cls);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        fillObjectStrProperties(t, cls, propertyMap);
    }

    public static <T> void fillObjectStrProperties(T t, Class<T> cls, Map<String, PropertyDescriptor> propertyMap) {
        if (null == t) {
            return;
        }
        Map<String, Field> dateFieldMap = getPropertyFieldMap(cls);
        fillObjectStrProperties(t, dateFieldMap, propertyMap);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void fillObjectStrProperties(T t, Map<String, Field> dateFieldMap, Map<String, PropertyDescriptor> propertyMap) {
        if (null == t) {
            return;
        }
        for (Map.Entry<String, PropertyDescriptor> entry : propertyMap.entrySet()) {
            PropertyDescriptor propertyDescriptor = entry.getValue();
            //小数、整数、时间类型的做下转换
            if (!(BigDecimal.class == propertyDescriptor.getPropertyType()) &&
                    !(Integer.class == propertyDescriptor.getPropertyType()) &&
                    !(Date.class == propertyDescriptor.getPropertyType())) {
                continue;
            }
            String fieldName = propertyDescriptor.getName();
            String strPropertyName = fieldName + "Str";
            if (!propertyMap.containsKey(strPropertyName)) {
                //没有定义对应字符串属性就跳过
                continue;
            }
            try {
                Object propertyValue = propertyDescriptor.getReadMethod().invoke(t);
                if (null == propertyValue) {
                    continue;
                }
                String strValue = null;
                if (propertyValue instanceof Date) {
                    //如果是日期类型的，找注解的格式转换
                    Field declaredField = dateFieldMap.get(fieldName);
                    DateTimeFormat dateTimeFormat = declaredField.getDeclaredAnnotation(DateTimeFormat.class);
                    if (null == dateTimeFormat) {
                        continue;
                    }
                    String dateTimeFormatPattern = DateUtil.DEFAULT_DATE_ALL_PATTERN;
                    if (StringUtils.isNotEmpty(dateTimeFormat.pattern())) {
                        dateTimeFormatPattern = dateTimeFormat.pattern();
                    }
                    strValue = DateStringDealUtil.getDateFormat(dateTimeFormatPattern).format(propertyValue);
                } else {
                    strValue = propertyValue.toString();
                }
                PropertyDescriptor strPropertyDescriptor = propertyMap.get(strPropertyName);
                strPropertyDescriptor.getWriteMethod().invoke(t, strValue);
            } catch (IllegalAccessException | InvocationTargetException e) {
                log.error("属性[" + fieldName + "]转换失败", e);
                log.error(e.getMessage(), e);
            }
        }
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(List<T> list, Class<T> cls) {
        convertObjectListStrProperties(list, cls, false);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(List<T> list, Class<T> cls, Integer scale) {
        convertObjectListStrProperties(list, cls, scale, false, null);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(List<T> list, Class<T> cls, boolean throwException) {
        convertObjectListStrProperties(list, cls, false, null);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(List<T> list, Class<T> cls, boolean throwException, Map<String, String> fieldTitleMap) {
        convertObjectListStrProperties(list, cls, null, false, null);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(List<T> list, Class<T> cls, Integer scale, boolean throwException, Map<String, String> fieldTitleMap) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(cls);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors)
                .collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        Map<String, Field> dateFieldMap = getPropertyFieldMap(cls);
        for (T t : list) {
            convertObjectListStrProperties(t, dateFieldMap, propertyMap, scale, throwException, fieldTitleMap);
        }
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(T t, Class<T> cls) {
        PropertyDescriptor[] propertyDescriptors = BeanUtils.getPropertyDescriptors(cls);
        Map<String, PropertyDescriptor> propertyMap = Arrays.stream(propertyDescriptors).collect(Collectors.toMap(PropertyDescriptor::getName, Function.identity()));
        convertObjectListStrProperties(t, cls, propertyMap, false);
    }

    public static <T> void convertObjectListStrProperties(T t, Class<T> cls, Map<String, PropertyDescriptor> propertyMap, boolean throwException) {
        convertObjectListStrProperties(t, cls, propertyMap, throwException, null);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(T t, Class<T> cls, Map<String, PropertyDescriptor> propertyMap, boolean throwException, Map<String, String> fieldTitleMap) {
        convertObjectListStrProperties(t, cls, propertyMap, null, throwException, fieldTitleMap);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(T t, Class<T> cls, Map<String, PropertyDescriptor> propertyMap, Integer scale,
                                                          boolean throwException, Map<String, String> fieldTitleMap) {
        Map<String, Field> dateFieldMap = getPropertyFieldMap(cls);
        convertObjectListStrProperties(t, dateFieldMap, propertyMap, scale, throwException, fieldTitleMap);
    }

    /**
     * 设置对象里面的几个str属性，前端用此字段，避免出现格式问题
     */
    public static <T> void convertObjectListStrProperties(T t, Map<String, Field> dateFieldMap, Map<String, PropertyDescriptor> propertyMap, Integer scale, boolean throwException, Map<String, String> fieldTitleMap) {
        if (null == t) {
            return;
        }
        for (Map.Entry<String, PropertyDescriptor> entry : propertyMap.entrySet()) {
            PropertyDescriptor propertyDescriptor = entry.getValue();
            if (!(propertyDescriptor.getPropertyType() == String.class)) {
                continue;
            }
            String fieldName = propertyDescriptor.getName();
            if (!fieldName.endsWith("Str")) {
                continue;
            }
            String originPropertyName = fieldName.substring(0, fieldName.indexOf("Str"));
            if (!propertyMap.containsKey(originPropertyName)) {
                //没有定义对应字符串属性就跳过
                continue;
            }
            Object propertyValueObj = null;
            try {
                propertyValueObj = propertyDescriptor.getReadMethod().invoke(t);
                if (null == propertyValueObj) {
                    continue;
                }
                String propertyValue = propertyValueObj.toString().trim();
                if (StringUtils.isEmpty(propertyValue)) {
                    continue;
                }
                Object originValue = null;
                PropertyDescriptor originPropertyDescriptor = propertyMap.get(originPropertyName);
                originPropertyDescriptor.getPropertyType();
                if (originPropertyDescriptor.getPropertyType() == Date.class) {
                    //如果是日期类型的，找注解的格式转换
                    Field declaredField = dateFieldMap.get(originPropertyDescriptor.getName());
                    DateTimeFormat dateTimeFormat = declaredField.getDeclaredAnnotation(DateTimeFormat.class);
                    if (null == dateTimeFormat) {
                        continue;
                    }
                    String dateTimeFormatPattern = DateUtil.DEFAULT_DATE_ALL_PATTERN;
                    if (StringUtils.isNotEmpty(dateTimeFormat.pattern())) {
                        dateTimeFormatPattern = dateTimeFormat.pattern();
                    }
                    //加个逻辑，如果是结束日期，时分秒设置成23:59:59
                    if (dateTimeFormatPattern.equals(DateUtil.DEFAULT_YEAR_MONTH_DAY) &&
                            (originPropertyName.contains(END_STR_UP)) || originPropertyName.contains(END_STR)) {
                        dateTimeFormatPattern = DateUtil.DEFAULT_DATE_ALL_PATTERN;
                        propertyValue = propertyValue + " " + DateUtil.DAY_LATEST_TIME;
                    }
                    originValue = DateStringDealUtil.getDateFormat(dateTimeFormatPattern).parse(propertyValue);
                } else if (originPropertyDescriptor.getPropertyType() == Integer.class) {
                    originValue = Integer.valueOf(propertyValue);
                } else if (originPropertyDescriptor.getPropertyType() == BigDecimal.class) {
                    if (null == scale) {
                        originValue = new BigDecimal(propertyValue);
                    } else {
                        //默认四舍五入
                        originValue = new BigDecimal(propertyValue).setScale(scale, RoundingMode.HALF_UP);
                    }
                }
                originPropertyDescriptor.getWriteMethod().invoke(t, originValue);
            } catch (IllegalAccessException | InvocationTargetException | ParseException | NumberFormatException e) {
                log.error("属性[" + fieldName + "][" + propertyValueObj + "]转换失败");
                log.error(e.getMessage(), e);
                if (throwException) {
                    String fieldTitle = fieldName;
                    if (null != fieldTitleMap && fieldTitleMap.containsKey(fieldName)) {
                        fieldTitle = fieldTitleMap.get(fieldName);
                    }
                    throw new IllegalArgumentException("属性[" + fieldTitle + "][" + propertyValueObj + "]转换失败");
                }
            }
        }
    }


    public static Map<String, Field> getPropertyFieldMap(Class<?> clazz) throws BeansException {
        Map<String, Field> propertyFieldMap = Maps.newHashMap();
        return getPropertyFieldMap(clazz, propertyFieldMap);
    }

    public static Map<String, Field> getPropertyFieldMap(Class<?> clazz, Map<String, Field> propertyFieldMap) throws BeansException {
        Field[] declaredFields = clazz.getDeclaredFields();
        for (Field declaredField : declaredFields) {
            propertyFieldMap.put(declaredField.getName(), declaredField);
        }
        if (clazz.getSuperclass() != Object.class) {
            getPropertyFieldMap(clazz.getSuperclass(), propertyFieldMap);
        }
        return propertyFieldMap;
    }
}
