package com.biz.crm.business.common.base.task;

import com.biz.crm.business.common.base.config.ExportTaskConfig;
import com.biz.crm.business.common.base.service.RedisLockService;
import com.biz.crm.business.common.base.util.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.orm.jpa.vendor.Database;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * 系统定时清除日志
 *
 * @author huxmld
 * @version v1.0.0
 * @describe 简述
 * @date 2022.10.14 13:43
 */
@EnableScheduling
@Component
@Slf4j
public class ExportTaskCornTask implements SchedulingConfigurer {

    private static final String DEFAULT_CRON = "0 2/30 * * * ?";

    private String cron = DEFAULT_CRON;

    @Resource
    private ExportTaskConfig exportTaskConfig;

    @Autowired(required = false)
    private RedisLockService redisLockService;


    @Autowired(required = false)
    private JdbcTemplate jdbcTemplate;


    @Value("${spring.jpa.database:}")
    private String databaseType;


    /**
     * 根据配置文件参数,系统启动完成后,自动设置自定义定时任务执行计划
     */
    @PostConstruct
    private void setDeleteMqLogCron() {
        this.setCron(exportTaskConfig.getCleanCron());
    }

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.addTriggerTask(this::exportTask, triggerContext -> {
            CronTrigger trigger = new CronTrigger(cron);
            return trigger.nextExecutionTime(triggerContext);
        });
    }

    public void setCron(String cron) {
        log.info("====>     导出异常任务自动重置任务:原来的cron：[{}]]更新后的cron：[{}]     <=====", this.cron, cron);
        this.cron = cron;
    }

    public String getCron() {
        return this.cron;
    }

    /**
     * 导出任务状态重置
     */
    private void exportTask() {
        if (Objects.isNull(exportTaskConfig.getIsOpen())
                || !exportTaskConfig.getIsOpen()
                || StringUtils.isEmpty(databaseType)) {
            return;
        }
        if (!Database.MYSQL.name().equalsIgnoreCase(databaseType)
                && !Database.ORACLE.name().equalsIgnoreCase(databaseType)) {
            return;
        }

        String lockKey = "export:task:lock:" + exportTaskConfig.getAppName();
        if (redisLockService != null) {
            boolean lock = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30);
            if (!lock) {
                return;
            }
        }

        String mainTableName = "ie_export_task";
        String detailTableName = "ie_export_task_detail";
        //默认为  4小时
        Integer hour = Objects.isNull(exportTaskConfig.getSaveHour()) ? 4 : exportTaskConfig.getSaveHour();
        Date startDate = DateUtil.dateAddDay(new Date(), -1);
        Date endDate = DateUtil.dateAddHour(new Date(), -hour);
        String startDateStr = DateUtil.date_yyyy_MM_dd_HH_mm_ss.format(startDate);
        String endDateStr = DateUtil.date_yyyy_MM_dd_HH_mm_ss.format(endDate);

        if (Database.MYSQL.name().equalsIgnoreCase(databaseType)) {
            mainTableName = mainTableName.toLowerCase();
            detailTableName = detailTableName.toLowerCase();
            try {
                String sql = "SELECT SCHEMA()";
                String schemaName = jdbcTemplate.queryForObject(sql, String.class);
                if (StringUtils.isNotEmpty(schemaName)) {
                    sql = "SELECT IF(EXISTS (SELECT 1 " + "    FROM INFORMATION_SCHEMA.TABLES " + "    WHERE TABLE_SCHEMA = '" + schemaName + "' AND TABLE_NAME = '" + mainTableName + "'" + "  ),'1','0') AS 'table_Status'";
                    String tableExit = jdbcTemplate.queryForObject(sql, String.class);
                    if (StringUtils.isNotEmpty(tableExit) && "1".equals(tableExit)) {
                        sql = "UPDATE " + mainTableName + " t " + " SET t.exec_status = '25' " + " WHERE t.create_time >'" + startDateStr + "'" + " and t.create_time <'" + endDateStr + "'" + " AND t.exec_status in ('0','5')";
                        int count = jdbcTemplate.update(sql);

                        sql = "UPDATE " + detailTableName + " t " + " SET t.exec_status = '25' " + " WHERE t.create_time >'" + startDateStr + "'" + " and t.create_time <'" + endDateStr + "'" + " AND t.exec_status in ('0','5')";
                        count = jdbcTemplate.update(sql);

                        sql = "UPDATE " + detailTableName + " t " + " SET t.load_status = '1' " + " WHERE t.create_time >'" + startDateStr + "'" + " and t.create_time <'" + endDateStr + "'" + " AND t.load_status  = '0'";
                        count = jdbcTemplate.update(sql);
                        log.info("=====>    {} 导出异常任务自动重置执行完成!    <=====", databaseType);
                    } else {
                        log.info("=====>    {} 导出异常任务自动重置未执行1,无导出功能!    <=====", databaseType);
                    }
                } else {
                    log.info("=====>    {} 导出异常任务自动重置未执行2,无导出功能!    <=====", databaseType);
                }
            } catch (Exception e) {
                log.error("=====>    {} 异常导出任务自动重置任务状态未失败异常    <=====", databaseType);
                log.error(e.getMessage(), e);
            }
        } else if (Database.ORACLE.name().equalsIgnoreCase(databaseType)) {
            mainTableName = mainTableName.toUpperCase();
            detailTableName = detailTableName.toUpperCase();
            try {
                String sql = "SELECT table_name FROM user_tables WHERE table_name = '" + mainTableName + "'";
                String tableExit = jdbcTemplate.queryForObject(sql, String.class);
                if (StringUtils.isNotEmpty(tableExit)) {
                    sql = "UPDATE " + mainTableName + " t " + " SET t.exec_status = '25' " + " WHERE t.create_time > to_date('" + startDateStr + "', 'yyyy-MM-dd HH24:MI:SS')" +
                            " and t.create_time < to_date('" + endDateStr + "', 'yyyy-MM-dd HH24:MI:SS') AND t.exec_status in ('0','5')";
                    int count = jdbcTemplate.update(sql);

                    sql = "UPDATE " + detailTableName + " t " + " SET t.exec_status = '25' " + " WHERE t.create_time > to_date('" + startDateStr + "', 'yyyy-MM-dd HH24:MI:SS')" +
                            " and t.create_time < to_date('" + endDateStr + "', 'yyyy-MM-dd HH24:MI:SS') AND t.exec_status in ('0','5')";
                    count = jdbcTemplate.update(sql);

                    sql = "UPDATE " + detailTableName + " t " + " SET t.load_status = '1' " + " WHERE t.create_time > to_date('" + startDateStr + "', 'yyyy-MM-dd HH24:MI:SS')" +
                            " and t.create_time < to_date('" + endDateStr + "', 'yyyy-MM-dd HH24:MI:SS') AND t.load_status  = '0'";
                    count = jdbcTemplate.update(sql);
                    log.info("=====>    {} 导出异常任务自动重置执行完成!    <=====", databaseType);
                } else {
                    log.info("=====>    {} 导出异常任务自动重置未执行1,无导出功能!    <=====", databaseType);
                }
            } catch (Exception e) {
                log.error("=====>    {} 异常导出任务自动重置任务状态未失败异常    <=====", databaseType);
                log.error(e.getMessage(), e);
            }
        }

        if (redisLockService != null) {
            redisLockService.unlock(lockKey);
        }
    }

}
