package com.biz.crm.business.common.base.aspect;

import com.biz.crm.business.common.base.aop.RequestControl;
import com.biz.crm.business.common.base.config.RequestControlProperties;
import lombok.RequiredArgsConstructor;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.DigestUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.UUID;
import java.util.concurrent.*;

/**
 * @Description
 * @Author lifei
 * @Date 2025/4/29 19:54
 **/
@Aspect
@Component
@RequiredArgsConstructor
public class RequestControlAspect {

    private final RequestControlProperties props;
    private final RedisTemplate<String,Object> redisTemplate;

    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(
            10, 50, 60, TimeUnit.SECONDS, new LinkedBlockingQueue<>()
    );

    @Around("@annotation(requestControl)")
    public Object around(ProceedingJoinPoint pjp, RequestControl requestControl) throws Throwable {
        String requestId = getRequestId(); // 从Header或生成唯一请求ID
        String limitKey = props.getLimit().getKeyPrefix() + requestId;

        // 限流控制（每个请求唯一标识最多 N 次/分钟）
        if (props.getLimit().isEnabled()) {
            Long count = redisTemplate.opsForValue().increment(limitKey);
            if (count != null && count == 1L) {
                redisTemplate.expire(limitKey, props.getLimit().getExpireSeconds(), TimeUnit.SECONDS);
            } else if (count != null && count > props.getLimit().getMaxCount()) {
                throw new RuntimeException("请求过于频繁");
            }
        }

        // 幂等性控制（请求只处理一次）
        String idempotentKey = "idem:" + requestId;
        Boolean locked = redisTemplate.opsForValue().setIfAbsent(idempotentKey, "1", props.getIdempotentExpireSeconds(), TimeUnit.SECONDS);
        if (Boolean.FALSE.equals(locked)) {
            throw new RuntimeException("重复请求，请勿重复提交");
        }

        // 超时控制
        Future<Object> future = executor.submit(() -> {
            try {
                return pjp.proceed();
            } catch (Throwable e) {
                throw new RuntimeException(e);
            }
        });
        try {
            return future.get(props.getTimeoutSeconds(), TimeUnit.SECONDS);
        } catch (TimeoutException e) {
            future.cancel(true);
            throw new TimeoutException("请求超时：" + props.getTimeoutSeconds() + " 秒");
        }
    }

    private String getRequestId() {
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.currentRequestAttributes()).getRequest();
        String token = request.getHeader("X-Request-ID");
        return StringUtils.hasText(token) ? token : DigestUtils.md5DigestAsHex(UUID.randomUUID().toString().getBytes());
    }
}
