package com.biz.crm.business.common.base.controller;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.common.ie.local.config.ImportExportProperties;
import com.biz.crm.common.ie.local.entity.ImportTask;
import com.biz.crm.common.ie.local.entity.ImportTemplateDetail;
import com.biz.crm.common.ie.local.repository.ImportTemplateDetailRepository;
import com.biz.crm.common.ie.local.service.ImportTaskService;
import com.biz.crm.common.ie.sdk.enums.ExcelFileTypeEnum;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelColumn;
import com.biz.crm.common.ie.sdk.excel.annotations.CrmExcelImport;
import com.biz.crm.common.ie.sdk.excel.process.ImportProcess;
import com.biz.crm.common.ie.sdk.excel.strategy.CrmExcelProcessStrategy;
import com.biz.crm.common.ie.sdk.vo.*;
import com.bizunited.nebula.venus.sdk.dto.Base64UploadDto;
import com.bizunited.nebula.venus.sdk.service.file.FileHandleService;
import com.bizunited.nebula.venus.sdk.vo.OrdinaryFileVo;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.Base64Utils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StopWatch;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Stream;

@Api(tags = {"导入任务: ImportTask: 导入任务"})
@RestController
@RequestMapping({"/v1/importTask/importTask"})
@Slf4j
public class ImportTaskDevelopmentController {

    @Autowired(required = false)
    private ImportTaskService importTaskService;

    @Autowired(required = false)
    private ImportExportProperties importExportProperties;

    @Autowired(required = false)
    private List<ImportProcess> importProcesses;

    @Autowired(required = false)
    private ImportTemplateDetailRepository importTemplateDetailRepository;

    @Autowired(required = false)
    private FileHandleService fileHandleService;

    @Autowired(required = false)
    private CrmExcelProcessStrategy crmExcelProcessStrategy;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Value("${spring.application.name:}")
    private String subsystem;

    /**
     * 需要venus 模块的文件上传下载支持
     *
     * @param taskCode 导入任务编码
     * @return
     */
    @ApiOperation("根据导入任务编码重新执行导入任务,开发专用,用于开发本地调试")
    @GetMapping({"/developmentDebugSpecialUse"})
    public Result<ImportTask> developmentDebugSpecialUse(@RequestParam("taskCode") String taskCode) {
        try {
            ImportTask task = this.importTaskService.findDetailByTaskCode(taskCode);
            IeExecutorVoForImport vo = new IeExecutorVoForImport();
            vo.setTaskCode(taskCode);
            vo.setApplicationName(task.getApplicationName());
            process(vo, task);
            return Result.ok(task);
        } catch (Exception var3) {
            log.error(var3.getMessage(), var3);
            return Result.error(var3.getMessage());
        }
    }

    private void process(IeExecutorVo vo, ImportTask task) {
        StopWatch sw = new StopWatch(CharSequenceUtil.format("导入任务执行{}", new Object[]{vo.getTaskCode()}));
        sw.start("1、任务预处理");
        ImportTemplateDetail templateDetail = this.importTemplateDetailRepository.findDetailByTemplateCode(task.getTemplateCode());
        Validate.notNull(templateDetail, "未配置导入的模板信息", new Object[0]);
        String templateFileCode = templateDetail.getFileCode();
        Validate.notEmpty(templateFileCode, "未配置导入的模板信息", new Object[0]);
        sw.stop();
        ImportProcess process = this.findImportProcess(task.getTemplateCode());
        CrmExcelImport crmExcelImport = (CrmExcelImport) process.findCrmExcelVoClass().getAnnotation(CrmExcelImport.class);
        Validate.notNull(crmExcelImport, "excel导入bean未配置CrmExcelImport注解", new Object[0]);
        Field[] fields = ReflectUtil.getFields(process.findCrmExcelVoClass());
        long count = Stream.of(fields).filter((a) -> Objects.nonNull(a.getAnnotation(CrmExcelColumn.class))).count();
        Validate.isTrue(count > 0L, "excel导入bean的字段属性未配置CrmExcelColumn注解", new Object[0]);
        sw.start("3、获取原文件+模板文件信息");
        Map<String, OrdinaryFileVo> downloadFileMap = Maps.newHashMap();
        OrdinaryFileVo ordinaryFileVo = this.fileHandleService.findById(task.getFileCode());
        if (ordinaryFileVo != null) {
            downloadFileMap.put(task.getFileCode(), ordinaryFileVo);
        }

        OrdinaryFileVo templateFileVo = this.fileHandleService.findById(templateFileCode);
        if (templateFileVo != null) {
            downloadFileMap.put(templateFileCode, templateFileVo);
        }

        boolean fileFlag = downloadFileMap.containsKey(task.getFileCode()) && (downloadFileMap.get(task.getFileCode())).getFileSize() > 0L;
        Validate.isTrue(fileFlag, CharSequenceUtil.format("未获取到任务{}对应的原文件信息", new Object[]{vo.getTaskCode()}), new Object[0]);
        boolean templateFileFlag = downloadFileMap.containsKey(templateFileCode) && ((OrdinaryFileVo) downloadFileMap.get(templateFileCode)).getFileSize() > 0L;
        Validate.isTrue(templateFileFlag, CharSequenceUtil.format("未获取到任务{}对应的模板文件信息", new Object[]{vo.getTaskCode()}), new Object[0]);
        sw.stop();
        sw.start("4、处理excel导入");
        ImportExcelLocalFile localExcelFileMap = this.createLocalExcelFile(downloadFileMap, task.getFileCode(), templateFileCode, task.getTaskCode());
        String parentPath = localExcelFileMap.getOrdinaryFile().getParentPath();
        try {
            this.excelHandler(task, process, localExcelFileMap);
        } catch (Exception var22) {
            log.error("导入出错了{},error[{}]", vo.getTaskCode(), Throwables.getStackTraceAsString(var22));
            Validate.isTrue(false, var22.getMessage(), new Object[0]);
        } finally {
            FileUtil.del(parentPath);
        }
        sw.stop();
        log.info(CharSequenceUtil.format("******导入任务执行{}，执行统计{}****", new Object[]{vo.getTaskCode(), sw.prettyPrint()}));
    }


    private ImportExcelLocalFile createLocalExcelFile(Map<String, OrdinaryFileVo> downloadFileMap, String fileCode, String templateFileCode, String taskCode) {
        ImportExcelLocalFile localFile = new ImportExcelLocalFile();
        if (downloadFileMap != null && downloadFileMap.containsKey(fileCode) && downloadFileMap.containsKey(templateFileCode)) {
            OrdinaryFileVo ordinaryFileVo = downloadFileMap.get(fileCode);
            OrdinaryFileVo templateFileVo = downloadFileMap.get(templateFileCode);
            Date now = new Date();
            String root = this.importExportProperties.getRoot();
            if (StringUtils.isBlank(root)) {
                root = FileUtil.getTmpDirPath();
            }

            if (root.endsWith("/")) {
                root = root.substring(0, root.length() - 1);
            }

            String parentPath = CharSequenceUtil.format("{}/import/{}/{}", new Object[]{root, DateUtil.format(now, "yyyy/MM-dd"), fileCode});
            String originalFileName = ordinaryFileVo.getOriginalFileName();
            File ordinaryFile = new File(parentPath, originalFileName);
            String templateFileName = originalFileName.substring(0, originalFileName.lastIndexOf("."));
            templateFileName = templateFileName + "_" + ExcelFileTypeEnum.TEMPLATE.getValue() + ".xlsx";
            File templateFile = new File(parentPath, templateFileName);
            String errorFileName = this.crmExcelProcessStrategy.getErrorExcelFileName(originalFileName);
            File errorFile = new File(parentPath, errorFileName);
            CountDownLatch latch = new CountDownLatch(2);
            byte[] bytes = this.fileHandleService.findContentByFilePathAndFileRename(ordinaryFileVo.getRelativeLocal(), ordinaryFileVo.getFileName());
            if (bytes != null) {
                byte[] finalBytes = bytes;
                Thread writeFileThread = new Thread(() -> {
                    File tmpFile = ordinaryFile;

                    try {
                        Thread.currentThread().setName("文件写入" + tmpFile);
                        FileUtil.writeBytes(finalBytes, tmpFile);
                        ExcelFileVo vo = this.convertExcelFileVo(tmpFile);
                        localFile.setOrdinaryFile(vo);
                    } catch (Exception var10) {
                        log.error("ERR230904-0101:导入文件写入异常:" + tmpFile, var10);
                    } finally {
                        latch.countDown();
                    }

                });
                writeFileThread.start();
                bytes = this.fileHandleService.findContentByFilePathAndFileRename(templateFileVo.getRelativeLocal(), templateFileVo.getFileName());
                if (bytes != null) {
                    byte[] finalBytes1 = bytes;
                    writeFileThread = new Thread(() -> {
                        File tmpFile = templateFile;

                        try {
                            Thread.currentThread().setName("文件写入" + tmpFile);
                            FileUtil.writeBytes(finalBytes1, tmpFile);
                            ExcelFileVo vo = this.convertExcelFileVo(tmpFile);
                            localFile.setTemplateFile(vo);
                        } catch (Exception var10) {
                            log.error("ERR230904-0201:模板文件写入异常:" + tmpFile, var10);
                        } finally {
                            latch.countDown();
                        }

                    });
                    writeFileThread.start();
                    ExcelFileVo errorFileExcelFileVo = this.convertExcelFileVo(errorFile);
                    localFile.setErrorFile(errorFileExcelFileVo);

                    try {
                        latch.await();
                    } catch (InterruptedException var17) {
                        log.error("写入本地文件发生异常{}", var17);
                        throw new RuntimeException("等待写入文件异常", var17);
                    }

                    Validate.isTrue(localFile.getOrdinaryFile() != null, "ERR230904-0301:导入文件下载失败 %s", new Object[]{taskCode});
                    Validate.isTrue(localFile.getTemplateFile() != null, "ERR230904-0302:导入模板下载失败 %s", new Object[]{taskCode});
                    return localFile;
                } else {
                    throw new RuntimeException("ERR230904-0202:下载模板文件异常:" + ordinaryFileVo);
                }
            } else {
                throw new RuntimeException("ERR230904-0102:下载导入文件异常:" + ordinaryFileVo);
            }
        } else {
            return localFile;
        }
    }

    private ImportProcess findImportProcess(String code) {
        if (!CollectionUtils.isEmpty(this.importProcesses) && !StringUtils.isBlank(code)) {
            Optional<ImportProcess> op = this.importProcesses.stream().filter((item) -> item.getTemplateCode().equals(code)).findFirst();
            Validate.notNull(op, CharSequenceUtil.format("未获取到模板编码{}对应的处理器信息", new Object[]{code}), new Object[0]);
            return op.orElse(null);
        } else {
            return null;
        }
    }


    private void excelHandler(ImportTask task, ImportProcess process, ImportExcelLocalFile localFile) {
        if (localFile == null || localFile.getOrdinaryFile() == null || localFile.getTemplateFile() == null) {
            Validate.isTrue(false, "任务文件加载失败", new Object[0]);
        }
        String taskCode = task.getTaskCode();
        StopWatch sw = new StopWatch(CharSequenceUtil.format("====导入任务执行{}-excel处理===", new Object[]{taskCode}));
        Map<String, Object> params = this.findParamsMap(task);
        TaskGlobalParamsVo paramsVo = this.findParams(task);
        sw.start("excel逻辑处理");
        ImportTaskHandlerResultVo resultVo = this.crmExcelProcessStrategy.importExcel(process, paramsVo, params, localFile);
        sw.stop();
        log.info(CharSequenceUtil.format("+++++导入任务执行{}-excel处理，执行统计{}++++", new Object[]{taskCode, sw.prettyPrint()}));
    }


    private Map<String, Object> findParamsMap(ImportTask importTask) {
        if (!Objects.isNull(importTask) && !StringUtils.isBlank(importTask.getParametersJson())) {
            Map<String, Object> map = Maps.newHashMap();
            JSONObject jsonObject = JSONUtil.parseObj(importTask.getParametersJson());
            Set<String> set = jsonObject.keySet();

            for (String item : set) {
                map.put(item, jsonObject.get(item));
            }

            return map;
        } else {
            return Maps.newHashMap();
        }
    }


    private TaskGlobalParamsVo findParams(ImportTask task) {
        Validate.notNull(task, "导入任务信息不存在", new Object[0]);
        TaskGlobalParamsVo vo = new TaskGlobalParamsVo();
        vo.setAppCode(task.getAppCode());
        vo.setApplicationName(task.getApplicationName());
        vo.setCreateAccount(task.getCreateAccount());
        vo.setCreateAccountName(task.getCreateName());
        vo.setTaskCreateTime(task.getCreateTime());
        vo.setTenantCode(task.getTenantCode());
        vo.setTaskCode(task.getTaskCode());
        return vo;
    }

    private List<OrdinaryFileVo> venusFileUpload(MultipartFile multipartFile) throws IOException {
        Base64UploadDto base64UploadDto = new Base64UploadDto();
        base64UploadDto.setCreator(this.loginUserService.findCurrentAccount());
        base64UploadDto.setFileNanmes(new String[]{multipartFile.getOriginalFilename()});
        base64UploadDto.setBase64Contents(new String[]{Base64Utils.encodeToString(multipartFile.getBytes())});
        List<OrdinaryFileVo> ordinaryFileVoList = this.fileHandleService.fileUpload(this.subsystem.replace("-", "_"), base64UploadDto, (String) null);
        Validate.notEmpty(ordinaryFileVoList, "文件上传失败！", new Object[0]);
        return ordinaryFileVoList;
    }

    private ExcelFileVo convertExcelFileVo(File file) {
        ExcelFileVo vo = new ExcelFileVo();
        vo.setFileName(file.getName());
        vo.setParentPath(file.getParent());

        try {
            vo.setPath(file.getCanonicalPath());
        } catch (IOException var4) {
            vo.setPath(file.getPath());
        }

        return vo;
    }

}
