package com.biz.crm.business.common.base.service;

import com.biz.crm.business.common.base.constant.RedisConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.Validate;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;


/**
 * @describe 加锁工具
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.13 18:07
 */
@Slf4j
@Component
public class RedisLockService {

    @Autowired(required = false)
    private RedissonClient redissonClient;

    /**
     * 默认锁等待时间
     */
    private final static int THREE = 3;
    /**
     * 默认过期时间  5分钟
     */
    private final static int FIVE = 5;

    /**
     * 默认过期等待时间5秒,加锁3分钟
     *
     * @param key
     */
    public void lock(String key) {
        Validate.notBlank(key, "RedisLockService key is null!!");
        boolean result = tryLock(key, TimeUnit.MINUTES, FIVE);
        Validate.isTrue(result, "操作人员过多,加锁[" + key + "]失败,请稍后再试!");
    }

    /**
     * @param key
     * @param unit     超时时间单位
     * @param lockTime 超时时间
     * @return
     */
    public void lock(String key, TimeUnit unit, long lockTime) {
        Validate.notBlank(key, "RedisLockService key is null!!");
        boolean result = tryLock(key, unit, lockTime);
        Validate.isTrue(result, "操作人员过多,加锁[" + key + "]失败,请稍后再试!");
    }


    /**
     * @param key
     * @param unit     超时时间单位
     * @param lockTime 超时时间
     * @param waitTime 等待时间/秒
     * @return
     */
    public void lock(String key, TimeUnit unit, long lockTime, long waitTime) {
        Validate.notBlank(key, "RedisLockService key is null!!");
        boolean result = tryLock(key, unit, lockTime, waitTime);
        Validate.isTrue(result, "操作人员过多,加锁[" + key + "]失败,请稍后再试!");
    }

    /**
     * 加锁  默认等待3秒
     *
     * @param key
     * @param unit     超时时间单位  日时分秒
     * @param lockTime 超时时间
     * @return
     */
    public boolean tryLock(String key, TimeUnit unit, long lockTime) {
        return tryLock(key, unit, lockTime, 0);
    }

    /**
     * @param key
     * @param unit     超时时间单位
     * @param lockTime 超时时间
     * @param waitTime 等待时间/秒
     * @return
     */
    public boolean tryLock(String key, TimeUnit unit, long lockTime, long waitTime) {
        Validate.notBlank(key, "RedisLockService key must not be empty!!");
        Validate.notNull(unit, "RedisLockService unit must not be empty!!");
        Validate.isTrue(lockTime > 0, "RedisLockService lockTime must be greater than 0!!");
        Validate.isTrue(waitTime >= 0, "RedisLockService waitTime must be greater than or equal to 0!!");
        boolean result = false;
        //秒转毫秒
        waitTime = waitTime * 1000;
        RLock rLock = this.createLock(key);
        try {
            result = rLock.tryLock(waitTime, lockTime, unit);
        } catch (InterruptedException e) {
            log.error("释放锁失败，锁：{}", key);
            log.error(e.getMessage(), e);
        }
        return result;

    }


    /**
     * 初始化一个分布式锁
     *
     * @param key 锁的关键字（在redis中一定独一无二的）
     * @return
     */
    protected RLock createLock(String key) {
        return redissonClient.getLock(key);
    }

    /**
     * 解锁
     *
     * @param key
     */
    public void unlock(String key) {
        Validate.notBlank(key, "RedisLockService key is null!!");
        String name = "";
        try {
            RLock lock = createLock(key);
            if (null == lock) {
                return;
            }
            name = lock.getName();
            if (lock.isLocked()) {
                if (lock.isHeldByCurrentThread()) {
                    lock.unlock();
                }
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            log.error("释放锁失败，锁：{} {}", key, name);
            log.error(e.getMessage(), e);
        }
    }

    /**
     * 验证是否加锁
     *
     * @param key
     * @return
     */
    public boolean isLock(String key) {
        Validate.notBlank(key, "RedisLockService key is null!!");
        RLock rlock = this.redissonClient.getLock(key);
        return rlock.isLocked();
    }

    /**
     * 验证是否加锁
     *
     * @return 有锁返回true，没锁返回false
     */
    public boolean isBatchLock(String prefix, List<String> keyList) {
        Validate.notBlank(prefix, "RedisLockService key prefix is null!!");
        for (String key : keyList) {
            //只要有一个是锁定的，就返回存在加锁
            if (isLock(prefix + key)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 批量加锁
     *
     * @param prefix   锁前缀，例：tpm:month_budget:lock:
     * @param keyList  keyList
     * @param timeUnit 时间单位，不传默认秒
     * @param time     时间，传非正数默认20
     * @return 成功标记
     */
    public boolean batchLock(String prefix, List<String> keyList, TimeUnit timeUnit, int time) {
        Validate.notNull(prefix, "加锁失败，锁前缀不能为空");
        if (CollectionUtils.isEmpty(keyList)) {
            throw new RuntimeException("加锁失败，key集合不能为空");
        }
        if (ObjectUtils.isEmpty(timeUnit)) {
            timeUnit = TimeUnit.SECONDS;
        }
        if (time <= 0) {
            time = RedisConstant.DEFAULT_LOCK_TIME;
        }
        boolean isLock = true;
        List<String> successKeys = new ArrayList<>();
        try {
            // 循环加锁，并记录成功的key
            for (String key : keyList) {
                isLock = this.tryLock(prefix + key, timeUnit, time);
                if (!isLock) {
                    return false;
                }
                successKeys.add(key);
            }
        } finally {
            // 存在加锁失败的情况，则先将成功的解锁
            if (!isLock && !CollectionUtils.isEmpty(successKeys)) {
                this.batchUnLock(prefix, successKeys);
            }
        }
        return true;
    }

    /**
     * 批量解锁
     *
     * @param prefix  锁前缀，例：tpm:month_budget:lock:
     * @param keyList keyList
     */
    public void batchUnLock(String prefix, List<String> keyList) {
        if (CollectionUtils.isEmpty(keyList)) {
            throw new RuntimeException("解锁失败，key集合不能为空");
        }
        keyList.forEach(key -> this.unlock(prefix + key));
    }

}
