package com.biz.crm.business.common.rocketmq.util;

import com.biz.crm.business.common.rocketmq.listener.GeneralMessageListener;
import com.biz.crm.business.common.rocketmq.listener.OrderMessageListener;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.spring.annotation.ConsumeMode;
import org.apache.rocketmq.spring.annotation.MessageModel;
import org.apache.rocketmq.spring.annotation.RocketMQMessageListener;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.util.Assert;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 阿里MQ转发开源rocketMq,阿里MQ当前不支持consumerGroup概念
 * 阿里用 groupId 实现consumerGroup概念,此方法为支持**项目
 * 其他项目有特殊需求,需重写此方法,并需重写
 * {@link GeneralMessageListener} 和{@link OrderMessageListener}
 * 阿里消费者(订阅关系)帮助文档  https://help.aliyun.com/document_detail/43523.html
 * <p>
 * 当你看到这里,我就不得不哔哔两句.
 * 当前有一句***,不知当讲不当讲,cao.
 * 当前有一句***,不知当讲不当讲,cao.
 * 当前有一句***,不知当讲不当讲,cao.
 *
 * @describe 阿里MQ
 * @author 洗洗睡吧, 都几点了
 * @version v1.0.0
 * @date 2012/12/11 23:59
 */
@Slf4j
public class AlibabaMqTopicAndTagUtil {

    /**
     * 是否已经构建
     */
    private static boolean hasBuild = false;
    /**
     * 监听  topic => tag 方法 set  普通|集群|无序
     */
    private static final Map<String, Map<String, Set<String>>> TOPIC_MAP = new HashMap<>(4);

    /**
     * 监听  topic => tag 方法 set  顺序
     */
    private static final Map<String, Map<String, Set<String>>> TOPIC_ORDER_MAP = new HashMap<>(4);

    /**
     * 获取tag 集合
     * 过个用 \\  隔开
     * order  为true 获取顺序tag
     * order  为false 获取集群tag
     * <p>
     * 此方法会映射本地和阿里广播消费者关系 但广播的消费者需要自行写并启动
     *
     * @return
     */
    private static void builderMap() {
        ApplicationContext applicationContext = ApplicationContextUtils.getContext();
        Environment env = applicationContext.getEnvironment();
        //阿里和本地MQ tag都是用 ||  做的拆分
        String keyCharacter = "\\|\\|";
        String[] serviceNames = applicationContext.getBeanNamesForAnnotation(RocketMQMessageListener.class);
        for (String serviceName : serviceNames) {
            Object bean = applicationContext.getBean(serviceName);
            Class<?> forName = bean.getClass();
            RocketMQMessageListener mqMessageListener = forName.getAnnotation(RocketMQMessageListener.class);
            Assert.hasText(mqMessageListener.topic(), "topic不能为空!");
            StringBuilder topicBuilder = new StringBuilder();
            //加载配置参数
            for (String s : mqMessageListener.topic().split("\\$")) {
                if (StringUtils.isEmpty(s)) {
                    continue;
                }
                if (s.contains("{")) {
                    s = s.replaceAll("\\{", "");
                    s = s.replaceAll("\\}", "");
                    topicBuilder.append(StringUtils.stripToEmpty(env.getProperty(s)));
                } else {
                    topicBuilder.append(s);
                }
            }
            String topic = topicBuilder.toString();
            Assert.hasText(topic, "topic不能为空!");
            //广播都为集群模式  消费者错误将无法消费
            for (String tag : mqMessageListener.selectorExpression().split(keyCharacter)) {
                MessageModel messageModel = mqMessageListener.messageModel();
                if (ConsumeMode.ORDERLY.equals(mqMessageListener.consumeMode())) {
                    if (MessageModel.CLUSTERING.equals(messageModel)) {
                        buildMap(topic, tag, serviceName, TOPIC_ORDER_MAP);
                    } else if (MessageModel.BROADCASTING.equals(messageModel)) {
                        throw new RuntimeException("不支持广播,请用消费组概念实现!");
                    } else {
                        throw new RuntimeException("MQ监听MessageModel错误");
                    }
                } else if (ConsumeMode.CONCURRENTLY.equals(mqMessageListener.consumeMode())) {
                    if (MessageModel.CLUSTERING.equals(messageModel)) {
                        buildMap(topic, tag, serviceName, TOPIC_MAP);
                    } else if (MessageModel.BROADCASTING.equals(messageModel)) {
                        throw new RuntimeException("不支持广播,请用消费组概念实现!");
                    } else {
                        throw new RuntimeException("MQ监听MessageModel错误");
                    }
                } else {
                    throw new RuntimeException("MQ监听ConsumeMode错误");
                }
            }
        }
        log.info("=====>    阿里MQ消费有序监听[{}]    <=====", TOPIC_ORDER_MAP);
        log.info("=====>    阿里MQ消费集群监听[{}]    <=====", TOPIC_MAP);
        hasBuild = true;
    }

    /**
     * 构建MAP
     *
     * @param topic
     * @param tag
     * @param serviceName
     * @param topicMap
     */
    private static void buildMap(String topic, String tag, String serviceName, Map<String, Map<String, Set<String>>> topicMap) {
        if (StringUtils.isEmpty(topic)
                || StringUtils.isEmpty(tag)
                || StringUtils.isEmpty(serviceName)) {
            return;
        }
        Map<String, Set<String>> tagMap = topicMap.getOrDefault(topic, Maps.newHashMap());
        Set<String> serviceSet = tagMap.getOrDefault(tag, Sets.newHashSet());
        serviceSet.add(serviceName);
        tagMap.put(tag, serviceSet);
        topicMap.put(topic, tagMap);
    }

    public static Map<String, Map<String, Set<String>>> getTopicMap() {
        if (!hasBuild) {
            builderMap();
        }
        return TOPIC_MAP;
    }

    public static Map<String, Map<String, Set<String>>> getTopicOrderMap() {
        if (!hasBuild) {
            builderMap();
        }
        return TOPIC_ORDER_MAP;
    }

}
