package com.biz.crm.business.common.rocketmq.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.aliyun.openservices.ons.api.Action;
import com.aliyun.openservices.ons.api.ConsumeContext;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.MessageListener;
import com.aliyun.openservices.ons.api.order.ConsumeOrderContext;
import com.aliyun.openservices.ons.api.order.MessageOrderListener;
import com.aliyun.openservices.ons.api.order.OrderAction;
import com.aliyun.openservices.shade.org.apache.commons.lang3.StringUtils;
import com.biz.crm.business.common.rocketmq.repository.RocketMqMessageLogDocumentRepository;
import com.biz.crm.business.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.business.common.rocketmq.vo.MqUserDetailVo;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.rocketmq.constant.MqConstant;
import com.biz.crm.business.common.rocketmq.document.RocketMqMessageLogDocument;
import com.biz.crm.business.common.rocketmq.util.RocketMqUtil;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.elasticsearch.client.transport.NoNodeAvailableException;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;


/**
 * @author luoqi
 * @date 2020-10-30 11:33
 * @version V1.0
 * @Description:
 */
@Slf4j
public abstract class AbstractRocketMqConsumer implements RocketMQListener<MqMessageVo>
        , MessageListener, MessageOrderListener {

    @Autowired(required = false)
    private RocketMqMessageLogDocumentRepository rocketMqMessageLogDocumentRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private RedisMutexService redisMutexService;

    @Autowired(required = false)
    private MqExceptionExtendService mqExceptionExtendService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    @Override
    public void onMessage(MqMessageVo message) {
        log.info("接收到的MQ消息内容{}", message);
        if (null == message) {
            log.error("消费MQ消息MqMessageVo为空，忽略本次消费");
            throw new IllegalArgumentException("消费MQ消息MqMessageVo为空，忽略本次消费");
        }

        Exception handleException = null;
        String json = JSON.toJSONString(message);
        String messageId = message.getId();
        //丢失日志id
        if (StringUtils.isEmpty(messageId)) {
            log.error("消费MQ消息记录日志失败：message = {}", json);
            return;
        }
        //重复消费验证
        String repeatKey = MqConstant.MQ_REPEAT + messageId;
        if (!message.isRepeatConsumer()
                && redisService.hasKey(repeatKey)) {
            log.error("MQ消息内容已被成功消费,不可被重复成功消费：message = {}", json);
            return;
        }
        Date beginDate = new Date();
        String callbackBegin = RocketMqUtil.DATETIME_FORMAT.format(beginDate);
        String endStatus = EnableStatusEnum.DISABLE.getCode();
        String callbackLog = "消息内容为空跳过业务处理!";
        // 调用用户实现的 handleMessage 消费消息
        this.setUserInfo(message);
        try {
            if (StringUtils.isNotEmpty(message.getMsgBody())) {
                Object result = this.handleMessage(message);
                callbackLog = JSON.toJSONString(result);
            }
            endStatus = EnableStatusEnum.ENABLE.getCode();
        } catch (Exception e) {
            log.error("消费MQ消息失败：message = {} >>>", json, e);
            callbackLog = ExceptionUtils.getStackTrace(e);
            handleException = e;
        }
        /**
         * 保存MQ消费日志
         */
        this.saveMqConsumerLog(message, callbackBegin, callbackLog, endStatus);
        Date endDate = new Date();
        log.info("MQ消息[{}] 开始消费时间[{}],结束消费时间[{}],耗时[{}]", messageId, callbackBegin,
                RocketMqUtil.DATETIME_FORMAT.format(endDate),
                DateUtil.millisecondToStrVague(endDate.getTime() - beginDate.getTime()));
        //抛出异常
        if (null != handleException) {
            throw new IllegalArgumentException("消费MQ消息失败：message = " + json, handleException);
        }
        //消费成功标记.redis保存8小时
        redisMutexService.getAndIncrement(MqConstant.MQ_REPEAT + messageId, 0, RocketMqUtil.getLockHour(), TimeUnit.HOURS);
    }

    /**
     * 设置用户信息
     *
     * @param message
     */
    private void setUserInfo(MqMessageVo message) {

        String currentAccount = message.getCurrentAccount();
        String accountJson = message.getAccountJson();
        if (StringUtils.isNotEmpty(accountJson)) {
            MqUserDetailVo userIdentity = JSON.parseObject(accountJson, MqUserDetailVo.class);
            loginUserService.refreshAuthentication(userIdentity);
        }

    }

    /**
     * 保存MQ消费日志
     *
     * @param message       mq消息内容
     * @param callbackBegin 开发处理消息时间
     * @param callbackLog   mq消费后返回信息
     * @param endStatus     结束标记
     */
    private void saveMqConsumerLog(MqMessageVo message, String callbackBegin,
                                   String callbackLog, String endStatus) {
        if (message == null
                || !RocketMqUtil.isSaveLog()) {
            return;
        }
        String messageId = message.getId();
        String json = JSON.toJSONString(message);
        String redisKey = MqConstant.MQ_MESSAGE + messageId;
        Object redisObj = redisService.get(redisKey);
        RocketMqMessageLogDocument logEntity = null;
        LocalDateTime localDateTime = LocalDateTime.now();
        if (redisObj != null) {
            if (redisObj instanceof RocketMqMessageLogDocument) {
                logEntity = (RocketMqMessageLogDocument) redisObj;
            } else {
                logEntity = JSONObject.parseObject(JSON.toJSONString(redisObj), RocketMqMessageLogDocument.class);
            }
        } else {
            Optional<RocketMqMessageLogDocument> logEsOptional = rocketMqMessageLogDocumentRepository.findById(messageId);
            if (logEsOptional.isPresent()) {
                logEntity = logEsOptional.get();
            } else {
                logEntity = RocketMqMessageLogDocument.buildLogVo(message);
                logEntity.setEndStatus(EnableStatusEnum.ENABLE.getCode());
            }
        }
        if (logEntity == null) {
            logEntity = RocketMqMessageLogDocument.buildLogVo(message);
            logEntity.setRemarks("未获取到MQ发送日志记录,消费时,重新构建");
        }
        if (StringUtils.isEmpty(logEntity.getTopic())) {
            logEntity.setTopic(RocketMqUtil.getTopic());
        }
        logEntity.setCallbackBegin(callbackBegin);
        logEntity.setUpdateDate(localDateTime.format(RocketMqUtil.YYYY_MM_DD));
        logEntity.setUpdateDateSecond(localDateTime.format(RocketMqUtil.HH_MM_SS));
        logEntity.setUpdateDateAll(localDateTime.format(RocketMqUtil.YYYY_MM_DD_HH_MM_SS));
        logEntity.setCallbackLog(callbackLog);
        logEntity.setEndStatus(endStatus);
        //回写日志到数据行
        try {
            logEntity.setCallbackEnd(localDateTime.format(RocketMqUtil.YYYY_MM_DD_HH_MM_SS));
            if (Objects.nonNull(logEntity.getCreateDateSort())) {
                logEntity.setConsumeTime(System.currentTimeMillis() - logEntity.getCreateDateSort());
            }
            logEntity.setId(messageId);
            rocketMqMessageLogDocumentRepository.save(logEntity);
            redisService.del(redisKey);
        } catch (IllegalStateException | NoNodeAvailableException e) {
            log.error("message = {} >>>", json, e);
            log.error("消费MQ消息，记录日志失败,elasticsearch异常!\n[{}]", RocketMqUtil.buildErrorInfo());
            if (mqExceptionExtendService != null) {
                mqExceptionExtendService.consumerMqException(e);
            }
        } catch (Exception e) {
            log.error("message = {} >>>", json, e);
            log.error("消费MQ消息，记录日志失败,elasticsearch异常!");
            if (mqExceptionExtendService != null) {
                mqExceptionExtendService.consumerMqException(e);
            }
        }
    }


    @Override
    public Action consume(Message message, ConsumeContext context) {
        if (null == message) {
            log.error("普通消费MQ消息：MqMessageVo 为空，忽略本次消费");
            throw new IllegalArgumentException("消费MQ消息：MqMessageVo 为空，忽略本次消费");
        }
        if (message.getBody() == null) {
            log.error("普通消费MQ消息：MqMessageVo 为空，忽略本次消费");
            throw new IllegalArgumentException("顺序消费MQ消息：MqMessageVo 为空，忽略本次消费");
        }
        MqMessageVo mqMessageBody = JSONObject.parseObject(message.getBody(), MqMessageVo.class);
        this.onMessage(mqMessageBody);
        return Action.CommitMessage;
    }

    @Override
    public OrderAction consume(Message message, ConsumeOrderContext context) {
        if (null == message) {
            log.error("顺序消费MQ消息：MqMessageVo 为空，忽略本次消费");
            throw new IllegalArgumentException("消费MQ消息：MqMessageVo 为空，忽略本次消费");
        }
        if (message.getBody() == null) {
            log.error("顺序消费MQ消息：MqMessageVo 为空，忽略本次消费");
            throw new IllegalArgumentException("顺序消费MQ消息：MqMessageVo 为空，忽略本次消费");
        }
        MqMessageVo mqMessageBody = JSONObject.parseObject(message.getBody(), MqMessageVo.class);
        this.onMessage(mqMessageBody);
        return OrderAction.Success;
    }

    /**
     * 消费消息
     *
     * @param message
     * @return
     */
    protected abstract Object handleMessage(MqMessageVo message);
}
