package com.biz.crm.business.common.rocketmq.util;

import com.aliyun.openservices.shade.org.apache.commons.lang3.StringUtils;
import com.biz.crm.business.common.rocketmq.config.RocketMqConfig;
import com.biz.crm.business.common.rocketmq.document.RocketMqMessageLogDocument;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.net.InetAddress;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Objects;

/**
 * @describe rocketmq 工具类
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.13 19:44
 */
@Slf4j
@Component
public class RocketMqUtil {


    public static final DateTimeFormatter YYYY_MM_DD = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter HH_MM_SS = DateTimeFormatter.ofPattern("HH:mm:ss");
    public static final DateTimeFormatter YYYY_MM_DD_HH_MM_SS = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    public static final SimpleDateFormat DATETIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
    private static RocketMqConfig rocketMqConfig;

    /**
     * 默认有序队列前缀
     */
    public static final String ORDER = "ORDER_";

    /**
     * 默认环境
     */
    private static String ENVIRONMENT = "_DEF";

    /**
     * 默认topic
     */
    private static String TOPIC = "TOPIC_DEF";

    /**
     * 是否保存MQ日志
     */
    private static Boolean IS_SAVE_LOG = true;

    /**
     * MQ消息消费成功后,锁定时间防止被重复消费
     */
    private static Integer LOCK_HOUR = 8;

    @Autowired
    public void setRocketMqConfig(RocketMqConfig feign) {
        RocketMqUtil.rocketMqConfig = feign;
        IS_SAVE_LOG = Objects.isNull(rocketMqConfig.getIsSaveLog()) ? true : rocketMqConfig.getIsSaveLog();
        log.info("=====>    rocketMq发送和消费记录[{}]保存ES日志    <=====", IS_SAVE_LOG ? "会" : "不会");

        LOCK_HOUR = Objects.isNull(rocketMqConfig.getLockHour()) ? 8 : rocketMqConfig.getLockHour();
        log.info("=====>    rocketMq消费成功锁定[{}]小时    <=====", LOCK_HOUR);

        TOPIC = StringUtils.isEmpty(rocketMqConfig.getTopic()) ? "TOPIC_DEF" : rocketMqConfig.getTopic();
        log.info("=====>    rocketMq 默认topic前缀[{}]    <=====", TOPIC);

        ENVIRONMENT = StringUtils.isEmpty(rocketMqConfig.getEnvironment()) ? "_DEF" : rocketMqConfig.getEnvironment();
        log.info("=====>    rocketMq 环境[{}];拼接在默认topic前缀的尾部    <=====", ENVIRONMENT);

        log.info("=====>    rocketMq 默认实际topic[{}]    <=====", TOPIC + ENVIRONMENT);

    }

    /**
     * mq环境
     *
     * @return String
     */
    public static String mqEnvironment() {
        return ENVIRONMENT;
    }

    /**
     * MQ消息消费成功后,锁定时间防止被重复消费
     *
     * @return String
     */
    public static Integer getLockHour() {
        return LOCK_HOUR;
    }

    /**
     * 是否保存日志
     *
     * @return String
     */
    public static boolean isSaveLog() {
        return IS_SAVE_LOG;
    }


    /**
     * 获取无序topic
     *
     * @return String
     */
    public static String getTopic() {
        return TOPIC + ENVIRONMENT;
    }

    /**
     * 获取有序topic
     *
     * @return String
     */
    public static String getOrderTopic() {
        return ORDER + getTopic();
    }

    /**
     * 检查待发送的MQ信息是否完整
     */
    public static void checkSendMsg(RocketMqMessageLogDocument logDocument) {
        Assert.hasText(logDocument.getTopic(), "MQ的topic不能为空!");
        Assert.hasText(logDocument.getTag(), "MQ的tag不能为空!");
        Assert.hasText(logDocument.getMsgBody(), "MQ发送内容不能为空!");
    }

    /**
     * 构建错误日志
     *
     * @return
     */
    public static String buildErrorInfo() {
        return buildErrorInfo("");
    }

    /**
     * 构建错误日志
     *
     * @return
     */
    public static String buildErrorInfo(String title) {
        StringBuilder msgBuffer = new StringBuilder();
        if (StringUtils.isEmpty(title)) {
            title = "系统异常:";
        }
        msgBuffer.append(title);
        if (StringUtils.isNotEmpty(rocketMqConfig.getApplicationName())) {
            msgBuffer.append("模块[").append(rocketMqConfig.getApplicationName()).append("]=>");
        }
        if (StringUtils.isNotEmpty(rocketMqConfig.getProfilesActive())) {
            msgBuffer.append("环境[").append(rocketMqConfig.getProfilesActive()).append("]=>");
        }
        if (StringUtils.isNotEmpty(rocketMqConfig.getServerAddr())) {
            msgBuffer.append("serverAddr[").append(rocketMqConfig.getServerAddr()).append("]=>");
        }
        if (StringUtils.isNotEmpty(rocketMqConfig.getNameServer())) {
            msgBuffer.append("namespace[").append(rocketMqConfig.getNameServer()).append("]=>");
        }
        try {
            String ip = InetAddress.getLocalHost().getHostAddress();
            msgBuffer.append("IP[").append(ip).append("]=>");
        } catch (Exception ipExp) {
            log.error(ipExp.getMessage(), ipExp);
        }
        msgBuffer.append("连接ES出错，请尽快确认处理!");
        msgBuffer.append(LocalDateTime.now().format(YYYY_MM_DD_HH_MM_SS));
        return msgBuffer.toString();
    }


    /**
     * rocketMq
     * 1、自建和华为云MQ使用 根据延迟的时间[秒],获取延迟级别, 只支持固定级别的延迟
     * 2、阿里MQ用的是绝对时间
     *
     * @author huxmld
     * @version v1.0.0
     * @date 2023-08-13 03:15
     */
    public static int getDelayTimeLevelByDelaySecond(long delaySecond) {
        /**
         * 1s   1s   1
         * 5s   5s   2
         * 10s  10s  3
         * 30s  30s  4
         * 1m   60s  5
         * 2m   120s 6
         * 3m   180s 7
         * 4m   240s 8
         * 5m   300s 9
         * 6m   360s 10
         * 7m   420s 11
         * 8m   480s 12
         * 9m   540s 13
         * 10m  600s 14
         * 20m  1200s 15
         * 30m  1800s 16
         * 1h   3600s 17
         * 2h   7200s 18
         */
        if (delaySecond < 1) {
            return 0;
        } else if (delaySecond == 1) {
            return 1;
        } else if (delaySecond <= 5) {
            return 2;
        } else if (delaySecond <= 10) {
            return 3;
        } else if (delaySecond <= 30) {
            return 4;
        } else if (delaySecond <= 60) {
            return 5;
        } else if (delaySecond <= 120) {
            return 6;
        } else if (delaySecond <= 180) {
            return 7;
        } else if (delaySecond <= 240) {
            return 8;
        } else if (delaySecond <= 300) {
            return 9;
        } else if (delaySecond <= 360) {
            return 10;
        } else if (delaySecond <= 420) {
            return 11;
        } else if (delaySecond <= 480) {
            return 12;
        } else if (delaySecond <= 540) {
            return 13;
        } else if (delaySecond <= 600) {
            return 14;
        } else if (delaySecond <= 1200) {
            return 15;
        } else if (delaySecond <= 1800) {
            return 16;
        } else if (delaySecond <= 3600) {
            return 17;
        } else {
            return 18;
        }
    }

}
