package com.biz.crm.business.common.rocketmq.service;

import com.alibaba.fastjson.JSON;
import com.biz.crm.business.common.rocketmq.constant.MqConstant;
import com.biz.crm.business.common.rocketmq.document.RocketMqMessageLogDocument;
import com.biz.crm.business.common.rocketmq.enums.RocketMqTypeEnum;
import com.biz.crm.business.common.rocketmq.event.RocketMqProducerEvent;
import com.biz.crm.business.common.rocketmq.repository.RocketMqMessageLogDocumentRepository;
import com.biz.crm.business.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.business.common.rocketmq.vo.MqMessageVo;
import com.biz.crm.business.common.sdk.enums.EnableStatusEnum;
import com.biz.crm.business.common.sdk.model.AbstractCrmUserIdentity;
import com.biz.crm.business.common.sdk.service.LoginUserService;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.elasticsearch.client.transport.NoNodeAvailableException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import org.springframework.transaction.event.TransactionPhase;
import org.springframework.transaction.event.TransactionalEventListener;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * @describe MQ 发送工具
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.13 18:07
 */
@Slf4j
@Component
public class RocketMqProducer {


    @Resource
    private MqCommonService mqCommonService;

    @Autowired(required = false)
    private RocketMqMessageLogDocumentRepository rocketMqMessageLogDocumentRepository;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private MqExceptionExtendService mqExceptionExtendService;

    @Autowired(required = false)
    private BuildMqInfoExtendService buildMqInfoExtendService;

    @Autowired(required = false)
    private LoginUserService loginUserService;

    /**
     * 默认大小
     */
    protected final Integer DEF_SIZE = 200;

    /**
     * 获取当前MQ类型
     *
     * @return RocketMqTypeEnum
     */
    public RocketMqTypeEnum getCurrentMqType() {
        return mqCommonService.getMqTypeEnum();
    }

    /**
     * 发送MQ消息
     * @param mqMessageVo
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:19
     */
    /**
     * @author huxmld
     * @version v1.0.0
     * @date 2023-08-13 03:38
     */
    public void sendMqMsg(MqMessageVo mqMessageVo) {
        sendMqMsg(mqMessageVo, 0);
    }

    /**
     * 发送MQ消息
     *
     * @param mqMessageVo
     * @param deliverSecond 延迟时间/秒  [区分自建MQ\华为MQ\阿里MQ]
     *                      自建MQ和华为云MQ  1s 5s 10s 30s 1m 2m 3m 4m 5m 6m 7m 8m 9m 10m 20m 30m 1h 2h
     *                      阿里MQ  延迟|定时3s投递, 设置为: System.currentTimeMillis() + 3000;
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:19
     */
    public void sendMqMsg(MqMessageVo mqMessageVo, long deliverSecond) {
        if (!StringUtils.hasText(mqMessageVo.getTopic())) {
            mqMessageVo.setTopic(RocketMqUtil.getTopic());
        }
        if (!mqMessageVo.getTopic().contains(RocketMqUtil.mqEnvironment())) {
            mqMessageVo.setTopic(mqMessageVo.getTopic() + RocketMqUtil.mqEnvironment());
        }
        RocketMqMessageLogDocument logVo = this.buildMqLog(mqMessageVo);
        log.info("发送的MQ消息内容[{}]", JSON.toJSONString(mqMessageVo));
        try {
            mqCommonService.sendMqMsg(logVo, mqMessageVo, deliverSecond);
        } catch (IllegalArgumentException e) {
            log.error("MQ消息发送失败[{}]", JSON.toJSONString(mqMessageVo));
            log.error(e.getMessage(), e);
            this.sendMqExceptionSaveLog(logVo, mqMessageVo, e);
            throw new IllegalArgumentException(e.getMessage());
        } catch (Exception e) {
            log.error("MQ消息发送失败[{}]", JSON.toJSONString(mqMessageVo));
            log.error(e.getMessage(), e);
            this.sendMqExceptionSaveLog(logVo, mqMessageVo, e);
            throw new IllegalArgumentException("MQ消息发送失败,消息ID[" + logVo.getId() + "]");
        }
    }

    /**
     * 发送MQ顺序消息
     *
     * @param mqMessageVo
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:19
     */
    public void sendMqOrderMsg(MqMessageVo mqMessageVo) {
        Assert.hasText(mqMessageVo.getTag(), "MQ的tag不能为空!");
        this.sendMqOrderMsg(mqMessageVo, mqMessageVo.getTag(), 0);
    }

    /**
     * 发送MQ顺序消息
     *
     * @param mqMessageVo
     * @param deliverSecond 延迟时间/秒  [区分自建MQ\华为MQ\阿里MQ]
     *                      自建MQ和华为云MQ  1s 5s 10s 30s 1m 2m 3m 4m 5m 6m 7m 8m 9m 10m 20m 30m 1h 2h
     *                      阿里MQ  延迟|定时3s投递, 设置为: System.currentTimeMillis() + 3000;
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:19
     */
    public void sendMqOrderMsg(MqMessageVo mqMessageVo, long deliverSecond) {
        Assert.hasText(mqMessageVo.getTag(), "MQ的tag不能为空!");
        this.sendMqOrderMsg(mqMessageVo, mqMessageVo.getTag(), deliverSecond);
    }


    /**
     * 发送MQ顺序消息
     *
     * @param mqMessageVo
     * @param shardingKey 唯一标记
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:20
     */
    public void sendMqOrderMsg(MqMessageVo mqMessageVo, String shardingKey) {
        this.sendMqOrderMsg(mqMessageVo, shardingKey, 0);
    }

    /**
     * 发送MQ顺序消息
     *
     * @param mqMessageVo
     * @param shardingKey   唯一标记
     * @param deliverSecond 延迟时间/秒  [区分自建MQ\华为MQ\阿里MQ]
     *                      自建MQ和华为云MQ  1s 5s 10s 30s 1m 2m 3m 4m 5m 6m 7m 8m 9m 10m 20m 30m 1h 2h
     *                      阿里MQ  延迟|定时3s投递, 设置为: System.currentTimeMillis() + 3000;
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:20
     */
    public void sendMqOrderMsg(MqMessageVo mqMessageVo, String shardingKey, long deliverSecond) {
        if (!StringUtils.hasText(mqMessageVo.getTopic())) {
            mqMessageVo.setTopic(RocketMqUtil.getOrderTopic());
        }
        Assert.hasText(shardingKey, "MQ顺序消息,唯一标记不能为空!(当无法确认唯一标记时,可传tag)");
        if (!mqMessageVo.getTopic().contains(RocketMqUtil.mqEnvironment())) {
            mqMessageVo.setTopic(mqMessageVo.getTopic() + RocketMqUtil.mqEnvironment());
        }
        RocketMqMessageLogDocument logVo = this.buildMqLog(mqMessageVo);
        log.info("发送的MQ消息内容[{}]", JSON.toJSONString(mqMessageVo));
        try {
            mqCommonService.sendMqOrderMsg(logVo, mqMessageVo, shardingKey, deliverSecond);
        } catch (IllegalArgumentException e) {
            log.error("MQ消息发送失败[{}]", JSON.toJSONString(mqMessageVo));
            log.error(e.getMessage(), e);
            this.sendMqExceptionSaveLog(logVo, mqMessageVo, e);
            throw new IllegalArgumentException(e.getMessage());
        } catch (Exception e) {
            log.error("MQ消息发送失败[{}]", JSON.toJSONString(mqMessageVo));
            log.error(e.getMessage(), e);
            this.sendMqExceptionSaveLog(logVo, mqMessageVo, e);
            throw new IllegalArgumentException("MQ消息发送失败,消息ID[" + logVo.getId() + "]");
        }
    }

    /**
     * 发送MQ消息异常时,保存日志到ES
     *
     * @param logVo
     * @param mqMessageVo
     * @param e
     */
    private void sendMqExceptionSaveLog(RocketMqMessageLogDocument logVo, MqMessageVo mqMessageVo, Exception e) {
        log.error("MQ消息发送失败:mqMessageVo = {} >>>", JSON.toJSONString(mqMessageVo), e);
        logVo.setSendLog(ExceptionUtils.getStackTrace(e));
        if (!StringUtils.hasText(logVo.getSendLog())) {
            logVo.setSendLog("MQ消息发送失败");
        }
        logVo.setSendStatus(EnableStatusEnum.DISABLE.getCode());
        if (RocketMqUtil.isSaveLog()) {
            rocketMqMessageLogDocumentRepository.save(logVo);
        }
    }

    /**
     * 构建mq消息ES日志信息
     *
     * @param mqMessageVo
     * @return
     */
    private RocketMqMessageLogDocument buildMqLog(MqMessageVo mqMessageVo) {
        if (buildMqInfoExtendService != null) {
            return buildMqInfoExtendService.buildMqLog(mqMessageVo);
        } else {
            if (!StringUtils.hasText(mqMessageVo.getCurrentAccount())) {
                mqMessageVo.setCurrentAccount(Objects.isNull(loginUserService) ? "" : loginUserService.getLoginAccountName());
            }
            if (!StringUtils.hasText(mqMessageVo.getAccountJson())) {
                AbstractCrmUserIdentity userIdentity = loginUserService.getAbstractLoginUser();
                mqMessageVo.setAccountJson(Objects.isNull(userIdentity) ? null : JSON.toJSONString(userIdentity));
            }
            RocketMqMessageLogDocument logDocument = RocketMqMessageLogDocument.buildLogVo(mqMessageVo);
            RocketMqUtil.checkSendMsg(logDocument);
            if (RocketMqUtil.isSaveLog()) {
                try {
                    logDocument = rocketMqMessageLogDocumentRepository.save(logDocument);
                } catch (IllegalStateException | NoNodeAvailableException e) {
                    log.error(e.getMessage(), e);
                    String errorMsg = RocketMqUtil.buildErrorInfo();
                    log.error("消费MQ消息，记录日志失败,elasticsearch异常!\n[{}]", errorMsg);
                    if (mqExceptionExtendService != null) {
                        mqExceptionExtendService.sendMqException(e);
                    }
                    throw new IllegalArgumentException(errorMsg);
                }
                redisService.set(MqConstant.MQ_MESSAGE + logDocument.getId(), logDocument, 60 * 10);
            } else {
                logDocument.setId(UUID.randomUUID().toString().replace("-", ""));
            }
            mqMessageVo.setId(logDocument.getId());
            return logDocument;
        }
    }

    /**
     * 事务完提交后发送MQ消息
     *
     * @param event
     * @return
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.10.13 19:22
     */
    @Async
    @TransactionalEventListener(phase = TransactionPhase.AFTER_COMMIT, classes = RocketMqProducerEvent.class)
    public void onRocketMqProducerEvent(RocketMqProducerEvent event) {
        if (event == null
                || (CollectionUtils.isEmpty(event.getGroupList()) && org.apache.commons.lang3.StringUtils.isBlank(event.getMsgBody()))
                || !StringUtils.hasText(event.getTag())) {
            log.error("事务完成后发送MQ消息失败");
            return;
        }
        String topic = event.getTopic();
        String tag = event.getTag();
        boolean isOrder = event.isOrder();
        String shardingKey = event.getShardingKey();
        if (isOrder
                && !StringUtils.hasText(shardingKey)) {
            shardingKey = tag;
        }
        String currentAccount = event.getCurrentAccount();
        String accountJson = event.getAccountJson();
        String businessKey = event.getBusinessKey();
        String businessType = event.getBusinessType();
        String operationType = event.getOperationType();
        String remarks = event.getRemarks();

        if (!StringUtils.isEmpty(event.getMsgBody())) {
            //如果有消息体，就直接推消息体
            MqMessageVo mqMessageVo = new MqMessageVo();
            mqMessageVo.setTopic(topic);
            mqMessageVo.setTag(tag);
            mqMessageVo.setMsgBody(event.getMsgBody());
            mqMessageVo.setCurrentAccount(currentAccount);
            mqMessageVo.setAccountJson(accountJson);
            mqMessageVo.setBusinessKey(businessKey);
            mqMessageVo.setBusinessType(businessType);
            mqMessageVo.setOperationType(operationType);
            mqMessageVo.setRemarks(remarks);
            if (isOrder) {
                this.sendMqOrderMsg(mqMessageVo, shardingKey);
            } else {
                this.sendMqMsg(mqMessageVo);
            }
            return;
        }


        Integer size = event.getGroupSize();
        if (size == null) {
            size = DEF_SIZE;
        }
        List<List> orderFormIdGroupList = Lists.partition(event.getGroupList(), size);
        AtomicInteger sentCountAtomic = new AtomicInteger(0);
        String finalShardingKey = shardingKey;
        orderFormIdGroupList.stream()
                .filter(k -> !CollectionUtils.isEmpty(k))
                .forEach(list -> {
                    try {
                        //沉睡0.15秒,防止,发送MQ消息过于频发
                        Thread.sleep(150);
                        sentCountAtomic.getAndIncrement();
                        MqMessageVo mqMessageVo = new MqMessageVo();
                        mqMessageVo.setTopic(topic);
                        mqMessageVo.setTag(tag);
                        mqMessageVo.setMsgBody(JSON.toJSONString(list));
                        mqMessageVo.setCurrentAccount(currentAccount);
                        mqMessageVo.setAccountJson(accountJson);
                        mqMessageVo.setBusinessKey(businessKey);
                        mqMessageVo.setBusinessType(businessType);
                        mqMessageVo.setOperationType(operationType);
                        mqMessageVo.setRemarks(remarks);
                        if (isOrder) {
                            this.sendMqOrderMsg(mqMessageVo, finalShardingKey);
                        } else {
                            this.sendMqMsg(mqMessageVo);
                        }
                    } catch (InterruptedException e) {
                        log.error("MQ异步发送消息失败 >>>");
                        log.error(e.getMessage(), e);
                    }
                });

    }

}
