package com.biz.crm.business.common.rocketmq.service.impl;

import com.alibaba.fastjson.JSON;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.SendResult;
import com.aliyun.openservices.ons.api.bean.OrderProducerBean;
import com.aliyun.openservices.ons.api.bean.ProducerBean;
import com.biz.crm.business.common.rocketmq.document.RocketMqMessageLogDocument;
import com.biz.crm.business.common.rocketmq.enums.RocketMqTypeEnum;
import com.biz.crm.business.common.rocketmq.service.MqCommonService;
import com.biz.crm.business.common.rocketmq.util.RocketMqUtil;
import com.biz.crm.business.common.rocketmq.vo.MqMessageVo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;


/**
 * 阿里 rocketMq
 *
 * @describe MQ
 * @author huxmld
 * @version v1.0.0
 * @date 2022/10/16 17:14
 */
@Slf4j
public class AlibabaMqServiceImpl implements MqCommonService {

    @Autowired(required = false)
    private ProducerBean producerBean;

    @Autowired(required = false)
    private OrderProducerBean orderProducerBean;


    /**
     * 获取消息类型  alibaba
     *
     * @return string
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022/10/16 17:14
     */
    @Override
    public RocketMqTypeEnum getMqTypeEnum() {
        return RocketMqTypeEnum.ALIBABA_CLOUD;
    }

    /**
     * 发送消息
     *
     * @param logDocument es日志
     * @param messageBody 消息内容
     * @return void
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022/10/16 17:14
     */
    @Override
    public void sendMqMsg(RocketMqMessageLogDocument logDocument, MqMessageVo messageBody) {
        this.sendMqMsg(logDocument, messageBody, 0);

    }


    /**
     * 发送消息
     *
     * @param logDocument   es日志
     * @param messageBody   消息内容
     * @param deliverSecond 延迟时间/秒
     * @return void
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022/10/16 17:14
     */
    @Override
    public void sendMqMsg(RocketMqMessageLogDocument logDocument, MqMessageVo messageBody, long deliverSecond) {
        try {
            RocketMqUtil.checkSendMsg(logDocument);
            Message msg = new Message(
                    // Message所属的Topic
                    logDocument.getTopic(),
                    // Message Tag 可理解为Gmail中的标签，对消息进行再归类，方便Consumer指定过滤条件在MQ服务器过滤
                    logDocument.getTag(),
                    // Message Body 可以是任何二进制形式的数据， MQ不做任何干预
                    // 需要Producer与Consumer协商好一致的序列化和反序列化方式
                    JSON.toJSONString(messageBody).getBytes());
            // 设置代表消息的业务关键属性，请尽可能全局唯一
            // 以方便您在无法正常收到消息情况下，可通过MQ 控制台查询消息并补发
            // 注意：不设置也不会影响消息正常收发
            msg.setKey(logDocument.getId());
            if (deliverSecond > 0) {
                /**
                 * <p> 设置消息的定时投递时间（绝对时间),最大延迟时间为7天. </p> <ol>
                 *   <li>
                 *     延迟投递: 延迟3s投递, 设置为: System.currentTimeMillis() + 3000;
                 * </li>
                 * <li>定时投递: 2016-02-01 11:30:00投递, 设置为: new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse("2016-02-01
                 * 11:30:00").getTime()</li> </ol>
                 */
                msg.setStartDeliverTime(System.currentTimeMillis() + deliverSecond * 1000);
            }

            // 发送消息，只要不抛异常就是成功
            SendResult sendResult = producerBean.send(msg);
            log.info("发送MQ返回信息{}", sendResult);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    /**
     * 同步发送顺序消费的消息
     *
     * @param logDocument es日志
     * @param messageBody 消息内容
     * @return void
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022/10/16 17:14
     */
    @Override
    public void sendMqOrderMsg(RocketMqMessageLogDocument logDocument, MqMessageVo messageBody, String shardingKey) {
        this.sendMqOrderMsg(logDocument, messageBody, shardingKey, 0);
    }

    /**
     * 同步发送顺序消费的消息
     *
     * @param logDocument   es日志
     * @param messageBody   消息内容
     * @param deliverSecond 延迟时间/秒
     * @return void
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022/10/16 17:14
     */
    @Override
    public void sendMqOrderMsg(RocketMqMessageLogDocument logDocument, MqMessageVo messageBody, String shardingKey,
                               long deliverSecond) {
        try {
            RocketMqUtil.checkSendMsg(logDocument);
            Message msg = new Message(
                    // Message所属的Topic
                    logDocument.getTopic(),
                    // Message Tag 可理解为Gmail中的标签，对消息进行再归类，方便Consumer指定过滤条件在MQ服务器过滤
                    logDocument.getTag(),
                    // Message Body 可以是任何二进制形式的数据， MQ不做任何干预
                    // 需要Producer与Consumer协商好一致的序列化和反序列化方式
                    JSON.toJSONString(messageBody).getBytes());
            // 设置代表消息的业务关键属性，请尽可能全局唯一
            // 以方便您在无法正常收到消息情况下，可通过MQ 控制台查询消息并补发
            // 注意：不设置也不会影响消息正常收发
            msg.setKey(logDocument.getId());
            if (deliverSecond > 0) {
                /**
                 * <p> 设置消息的定时投递时间（绝对时间),最大延迟时间为7天. </p> <ol>
                 *   <li>
                 *     延迟投递: 延迟3s投递, 设置为: System.currentTimeMillis() + 3000;
                 * </li>
                 * <li>定时投递: 2016-02-01 11:30:00投递, 设置为: new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse("2016-02-01
                 * 11:30:00").getTime()</li> </ol>
                 */
                msg.setStartDeliverTime(System.currentTimeMillis() + deliverSecond * 1000);
            }
            SendResult sendResult = orderProducerBean.send(msg, shardingKey);
            log.info("发送MQ返回信息{}", sendResult);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }
}
