package com.biz.crm.business.common.rocketmq.task;

import com.biz.crm.business.common.base.service.RedisLockService;
import com.biz.crm.business.common.rocketmq.config.RocketMqConfig;
import com.biz.crm.business.common.rocketmq.constant.MqConstant;
import com.biz.crm.business.common.rocketmq.document.RocketMqMessageLogDocument;
import com.biz.crm.business.common.rocketmq.service.MqCleanLogExtendService;
import com.biz.crm.business.common.rocketmq.util.RocketMqUtil;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.ByQueryResponse;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 系统定时清除MQ日志
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.14 13:43
 */
@EnableScheduling
@Component
@Slf4j
public class DeleteMqLogCornTask implements SchedulingConfigurer {

    private static final String DEFAULT_CRON = "0 0 5 * * ?";

    private String cron = DEFAULT_CRON;

    @Resource
    private ElasticsearchRestTemplate elasticsearchRestTemplate;
    @Resource
    private RocketMqConfig rocketMqConfig;
    @Autowired(required = false)
    private MqCleanLogExtendService mqCleanLogExtendService;
    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 根据配置文件参数,系统启动完成后,自动设置自定义定时任务执行计划
     */
    @PostConstruct
    private void setDeleteMqLogCron() {
        this.setCron(rocketMqConfig.getCleanCron());
    }

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.addTriggerTask(this::deleteData, triggerContext -> {
            CronTrigger trigger = new CronTrigger(cron);
            return trigger.nextExecutionTime(triggerContext);
        });
    }

    public void setCron(String cron) {
        log.info("====>     清理MQ发送日志定时任务:原来的cron：[{}]]更新后的cron：[{}]     <=====", this.cron, cron);
        this.cron = cron;
    }

    public String getCron() {
        return this.cron;
    }

    /**
     * 清理MQ发送日志
     */
    private void deleteData() {
        String lockKey = MqConstant.MQ_DELETE_LOG;
        if (redisLockService != null) {
            boolean lock = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30);
            if (!lock) {
                return;
            }
        }
        if (mqCleanLogExtendService == null) {
            try {
                Integer saveDay = rocketMqConfig.getSaveDay();
                if (saveDay == null) {
                    saveDay = 14;
                }
                Long minNum = rocketMqConfig.getMinNum();
                if (minNum == null) {
                    minNum = 10000L;
                }
                long totalCount = elasticsearchRestTemplate.count(new NativeSearchQuery(QueryBuilders.boolQuery()), RocketMqMessageLogDocument.class);
                if (totalCount > minNum) {
                    long saveDayMillis = 1000L * 60 * 60 * 24 * saveDay;
                    long cleanMillis = System.currentTimeMillis() - saveDayMillis;
                    log.info("MQ日志清理[{}]及其以前的", RocketMqUtil.DATETIME_FORMAT.format(new Date(cleanMillis)));
                    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
                    boolQueryBuilder.must(QueryBuilders.rangeQuery("createDateSort")
                            .lte(cleanMillis));
                    NativeSearchQuery deleteQuery = new NativeSearchQuery(boolQueryBuilder);
                    ByQueryResponse queryResponse = elasticsearchRestTemplate.delete(deleteQuery, RocketMqMessageLogDocument.class);
                    log.info("MQ日志清理数量:[{}]条", queryResponse.getDeleted());
                } else {
                    log.info("MQ日志当前总[{}]条,小于日志保有最小量[{}],不清理日志。", totalCount, minNum);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        } else {
            mqCleanLogExtendService.cleanMqLog();
        }
        if (redisLockService != null) {
            redisLockService.unlock(lockKey);
        }
    }

}
