package com.biz.crm.business.common.log.local.service.internal;

import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.log.local.repository.ExternalLogEsDocument;
import com.biz.crm.business.common.log.local.repository.ExternalLogEsRepository;
import com.biz.crm.business.common.log.local.service.ExternalLogService;
import com.biz.crm.business.common.log.sdk.dto.ExternalLogDetailDto;
import com.biz.crm.business.common.log.sdk.dto.ExternalLogListDto;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.data.elasticsearch.core.query.NativeSearchQueryBuilder;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * @author huojia
 * @date 2022年10月21日 17:23
 */
@Slf4j
@Service
public class ExternalLogServiceImpl implements ExternalLogService {

    @Autowired(required = false)
    private ElasticsearchRestTemplate esTemplate;
    @Autowired(required = false)
    private ExternalLogEsRepository externalLogEsRepository;
    private final static int MAX_TOTAL = 10000;

    /**
     * 保存日志
     *
     * @param entity
     * @return void
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.4 13:41
     */
    @Override
    public void saveEsLog(ExternalLogEsDocument entity) {
        if (ObjectUtils.isEmpty(entity)) {
            return;
        }
        entity.setTenantCode(TenantUtils.getTenantCode());
        externalLogEsRepository.save(entity);
    }

    /**
     * 分页查询列表
     *
     * @param pageable
     * @param reqVo
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.biz.crm.external.business.log.sdk.vo.ExternalLogListVo>
     * @describe 简述
     * @author huxmld
     * @version v1.0.0
     * @date 2022.11.4 13:41
     */
    @Override
    public Page<ExternalLogEsDocument> findByConditions(Pageable pageable, ExternalLogListDto reqVo) {
        pageable = ObjectUtils.defaultIfNull(pageable, PageRequest.of(0, 50));
        reqVo = Optional.ofNullable(reqVo).orElse(new ExternalLogListDto());
        BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
        int pageNumber = pageable.getPageNumber();
        int pageSize = pageable.getPageSize();
        if (pageNumber > 0) {
            pageNumber = pageNumber - 1;
        }
        Assert.isTrue((pageNumber + 1) * pageSize < MAX_TOTAL,
                "分页查询,单次查询总数不可超过[" + MAX_TOTAL + "]条,请输入更多查询条件!");
        Pageable pageEs = PageRequest.of(pageNumber, pageable.getPageSize());
        if (StringUtils.isNotBlank(reqVo.getId())) {
            queryBuilder.must(QueryBuilders.queryStringQuery(reqVo.getId()).field("id"));
        }
        // 租户编号精确查询
        if (StringUtils.isNotBlank(reqVo.getTenantCode())) {
            queryBuilder.must(QueryBuilders.queryStringQuery(reqVo.getTenantCode()).field("tenantCode"));
        }

        // 映射日志意义描述，分词查询
        if (StringUtils.isNotBlank(reqVo.getMethod())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("method", this.getLikeString(reqVo.getMethod())));
        }// 映射日志意义描述，分词查询
        if (StringUtils.isNotBlank(reqVo.getRequestUri())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("requestUri", this.getLikeString(reqVo.getRequestUri())));
        }
        // 映射日志意义描述，分词查询
        if (StringUtils.isNotBlank(reqVo.getMethodMsg())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("methodMsg", this.getLikeString(reqVo.getMethodMsg())));
        }
        // 账号ID
        if (StringUtils.isNotBlank(reqVo.getAccessId())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("accessId", this.getLikeString(reqVo.getAccessId())));
        }
        // 请求加密账号
        if (StringUtils.isNotBlank(reqVo.getAccessKey())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("accessKey", this.getLikeString(reqVo.getAccessKey())));
        }
        // 请求账号
        if (StringUtils.isNotBlank(reqVo.getUserName())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("userName", this.getLikeString(reqVo.getUserName())));
        }
        // 请求账户名称
        if (StringUtils.isNotBlank(reqVo.getFullName())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("fullName", this.getLikeString(reqVo.getFullName())));
        }
        // 调用类型
        if (reqVo.getInvokeType() != null) {
            queryBuilder.must(QueryBuilders.matchQuery("invokeType", reqVo.getInvokeType()));
        }

        // 请求源
        if (StringUtils.isNotBlank(reqVo.getSourceMain())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("sourceMain", this.getLikeString(reqVo.getSourceMain())));
        }

        // 目标源
        if (StringUtils.isNotBlank(reqVo.getTargetMain())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("targetMain", this.getLikeString(reqVo.getTargetMain())));
        }

        // 状态
        if (StringUtils.isNotBlank(reqVo.getStatus())) {
            queryBuilder.must(QueryBuilders.queryStringQuery(reqVo.getStatus()).field("status"));
        }

        // 消息文本
        if (StringUtils.isNotBlank(reqVo.getTipMsg())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("tipMsg", this.getLikeString(reqVo.getTipMsg())));
        }

        // 返回数据条数
        if (reqVo.getRespJsonSize() != null) {
            queryBuilder.must(QueryBuilders.matchQuery("respJsonSize", reqVo.getRespJsonSize()));
        }

        // 返回数据条数
        if (StringUtils.isNotBlank(reqVo.getExceptionStack())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("exceptionStack", this.getLikeString(reqVo.getExceptionStack())));
        }

        // TOKEN
        if (StringUtils.isNotBlank(reqVo.getToken())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("token", this.getLikeString(reqVo.getToken())));
        }

        // 业务标记
        if (StringUtils.isNotBlank(reqVo.getBusinessKey())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("businessKey", this.getLikeString(reqVo.getBusinessKey())));
        }

        // 业务类型
        if (StringUtils.isNotBlank(reqVo.getBusinessType())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("businessType", this.getLikeString(reqVo.getBusinessType())));
        }
        // 请求时间起，范围查询
        if (StringUtils.isNotBlank(reqVo.getReqDateStart())) {
            String dateStr = reqVo.getReqDateStart();
            if (DateUtil.DEFAULT_YEAR_MONTH_DAY.length() == dateStr.length()) {
                dateStr = dateStr + " " + DateUtil.DAY_EARLIEST_TIME;
            }
            queryBuilder.must(QueryBuilders.rangeQuery("reqDate").gte(dateStr));
        }

        // 请求时间止，范围查询
        if (StringUtils.isNotBlank(reqVo.getReqDateEnd())) {
            String dateStr = reqVo.getReqDateEnd();
            if (DateUtil.DEFAULT_YEAR_MONTH_DAY.length() == dateStr.length()) {
                dateStr = dateStr + " " + DateUtil.DAY_LATEST_TIME;
            }
            queryBuilder.must(QueryBuilders.rangeQuery("reqDate").lte(dateStr));
        }

        // 响应时间起
        if (StringUtils.isNotBlank(reqVo.getRespDateStart())) {
            String dateStr = reqVo.getRespDateStart();
            if (DateUtil.DEFAULT_YEAR_MONTH_DAY.length() == dateStr.length()) {
                dateStr = dateStr + " " + DateUtil.DAY_EARLIEST_TIME;
            }
            queryBuilder.must(QueryBuilders.rangeQuery("respDate").gte(dateStr));
        }

        // 响应时间止
        if (StringUtils.isNotBlank(reqVo.getRespDateEnd())) {
            String dateStr = reqVo.getRespDateEnd();
            if (DateUtil.DEFAULT_YEAR_MONTH_DAY.length() == dateStr.length()) {
                dateStr = dateStr + " " + DateUtil.DAY_LATEST_TIME;
            }
            queryBuilder.must(QueryBuilders.rangeQuery("respDate").lte(dateStr));
        }

        // 请求时间
        if (StringUtils.isNotBlank(reqVo.getReqDate())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("reqDate", reqVo.getReqDate() + "*"));
        }

        // 返回时间
        if (StringUtils.isNotBlank(reqVo.getRespDate())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("respDate", reqVo.getRespDate() + "*"));
        }

        // 请求头部参数
        if (StringUtils.isNotBlank(reqVo.getReqHead())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("reqHead", this.getLikeString(reqVo.getReqHead())));
        }

        // 请求参数
        if (StringUtils.isNotBlank(reqVo.getReqJson())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("reqJson", this.getLikeString(reqVo.getReqJson())));
        }

        // 返回参数
        if (StringUtils.isNotBlank(reqVo.getRespJson())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("respJson", this.getLikeString(reqVo.getRespJson())));
        }

        // 消息文本
        if (StringUtils.isNotBlank(reqVo.getTipMsg())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("tipMsg", this.getLikeString(reqVo.getTipMsg())));
        }

        // 异常信息
        if (StringUtils.isNotBlank(reqVo.getExceptionStack())) {
            queryBuilder.must(QueryBuilders.wildcardQuery("exceptionStack", this.getLikeString(reqVo.getExceptionStack())));
        }

        // 其他的需要分词查询
        if (StringUtils.isNotBlank(reqVo.getQueryKey())) {
            queryBuilder.must(QueryBuilders.multiMatchQuery(reqVo.getQueryKey(),
                    "reqHead", "reqJson", "respJson", "tipMsg", "exceptionStack"));
        }

        NativeSearchQuery nativeSearchQuery = new NativeSearchQueryBuilder()
                .withQuery(queryBuilder)
                .withPageable(pageEs)
                .withSorts(SortBuilders.fieldSort("reqDate").order(SortOrder.DESC))
                .build();

        SearchHits<ExternalLogEsDocument> searchHits = this.esTemplate.search(nativeSearchQuery,
                ExternalLogEsDocument.class);

        Page<ExternalLogEsDocument> resultPage = new Page<>();
        resultPage.setSize(pageable.getPageSize());
        resultPage.setCurrent(pageable.getPageNumber());
        resultPage.setTotal(searchHits.getTotalHits() < MAX_TOTAL ? searchHits.getTotalHits() : MAX_TOTAL);
        boolean hasContent = reqVo.isHasContent();
        if (searchHits.hasSearchHits()) {
            List<ExternalLogEsDocument> listVos = new ArrayList<>();
            searchHits.forEach(item -> {
                ExternalLogEsDocument document = item.getContent();
                if (!hasContent) {
                    document.setReqHead(null);
                    document.setReqJson(null);
                    document.setRespJson(null);
                }
                listVos.add(document);
            });
            resultPage.setRecords(listVos);
        } else {
            resultPage.setCurrent(0);
        }
        return resultPage;
    }

    /**
     * es like查询
     *
     * @param queryString
     * @return
     */
    private String getLikeString(String queryString) {
        if (StringUtils.isEmpty(queryString)) {
            return "";
        }
        return "*" + queryString + "*";
    }

    /**
     * 获取详情
     *
     * @param id
     * @return com.biz.crm.business.common.log.local.repository.ExternalLogEsDocument
     * @author huxmld
     * @version v1.0.0
     * @date 2023.1.30 22:13
     */
    @Override
    public ExternalLogEsDocument findDetailById(String id) {
        if (StringUtils.isEmpty(id)) {
            return null;
        }
        return externalLogEsRepository.findById(id).orElse(null);
    }

    @Override
    public List<ExternalLogDetailDto> findByIdList(List<String> idList) {
        if (CollectionUtils.isEmpty(idList)) {
            return Lists.newArrayList();
        }
        Iterable<ExternalLogEsDocument> documents = externalLogEsRepository.findAllById(idList);
        List<ExternalLogEsDocument> documentList = Lists.newArrayList(documents);
        if (CollectionUtils.isEmpty(documentList)) {
            return Lists.newArrayList();
        }
        String json = JSONArray.toJSONString(documentList);
        return JSONArray.parseArray(json, ExternalLogDetailDto.class);
    }

}
