package com.biz.crm.business.common.log.local.task;

import com.biz.crm.business.common.log.local.config.ExternalLogConfig;
import com.biz.crm.business.common.log.local.repository.ExternalLogEsDocument;
import com.biz.crm.business.common.log.local.service.CleanLogExtendService;
import com.biz.crm.business.common.base.service.RedisLockService;
import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.log.local.constant.LogConstant;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.ByQueryResponse;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 清理外部系统ES日志
 *
 * @describe 简述
 * @author huxmld
 * @version v1.0.0
 * @date 2022.10.14 13:43
 */
@EnableScheduling
@Component
@Slf4j
public class DeleteExternalLogCornTask implements SchedulingConfigurer {

    private static final String DEFAULT_CRON = "0 30 6 * * ?";

    private String cron = DEFAULT_CRON;
    private boolean openClean;

    @Resource
    private ElasticsearchRestTemplate elasticsearchRestTemplate;
    @Resource
    private ExternalLogConfig externalLogConfig;
    @Autowired(required = false)
    private CleanLogExtendService cleanLogExtendService;
    @Autowired(required = false)
    private RedisLockService redisLockService;
    /**
     * 清理对外日志应用-必须是 crm-mdm
     */
    private final static String APPLICATION_NAME = "crm-mdm";

    /**
     * 根据配置文件参数,系统启动完成后,自动设置自定义定时任务执行计划
     */
    @PostConstruct
    private void setDeleteLogCron() {
        this.setCron(externalLogConfig.getCleanCron());
        openClean = externalLogConfig.isOpenClean();
    }

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.addTriggerTask(this::deleteData, triggerContext -> {
            CronTrigger trigger = new CronTrigger(cron);
            return trigger.nextExecutionTime(triggerContext);
        });
    }

    public void setCron(String cron) {
        log.info("====>     清理外部系统ES日志:原来的cron：[{}]]更新后的cron：[{}]     <=====", this.cron, cron);
        this.cron = cron;
    }


    /**
     * 清理日志
     */
    private void deleteData() {
        if (!APPLICATION_NAME.equalsIgnoreCase(externalLogConfig.getApplicationName())
                || !openClean) {
            return;
        }
        String lockKey = LogConstant.FOREIGN_LOG_DEL;
        if (redisLockService != null) {
            boolean lock = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30);
            if (!lock) {
                return;
            }
        }
        log.info("=====>    清理外部系统ES日志 start    <=====");
        if (cleanLogExtendService == null) {
            try {
                Integer saveDay = externalLogConfig.getSaveDay();
                if (saveDay == null) {
                    saveDay = 14;
                }
                Long minNum = externalLogConfig.getMinNum();
                if (minNum == null) {
                    minNum = 10000L;
                }
                log.info("清理外部系统ES日志清理[{}]天及其以前的;最小存放数量[{}]条", saveDay, minNum);
                long totalCount = elasticsearchRestTemplate.count(new NativeSearchQuery(QueryBuilders.boolQuery()), ExternalLogEsDocument.class);
                if (totalCount > minNum) {
                    log.info("清理外部系统ES日志当前总[{}]条,大于日志保有最小量[{}],开始清理日志。", totalCount, minNum);
                    long saveDayMillis = 1000L * 60 * 60 * 24 * saveDay;
                    long cleanMillis = System.currentTimeMillis() - saveDayMillis;
                    String cleanDate = DateUtil.date_yyyy_MM_dd_HH_mm_ss.format(new Date(cleanMillis));
                    log.info("清理外部系统ES日志清理[{}]及其以前的", cleanDate);
                    BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
                    boolQueryBuilder.must(QueryBuilders.rangeQuery("reqDate").lte(cleanDate));
                    NativeSearchQuery deleteQuery = new NativeSearchQuery(boolQueryBuilder);
                    ByQueryResponse queryResponse = elasticsearchRestTemplate.delete(deleteQuery, ExternalLogEsDocument.class);
                    log.info("清理外部系统ES日志清理数量:[{}]条", queryResponse.getDeleted());
                } else {
                    log.info("清理外部系统ES日志当前总[{}]条,小于日志保有最小量[{}],不清理日志。", totalCount, minNum);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        } else {
            cleanLogExtendService.cleanMqLog();
        }
        if (redisLockService != null) {
            redisLockService.unlock(lockKey);
        }
        log.info("=====>    清理外部系统ES日志 end    <=====");
    }

}
