package com.biz.crm.business.common.page.cache.service.internal;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.page.cache.constant.BusinessPageCacheConstant;
import com.biz.crm.business.common.page.cache.service.BusinessPageCacheHelper;
import com.biz.crm.business.common.page.cache.service.BusinessPageCacheService;
import com.biz.crm.business.common.sdk.enums.BooleanEnum;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 分页缓存服务
 *
 * @author wanghaojia
 * @date 2022/12/7 11:46
 */
public class BusinessPageCacheServiceImpl<Vo, Dto> implements BusinessPageCacheService<Vo, Dto> {

    @Autowired(required = false)
    protected BusinessPageCacheHelper<Vo, Dto> helper;

    @Autowired(required = false)
    private RedisService redisService;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    /**
     * 分页获取缓存数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return 分页缓存数据
     */
    @Override
    public Page<Vo> findCachePageList(Pageable pageable, Dto dto, String cacheKey) {
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);
        String redisCacheInitKey = helper.getRedisCacheInitKey(cacheKey);
        Page<Vo> page = new Page<>(pageable.getPageNumber(), pageable.getPageSize());
        page.setTotal(0);
        page.setRecords(Lists.newArrayList());

        a:
        if (redisService.hasKey(redisCacheIdKey)) {
            //redis里面有的话直接从redis里面取
            Long total = redisService.lSize(redisCacheIdKey);
            page.setTotal(total);
            List<Object> idList = redisService.lRange(redisCacheIdKey, page.offset(), page.offset() + page.getSize() - 1);
            if (!CollectionUtils.isEmpty(idList)) {
                List<Object> dataList = redisTemplate.opsForHash().multiGet(redisCacheDataKey, idList);
                List<Dto> dtoList = (List<Dto>) dataList;
                List<Vo> voList = helper.dtoListToVoList(dtoList);
                page.setRecords(voList);
            }
        } else if (!redisService.hasKey(redisCacheInitKey) && null != dto) {
            //标记为已初始化，不重复初始化
            redisService.set(redisCacheInitKey, BooleanEnum.TRUE, helper.getExpireTime());
            //放到try catch里面 是为了以防分页从数据库查询报错了 初始化状态被确认 修正报错过后查询为空的情况
            try {
                //redis里面没有
                List<Dto> dtoList = helper.findDtoListFromRepository(dto, cacheKey);

                if (CollectionUtils.isEmpty(dtoList)) {
                    redisService.del(redisCacheInitKey);
                    break a;
                }

                if (helper.initToCacheFromRepository()) {
                    helper.putCache(cacheKey, dtoList);
                }

                //放到缓存里面
                page.setTotal(dtoList.size());
                long start = page.offset();
                if (page.getTotal() > start) {
                    long end = page.offset() + page.getSize();
                    if (page.getTotal() < end) {
                        end = page.getTotal();
                    }
                    List<Dto> recordDtoList = dtoList.subList((int) page.offset(), (int) end);
                    List<Vo> voList = helper.dtoListToVoList(recordDtoList);
                    page.setRecords(voList);
                }
            } catch (Exception e) {
                redisService.del(redisCacheInitKey);
            }
        }
        //更新下VO里面的字段值
        helper.fillVoListProperties(page.getRecords());
        return page;
    }

    /**
     * 获取缓存中的所有明细行
     *
     * @param cacheKey 缓存key
     */
    @Override
    public List<Dto> findCacheList(String cacheKey) {
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        Set<String> idKeys = redisTemplate.keys(redisCacheIdKey + "*");
        List<Object> dataList = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(idKeys)) {
            for (String idKey : idKeys) {
                String dataKey = idKey.replace(BusinessPageCacheConstant.REDIS_KEY_ID, BusinessPageCacheConstant.REDIS_KEY_DATA);
                List<Object> idList = redisService.lRange(idKey, 0, -1);
                dataList.addAll(redisTemplate.opsForHash().multiGet(dataKey, idList));
            }
        }
        return (List<Dto>) dataList;
    }

    /**
     * 2、新增一行接口，在缓存中行首插入一条数据
     *
     * @param cacheKey 缓存key
     * @param itemList 要保存的当前页数据
     */
    @Override
    public void addItemCache(String cacheKey, List<Dto> itemList) {
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);

        //过滤出来需要更新的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        List<Dto> updateList = itemList.stream().filter(item -> idList.contains(helper.getDtoKey(item))).collect(Collectors.toList());
        helper.updateItem(cacheKey, updateList);
        helper.filterSaveItem(cacheKey, idList, itemList, updateList);

        List<Dto> newItemList = helper.newItem(cacheKey, itemList);
        Object[] newIdArr = newItemList.stream().map(helper::getDtoKey).toArray();
        helper.doSaveNewKey(cacheKey, newIdArr);
        if (!CollectionUtils.isEmpty(newItemList)) {
            updateList.addAll(newItemList);
        }
        Map<Object, Dto> updateMap = updateList.stream().collect(Collectors.toMap(helper::getDtoKey, Function.identity()));
        helper.doSaveItem(cacheKey, updateMap);
    }

    /**
     * 3、复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param cacheKey 缓存key
     * @param itemList 要保存的当前页数据
     */
    @Override
    public void copyItemListCache(String cacheKey, List<Dto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);

        //过滤出来需要更新的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        List<Dto> updateList = itemList.stream().filter(item -> idList.contains(helper.getDtoKey(item))).collect(Collectors.toList());
        helper.updateItem(cacheKey, updateList);

        List<Dto> copyList = itemList.stream().filter(item -> BooleanEnum.TRUE.getNumStr().equals(helper.getCheckedStatus(item))).collect(Collectors.toList());
        List<Dto> newItemList = helper.copyItem(cacheKey, copyList);
        Object[] newIdArr = newItemList.stream().map(helper::getDtoKey).toArray();
        helper.doSaveNewKey(cacheKey, newIdArr);

        updateList.addAll(newItemList);
        Map<Object, Dto> updateMap = updateList.stream().collect(Collectors.toMap(helper::getDtoKey, Function.identity()));
        helper.doSaveItem(cacheKey, updateMap);
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void saveCurrentPageCache(String cacheKey, List<Dto> saveList) {
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        helper.updateItem(cacheKey, saveList);
        if (CollectionUtils.isEmpty(saveList)) {
            return;
        }
        Map<Object, Dto> updateMap = saveList.stream().collect(Collectors.toMap(helper::getDtoKey, Function.identity()));
        redisTemplate.opsForHash().putAll(redisCacheDataKey, updateMap);
        redisService.expire(redisCacheDataKey, helper.getExpireTime());
    }

    /**
     * 保存缓存的集合
     *
     * @param cacheKey
     */
    @Override
    public void saveListCache(String cacheKey, List<Dto> itemList) {

        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);

        //过滤出来需要更新的数据
        List<Object> idList = redisService.lRange(redisCacheIdKey, 0, -1);
        List<Dto> updateList = itemList.stream().filter(item -> idList.contains(helper.getDtoKey(item))).collect(Collectors.toList());
        helper.updateItem(cacheKey, updateList);
        helper.filterSaveItem(cacheKey, idList, itemList, updateList);
        Object[] newIdArr = itemList.stream().map(helper::getDtoKey).toArray();
        helper.doSaveNewKey(cacheKey, newIdArr);

        updateList.addAll(itemList);
        Map<Object, Dto> updateMap = updateList.stream().collect(Collectors.toMap(helper::getDtoKey, Function.identity()));
        helper.doSaveItem(cacheKey, updateMap);
    }

    /**
     * 7、多行删除并保存当前页数据并返回指定页数据接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void deleteCacheList(String cacheKey, List<Dto> itemList) {
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);

        List<Object> deleteIdList = Lists.newArrayList();
        Map<Object, Dto> updateMap = Maps.newHashMap();
        List<Dto> updateList = Lists.newArrayList();
        for (Dto dto : itemList) {
            if (BooleanEnum.TRUE.getNumStr().equals(helper.getCheckedStatus(dto))) {
                //选中状态，代表该行数据要删除
                deleteIdList.add(helper.getDtoKey(dto));
            } else {
                updateMap.put(helper.getDtoKey(dto), dto);
                updateList.add(dto);
            }
        }
        if (updateMap.size() > 0) {
            helper.updateItem(cacheKey, updateList);
            helper.doSaveItem(cacheKey, updateMap);
        }
        if (!CollectionUtils.isEmpty(deleteIdList)) {
            for (Object id : deleteIdList) {
                redisService.lRemove(redisCacheIdKey, 0, id);
            }
            redisTemplate.opsForHash().delete(redisCacheDataKey, deleteIdList.toArray());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @Override
    public void clearCache(String cacheKey) {
        String redisCacheIdKey = helper.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = helper.getRedisCacheDataKey(cacheKey);
        String redisCacheInitKey = helper.getRedisCacheInitKey(cacheKey);
        Set<String> delKeys = Sets.newHashSet();
        Set<String> idKeys = redisTemplate.keys(redisCacheIdKey + "*");
        if (!CollectionUtils.isEmpty(idKeys)) {
            delKeys.addAll(idKeys);
        }
        Set<String> dataKeys = redisTemplate.keys(redisCacheDataKey + "*");
        if (!CollectionUtils.isEmpty(dataKeys)) {
            delKeys.addAll(dataKeys);
        }
        Set<String> initKeys = redisTemplate.keys(redisCacheInitKey + "*");
        if (!CollectionUtils.isEmpty(initKeys)){
            delKeys.addAll(initKeys);
        }
        if (!CollectionUtils.isEmpty(delKeys)) {
            redisTemplate.delete(delKeys);
        }
    }

}
