package com.biz.crm.business.common.page.cache.service;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.business.common.base.util.JsonPropertyUtil;
import com.biz.crm.business.common.page.cache.constant.BusinessPageCacheConstant;
import com.biz.crm.business.common.sdk.service.RedisService;
import com.bizunited.nebula.common.service.NebulaToolkitService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 分页缓存接口帮助类
 *
 * @author wanghaojia
 * @date 2022/12/7 11:53
 */
public abstract class BusinessPageCacheHelper<Vo, Dto> {

    @Autowired(required = false)
    protected RedisService redisService;

    @Autowired(required = false)
    private RedisTemplate<String, Object> redisTemplate;

    @Autowired(required = false)
    private NebulaToolkitService nebulaToolkitService;

    /**
     * 缓存键前缀
     */
    public abstract String getCacheKeyPrefix();

    /**
     * @return Dto Class对象
     */
    public abstract Class<Dto> getDtoClass();

    /**
     * @return Vo Class对象
     */
    public abstract Class<Vo> getVoClass();

    /**
     * 从数据库查询初始化数据
     */
    public abstract List<Dto> findDtoListFromRepository(Dto dto, String cacheKey);

    /**
     * 是否初始化数据到缓存，默认为true
     */
    public boolean initToCacheFromRepository() {
        return true;
    }

    /**
     * 添加数据到缓存
     *
     * @param cacheKey 缓存key
     * @param list     要添加的数据
     */
    public void putCache(String cacheKey, List<Dto> list) {
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);

        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        redisService.lPushAll(redisCacheIdKey, this.getExpireTime(), list.stream().map(this::getDtoKey).toArray());
        Map<Object, Dto> newDataMap = list.stream().collect(Collectors.toMap(this::getDtoKey, Function.identity(), (o, n) -> n));
        redisTemplate.opsForHash().putAll(redisCacheDataKey, newDataMap);
        redisService.expire(redisCacheDataKey, this.getExpireTime());
    }

    /**
     * 新增数据
     */
    public abstract List<Dto> newItem(String cacheKey, List<Dto> itemList);

    public void doSaveNewKey(String cacheKey, Object[] newIdArr) {
        if (newIdArr.length == 0) {
            return;
        }
        String redisCacheIdKey = this.getRedisCacheIdKey(cacheKey);
        redisTemplate.opsForList().leftPushAll(redisCacheIdKey, newIdArr);
        redisService.expire(redisCacheIdKey, this.getExpireTime());
    }

    /**
     * 复制数据
     */
    public abstract List<Dto> copyItem(String cacheKey, List<Dto> itemList);

    /**
     * 更新数据
     */
    public void updateItem(String cacheKey, List<Dto> itemList) {
    }

    public void doSaveItem(String cacheKey, Map<Object, Dto> updateMap) {
        if (null == updateMap || updateMap.size() == 0) {
            return;
        }
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        redisTemplate.opsForHash().putAll(redisCacheDataKey, updateMap);
        redisService.expire(redisCacheDataKey, this.getExpireTime());
    }

    /**
     * 获取Dto中的主键
     *
     * @return 主键
     */
    public abstract Object getDtoKey(Dto dto);

    /**
     * 获取是否选中状态
     *
     * @return 主键
     */
    public abstract String getCheckedStatus(Dto dto);


    /**
     * 获取redis分页缓存id key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    public String getRedisCacheIdKey(String cacheKey) {
        return this.getCacheKeyPrefix() + BusinessPageCacheConstant.REDIS_KEY_ID + cacheKey;
    }

    /**
     * 获取redis分页缓存data key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    public String getRedisCacheDataKey(String cacheKey) {
        return this.getCacheKeyPrefix() + BusinessPageCacheConstant.REDIS_KEY_DATA + cacheKey;
    }

    /**
     * 获取redis分页缓存是否初始化 key
     *
     * @param cacheKey 前端传过来的key
     * @return redis的缓存key
     */
    public String getRedisCacheInitKey(String cacheKey) {
        return this.getCacheKeyPrefix() + BusinessPageCacheConstant.REDIS_KEY_INIT + cacheKey;
    }


    /**
     * 获取redis分页缓存锁
     *
     * @param cacheKey
     * @return
     */
    public String getRedisPageCacheLockKey(String cacheKey) {
        return this.getCacheKeyPrefix() + BusinessPageCacheConstant.REDIS_PAGE_KEY_LOCK + cacheKey;
    }

    /**
     * 缓存过期时间
     */
    public long getExpireTime() {
        return BusinessPageCacheConstant.DEFAULT_PAGE_CACHE_EXPIRE_TIME;
    }

    /**
     * 填充VO属性
     *
     * @param records 返回的数据
     */
    public void fillVoListProperties(List<Vo> records) {
    }

    public List<Vo> dtoListToVoList(List<Dto> dtoList) {
        return JSONObject.parseArray(JSONObject.toJSONString(dtoList), this.getVoClass());
    }

    /**
     * 获取总条数
     *
     * @param cacheKey 缓存key
     * @return Integer
     */
    public Integer getTotal(String cacheKey) {
        return redisService.lSize(this.getRedisCacheIdKey(cacheKey)).intValue();
    }

    /**
     * 获取总条数
     *
     * @param cacheKey 缓存key
     * @return Integer
     */
    public Dto getDtoByKey(String cacheKey, String dtoKey) {
        String redisCacheDataKey = this.getRedisCacheDataKey(cacheKey);
        Object object = redisTemplate.opsForHash().get(redisCacheDataKey, dtoKey);
        if (null == object) {
            return null;
        }
        return (Dto) object;
    }

    public void filterSaveItem(String cacheKey, List<Object> idList, List<Dto> itemList, List<Dto> updateList) {
        if (CollectionUtils.isEmpty(idList)) {
            return;
        }
        if (CollectionUtils.isEmpty(itemList)) {
            return;
        }
        itemList.removeIf(item -> idList.contains(this.getDtoKey(item)));
    }
}
