package com.biz.crm.business.common.page.cache.service.internal;

import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.base.util.UuidCrmUtil;
import com.biz.crm.business.common.page.cache.repository.BusinessImportCacheEsDocument;
import com.biz.crm.business.common.page.cache.repository.BusinessImportCacheEsRepository;
import com.biz.crm.business.common.page.cache.service.BusinessImportCacheService;
import com.bizunited.nebula.common.util.tenant.TenantUtils;
import org.apache.commons.lang3.StringUtils;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.SearchHits;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.stereotype.Service;

import java.util.Date;

/**
 * 导入缓存处理
 * @author wanghaojia
 * @date 2023/12/21 22:04
 */
@Service
public class BusinessImportCacheServiceImpl implements BusinessImportCacheService {


    @Autowired(required = false)
    private BusinessImportCacheEsRepository repository;

    @Autowired(required = false)
    private ElasticsearchRestTemplate esTemplate;

    /**
     * 保存缓存
     */
    @Override
    public void saveImportCache(String businessKey, String cacheKey, String dataListJson) {
        if (StringUtils.isEmpty(dataListJson)) {
            return;
        }
        BusinessImportCacheEsDocument entity = new BusinessImportCacheEsDocument();
        entity.setId(UuidCrmUtil.general());
        entity.setTenantCode(TenantUtils.getTenantCode());
        entity.setReqDate(DateUtil.getDateStrByFormat(new Date(),DateUtil.DEFAULT_DATE_ALL_PATTERN));
        entity.setBusinessKey(businessKey);
        entity.setCacheKey(cacheKey);
        entity.setDataListJson(dataListJson);
        repository.save(entity);
    }

    /**
     * 更新缓存
     */
    @Override
    public void updateImportCache(String businessKey, String cacheKey, String dataListJson) {
        if (StringUtils.isEmpty(cacheKey)) {
            return;
        }
        BusinessImportCacheEsDocument cacheEsDocument = getByCacheKey(businessKey, cacheKey);
        cacheEsDocument.setDataListJson(dataListJson);
        repository.save(cacheEsDocument);
    }


    /**
     * 更新缓存
     */
    @Override
    public void saveOrUpdateImportCache(String businessKey, String cacheKey, String dataListJson) {
        if (StringUtils.isEmpty(cacheKey)) {
            return;
        }
        BusinessImportCacheEsDocument cacheEsDocument = getByCacheKey(businessKey, cacheKey);
        if(null == cacheEsDocument){
            saveImportCache(businessKey,cacheKey,dataListJson);
            return;
        }
        cacheEsDocument.setDataListJson(dataListJson);
        repository.save(cacheEsDocument);
    }

    @Override
    public String getDataListJson(String cacheKey) {
        return getDataListJson(null,cacheKey);
    }

    @Override
    public String getDataListJson(String businessKey,String cacheKey) {
        BusinessImportCacheEsDocument cacheEsDocument = getByCacheKey(businessKey, cacheKey);
        if (null != cacheEsDocument){
            return cacheEsDocument.getDataListJson();
        }
        return null;
    }

    @Override
    public BusinessImportCacheEsDocument getByCacheKey(String businessKey, String cacheKey) {
        if (StringUtils.isEmpty(cacheKey)) {
            return null;
        }
        BoolQueryBuilder queryBuilder = QueryBuilders.boolQuery();
        if (StringUtils.isNotEmpty(businessKey)){
            queryBuilder.must(QueryBuilders.queryStringQuery(businessKey).field("businessKey"));
        }
        queryBuilder.must(QueryBuilders.queryStringQuery(cacheKey).field("cacheKey"));
        queryBuilder.must(QueryBuilders.queryStringQuery(TenantUtils.getTenantCode()).field("tenantCode"));
        NativeSearchQuery query = new NativeSearchQuery(queryBuilder);
        SearchHits<BusinessImportCacheEsDocument> search = esTemplate.search(query, BusinessImportCacheEsDocument.class);
        if (search.hasSearchHits()){
            //只有一条数据
            return search.getSearchHit(0).getContent();
        }
        return null;
    }

}
