package com.biz.crm.business.common.page.cache.task;

import com.biz.crm.business.common.base.service.RedisLockService;
import com.biz.crm.business.common.base.util.DateUtil;
import com.biz.crm.business.common.page.cache.constant.BusinessPageCacheConstant;
import com.biz.crm.business.common.page.cache.repository.BusinessImportCacheEsDocument;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.elasticsearch.core.ElasticsearchRestTemplate;
import org.springframework.data.elasticsearch.core.query.ByQueryResponse;
import org.springframework.data.elasticsearch.core.query.NativeSearchQuery;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Calendar;
import java.util.concurrent.TimeUnit;

/**
 * 导入缓存清除日志
 *
 * @author wanghaojia
 * @date 2023/12/21 22:04
 */
@EnableScheduling
@Component
@Slf4j
public class DeleteBusinessImportCacheCornTask implements SchedulingConfigurer {

    /**
     * 应用名称
     */
    @Value("${spring.application.name:}")
    private String applicationName;

    @Resource
    private ElasticsearchRestTemplate elasticsearchRestTemplate;

    @Autowired(required = false)
    private RedisLockService redisLockService;

    /**
     * 清理对外日志应用-必须是 crm-tpm
     */
    private final static String APPLICATION_NAME = "crm-tpm";

    @Override
    public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
        taskRegistrar.addTriggerTask(this::deleteData, triggerContext -> {
            CronTrigger trigger = new CronTrigger("0 0 6 * * ?");
            return trigger.nextExecutionTime(triggerContext);
        });
    }

    /**
     * 清理日志
     */
    private void deleteData() {
        if (!APPLICATION_NAME.equalsIgnoreCase(applicationName)) {
            return;
        }
        String lockKey = BusinessPageCacheConstant.BUSINESS_IMPORT_CACHE_DEL;
        if (redisLockService != null) {
            boolean lock = redisLockService.tryLock(lockKey, TimeUnit.MINUTES, 30);
            if (!lock) {
                return;
            }
        }
        try {
            Calendar instance = Calendar.getInstance();
            instance.add(Calendar.DATE,-1);
            String cleanDate = DateUtil.getDateStrByFormat(instance.getTime(), DateUtil.DEFAULT_DATE_ALL_PATTERN);
            BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
            boolQueryBuilder.must(QueryBuilders.rangeQuery("reqDate").lte(cleanDate));
            NativeSearchQuery deleteQuery = new NativeSearchQuery(boolQueryBuilder);
            ByQueryResponse queryResponse = elasticsearchRestTemplate.delete(deleteQuery, BusinessImportCacheEsDocument.class);
            log.info("导入缓存ES日志清理[{}]及其以前的 清理数量:[{}]条", cleanDate, queryResponse.getDeleted());
        } catch (Exception e) {
            log.error("=====>    导入缓存ES日志清理 异常   <=====");
            log.error(e.getMessage(), e);
        }
        if (redisLockService != null) {
            redisLockService.unlock(lockKey);
        }
    }

}
