package com.biz.crm.business.common.page.cache.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.business.common.base.service.RedisLockService;
import com.biz.crm.business.common.page.cache.constant.BusinessPageCacheConstant;
import com.biz.crm.business.common.page.cache.service.BusinessPageCacheHelper;
import com.biz.crm.business.common.page.cache.service.BusinessPageCacheService;
import com.biz.crm.business.common.sdk.model.Result;
import com.biz.crm.business.common.sdk.service.RedisService;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 分页缓存接口
 *
 * @author wanghaojia
 * @date 2022/12/7 14:54
 */
@Slf4j
public class BusinessPageCacheController<Vo, Dto> {

    @Autowired(required = false)
    protected BusinessPageCacheService<Vo, Dto> businessPageCacheService;

    @Autowired(required = false)
    protected BusinessPageCacheHelper<Vo, Dto> helper;


    @Autowired(required = false)
    private RedisLockService redisLockService;

    /*===============================================================明细列表缓存分页Start============================================================*/

    /**
     * 1、获取明细列表缓存分页接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "获取明细列表缓存分页接口")
    @GetMapping("findCachePageList")
    public Result<Page<Vo>> findCachePageList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                              @ApiParam(name = "dto", value = "查询实体") Dto dto) {
        String pageCacheLockKey = helper.getRedisPageCacheLockKey(cacheKey);
        try {
            Boolean lockFlag = redisLockService.tryLock(pageCacheLockKey, TimeUnit.MINUTES, BusinessPageCacheConstant.DEFAULT_PAGE_CACHE_EXPIRE_TIME);
            Validate.isTrue(lockFlag, "分页数据正在加载,请不要重复点击");
            Page<Vo> page = this.businessPageCacheService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        } finally {
            redisLockService.unlock(pageCacheLockKey);
        }
    }

    /**
     * 2、新增一行接口：保存当前页数据后，在缓存中行首插入一条数据并返回第一页数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @param itemList 要保存的当前页数据
     * @return
     */
    @ApiOperation(value = "新增一行接口，保存当前页数据后，在缓存中行首插入一条数据并返回第一页数据")
    @PostMapping("addItemCache")
    public Result<Page<Vo>> addItemCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                         @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                         @ApiParam(name = "dto", value = "查询实体") Dto dto,
                                         @ApiParam(value = "当前页数据") @RequestBody List<Dto> itemList) {
        try {
            this.businessPageCacheService.addItemCache(cacheKey, itemList);
            Page<Vo> page = this.businessPageCacheService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }


    /**
     * 3、复制行接口：保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "复制行接口，保存当前页数据后，在缓存中行首复制选中数据并返回第一页数据")
    @PostMapping("copyItemListCache")
    public Result<Page<Vo>> copyItemListCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                              @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                              @ApiParam(name = "dto", value = "查询实体") Dto dto,
                                              @ApiParam(value = "当前页数据，包含要复制的行勾选信息") @RequestBody List<Dto> itemList) {
        try {
            this.businessPageCacheService.copyItemListCache(cacheKey, itemList);
            Page<Vo> page = this.businessPageCacheService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 4、保存当前页数据到缓存并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "保存当前页数据到缓存并返回指定页数据接口")
    @PostMapping("saveCurrentPageCache")
    public Result<Page<Vo>> saveCurrentPageCache(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                                 @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                                 @ApiParam(name = "dto", value = "查询实体") Dto dto,
                                                 @ApiParam(value = "当前页数据") @RequestBody List<Dto> saveList) {
        try {
            this.businessPageCacheService.saveCurrentPageCache(cacheKey, saveList);
            Page<Vo> page = this.businessPageCacheService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 7、删除行接口：删除有选中标记的行，更新未标记数据，并返回指定页数据接口
     *
     * @param pageable 分页参数
     * @param cacheKey 缓存key
     * @return
     */
    @ApiOperation(value = "多行删除并返回指定页数据接口")
    @PostMapping("deleteCacheList")
    public Result<Page<Vo>> deleteCacheList(@ApiParam(name = "pageable", value = "分页对象") @PageableDefault(50) Pageable pageable,
                                            @ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey,
                                            @ApiParam(name = "dto", value = "查询实体") Dto dto,
                                            @ApiParam(value = "当前页数据，包含要删除的行勾选信息") @RequestBody List<Dto> itemList) {
        try {
            this.businessPageCacheService.deleteCacheList(cacheKey, itemList);
            Page<Vo> page = this.businessPageCacheService.findCachePageList(pageable, dto, cacheKey);
            return Result.ok(page);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /**
     * 6、清理缓存接口
     *
     * @param cacheKey 缓存key
     */
    @ApiOperation(value = "清理缓存接口")
    @PostMapping("clearCache")
    public Result clearCache(@ApiParam(name = "cacheKey", value = "缓存键") @RequestParam String cacheKey) {
        try {
            this.businessPageCacheService.clearCache(cacheKey);
            return Result.ok();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return Result.error(e.getMessage());
        }
    }

    /*===============================================================明细列表缓存分页End============================================================*/


    /**
     * 验证缓存分页锁是否被释放
     * @param cacheKey
     */
    public void checkPageCacheLock(String cacheKey){
        String pageCacheLockKey = helper.getRedisPageCacheLockKey(cacheKey);
        Boolean lockFlag = redisLockService.isLock(pageCacheLockKey);
        Validate.isTrue(!lockFlag, "明细数据正在加载,请等待加载完成");
    }
}
