package com.biz.crm.common.sequese.local.algorithm.snowflake.service;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.biz.crm.common.sequese.sdk.generator.constant.SequeseConstant;
import com.biz.crm.common.sequese.sdk.generator.service.aigorithm.CrmSequeseGeneratorBySnowFlake;
import com.bizunited.nebula.common.service.redis.RedisMutexService;
import cn.hutool.core.lang.generator.SnowflakeGenerator;
import lombok.extern.slf4j.Slf4j;

/**
 * 雪花算法,返回数值
 * 
 * @author Ken.xu
 * @version 1.0 Copyright 2023年4月23日 下午4:39:37
 */
@Slf4j
@Component
public class CrmSequeseGeneratorBySnowFlakeImpl implements CrmSequeseGeneratorBySnowFlake {
  /**
   * 服务对象
   */
  private Map<String, SnowflakeGenerator> generatorMap = new HashMap<>(16);

  @Autowired
  private RedisMutexService redisMutexService;
  /**
   * 数据中心map
   */
  // XZK TODO 需要放出接口，让其他系统可以扩展数据中心
  private static final Map<String, Integer> dataCenterMap = new HashMap<>(10);
  /**
   * 全部不匹配，则使用默认数据分区
   * 
   */
  public static final String DEFAULT_DATA_CENTER = "default";
  /**
   * 单例类
   */
  private volatile static SnowflakeGenerator generator;


  @Override
  public Long currVal(String subSystem, String bizCode) {
    log.warn("当前算法不支持");
    return null;
  }

  /**
   * 当前算法规则
   * 
   * @return
   */
  @Override
  public String getSequeseRuleCode() {
    return "雪花算法";
  }

  /**
   * 雪花算法，对bizCode不敏感，直接忽略
   */
  @Override
  public Long nextVal(String subSystem, String bizCode) {
    SnowflakeGenerator generator = this.getGeneratorCache(subSystem, bizCode);
    return generator.next();
  }

  @Override
  public Long[] nextValArray(String subSystem, String bizCode, int seqNum) {
    // 数据纠正
    if (seqNum < 1) {
      seqNum = 1;
    }
    SnowflakeGenerator generator = this.getGeneratorCache(subSystem, bizCode);
    // 批量赋值
    Long[] batchNextVal = new Long[seqNum];
    for (int idx = 0; idx < batchNextVal.length; idx++) {
      batchNextVal[idx] = generator.next();
    }
    return batchNextVal;
  }

  @Override
  public void initAlgorithm() {
    dataCenterMap.put(DEFAULT_DATA_CENTER, 0);
    dataCenterMap.put("crm-mdm", 1);
    dataCenterMap.put("crm-dms", 2);
    dataCenterMap.put("crm-sfa", 3);
    dataCenterMap.put("crm-ems", 4);
    dataCenterMap.put("crm-kms", 5);
    dataCenterMap.put("crm-tpm", 6);
    dataCenterMap.put("crm-octm", 7);
    dataCenterMap.put("workflow", 8);
  }

  /** dataCenterId 数据中心id,最大0-31【默认NULL: 0, 1：CRM-MDM:1 CRM-DMS:2。。。。 】 **/
  private int getDataCenterId(String key) {
    Integer dataCenterId = dataCenterMap.get(key);
    if (dataCenterId == null) {
      log.warn("{}未在系统注册，使用默认0数据中心", key);
      dataCenterId = dataCenterMap.get(DEFAULT_DATA_CENTER);
    }
    return dataCenterId;
  }

  private SnowflakeGenerator getSnowflakeGenerator(String key) {

    long dataCenterId = this.getDataCenterId(key);
    long workerId = this.getWorkerId(key);
    // 单例模式-双重校验锁
    if (ObjectUtils.isEmpty(generator)) {
      synchronized (SnowflakeGenerator.class) {
        if (ObjectUtils.isEmpty(generator)) {
          generator = new SnowflakeGenerator(workerId, dataCenterId);
        }
      }
    }
    return generator;
  }

  /**
   * 返回主键,该算法，对bizCode不敏感，雪花算法对租户不敏感，不需要租户信息
   */
  @Override
  public String getKey(String subSystem, String bizCode) {
    return String.format("%s", subSystem.toUpperCase());
  }

  private SnowflakeGenerator getGeneratorCache(String subSystem, String bizCode) {
    String key = getKey(subSystem, null);
    SnowflakeGenerator generator = this.generatorMap.get(key);
    if (generator == null) {
      // 根据dataCenterId创建一个对象，并存入generatorMap
      generator = this.getSnowflakeGenerator(key);
      generatorMap.put(key, generator);
    }
    return generator;
  }

  /**
   * workerId 工作机器节点id,最大0-31【一个服务，最多32个节点，可以用redis锁的形式进行记录，如果无法获取到锁，则尝试获取下一个锁，如果都无法获取，则默认0号节点】
   */
  private long getWorkerId(String subSystem) {
    // 通过redis或nacos注册中心，进行抢占工作节点
    for (int i = 0; i < SequeseConstant.MAX_TOTAL_WORK_ID; i++) {
      log.info("SnowFlake空节点: {}", i);
      // 对节点取余
      i = Math.floorMod(i, SequeseConstant.MAX_WORK_ID);
      String key = String.format(SequeseConstant.WORK_ID_LOCK_KEY, subSystem, i);
      try {
        if (this.redisMutexService.tryLock(key, TimeUnit.SECONDS, 10)) {
          log.info("SnowFlake 当前节点数:{}, 获取机器节点key: {}", i, key);
          return i;
        }
      } catch (Exception ex) {
        log.error(ex.getMessage(), ex);
        throw new RuntimeException(ex.getMessage(), ex);
      }
    }
    return SequeseConstant.DEFAULT_WORK_ID;
  }

}
