package com.biz.crm.common.form.local.service.internal;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.biz.crm.common.form.local.service.DynamicFormServiceOpStrategy;
import com.biz.crm.common.form.local.utils.DynamicFormUtils;
import com.biz.crm.common.form.sdk.DynamicFieldConfiguration;
import com.biz.crm.common.form.sdk.DynamicFormService;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.DynamicFormsOperationStrategy;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.google.common.collect.Maps;

/**
 * 动态表单内容操作信息的标品实现
 * @author yinwenjie
 * @param <T>
 */
public class DynamicFormServiceImpl<T> implements DynamicFormService<T> {
  /**
   * 主业务模型的class信息
   */
  private Class<T> formClass;
  
  @Autowired
  private List<OperationStrategy<? extends DynamicForm>> operationStraties; 
  
  @Autowired(required = false)
  private List<DynamicFormServiceOpStrategy> dynamicFormServiceOpStraties;
  
  /**
   * 已被设定的主模型中的动态表单结构对应的字段/属性信息
   */
  private Map<String , DynamicFieldConfiguration<T>> dynamicFieldConfigurationMapping;
  
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicFormServiceImpl.class);
  
  public DynamicFormServiceImpl(Class<T> formClass , Map<String , DynamicFieldConfiguration<T>> dynamicFieldConfigurationMapping) {
    this.formClass = formClass;
    // 为了保证外部的Map和DynamicFormServiceImpl内部的Map不会出现引用共享的问题
    // 这里做一个Maps的中度副本
    this.dynamicFieldConfigurationMapping = Maps.newLinkedHashMap(dynamicFieldConfigurationMapping);
  }
  
  @Override
  public T dynamic(JSONObject formJson) {
    if(formJson == null || CollectionUtils.isEmpty(dynamicFormServiceOpStraties)) {
      return null;
    }
    // 1、======== 转换主模型对象
    T mainForm = JSON.parseObject(formJson.toJSONString(), formClass);
    
    // 2、========
    // 这是主模型上，所有已Map形式存在的需要进行动态表单结构处理的字段名信息
    Set<String> dynamicFieldCodes = dynamicFieldConfigurationMapping.keySet();
    for (String dynamicFieldCode : dynamicFieldCodes) {
      Field dynamicField = this.findDynamicField(dynamicFieldCode);
      if(dynamicField == null) {
        continue;
      }
      
      // 2.1、======= 根据字段/属性，对应的DynamicFieldConfiguration信息，找出每个Key对应的Value值；
      // 找到正确的DynamicFormServiceOpStrategy，服务操作策略
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      if(dynamicKeys == null || dynamicKeys.length == 0) {
        continue;
      }
      Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
      Class<?> collectionClass = dynamicFieldConfiguration.getCollectionClass();
      JSONObject dynamicFieldValueMapping = formJson.getJSONObject(dynamicFieldCode);
      Validate.notNull(dynamicFieldValueMapping , "错误的动态字段指定（可能并没有传入）：%s，请检查!!" , dynamicFieldCode);
      DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = this.findDynamicFormServiceOpStrategy(this.dynamicFormServiceOpStraties, isValueCollection, collectionClass);
      if(currentDynamicFormServiceOpStrategy == null) {
        LOGGER.warn("进行动态模型数据转换时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json为数组），请检查!!");
        continue;
      }
      Map<String , ?> targetMapValues = currentDynamicFormServiceOpStrategy.createMapbeforeOp();
      
      // 2.2、======== 开始逐条构建匹配的对象信息
      // 首先从dynamicFieldConfiguration中，找到各种计算的前置要素
      for (String dynamicKey : dynamicKeys) {
        JSON mappingValue = null;
        if(!isValueCollection) {
          mappingValue = dynamicFieldValueMapping.getJSONObject(dynamicKey);
        } else {
          mappingValue = dynamicFieldValueMapping.getJSONArray(dynamicKey);
        }
        if(mappingValue == null) {
          continue;
        }
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
        if(operationStrategy == null) {
          continue;
        }
        Class<? extends DynamicForm> targetDynamicFormClass = operationStrategy.dynamicFormClass();
        // 开始正式进行转换，注意，根据isValueCollection分成两种场景
        currentDynamicFormServiceOpStrategy.convert(targetMapValues, mainForm, dynamicField, dynamicKey, mappingValue, targetDynamicFormClass, isValueCollection, collectionClass);
      }
      
      // 2.4、====== 写入主模型对应的字段/属性
      try {
        if(!CollectionUtils.isEmpty(targetMapValues))  {
          dynamicField.set(mainForm, targetMapValues);
        }
      } catch (IllegalArgumentException | IllegalAccessException e) {
        LOGGER.error(e.getMessage() , e);
        continue;
      }
    }
    
    // 最后进行返回
    return mainForm;
  }

  private DynamicFormServiceOpStrategy findDynamicFormServiceOpStrategy(List<DynamicFormServiceOpStrategy> dynamicFormServiceOpStraties , 
                                                                        boolean isValueCollection, 
                                                                        Class<?> collectionClass) { 
    if(CollectionUtils.isEmpty(dynamicFormServiceOpStraties)) { 
      return null;
    }
    // 排一下序
    List<DynamicFormServiceOpStrategy> sortedDynamicFormServiceOpStraties = dynamicFormServiceOpStraties.stream().sorted((source , target) -> source.sort() - target.sort()).collect(Collectors.toList());
    DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = null;
    for (DynamicFormServiceOpStrategy dynamicFormServiceOpStrategy : sortedDynamicFormServiceOpStraties) { 
      if(dynamicFormServiceOpStrategy.support(isValueCollection, collectionClass)) { 
        currentDynamicFormServiceOpStrategy = dynamicFormServiceOpStrategy;
      } 
    } 
    return currentDynamicFormServiceOpStrategy;
  } 
  
  /**
   * 根据字段名，寻找字段信息
   * @param dynamicFieldCode
   * @return
   */
  private Field findDynamicField(String dynamicFieldCode) {
    Field dynamicField = null;
    try {
      dynamicField = formClass.getDeclaredField(dynamicFieldCode);
      dynamicField.setAccessible(true);
    } catch (NoSuchFieldException | SecurityException e) {
      LOGGER.error("没有找到指定的字段信息: {}" , dynamicFieldCode);
      LOGGER.error(e.getMessage() , e);
      return null;
    }
    return dynamicField;
  }
  
  @Override
  @Transactional
  public void createDynamicDetails(T formObject, String parentCode) {
    this.saveDynamicDetails(formObject, parentCode, false);
  }

  @SuppressWarnings("unchecked")
  private void saveDynamicDetails(T formObject, String parentCode , boolean isModify) {
    Validate.isTrue(this.formClass == formObject.getClass() , "当前正在保存的主业务数据类型和指定的DynamicFormService所服务的主业务数据类型不匹配，请检查!!");
    Validate.notNull(formObject , "当前正在保存的主业务数据为空，请检查!!");
    Validate.notBlank(parentCode , "当前正在保存的主业务数据，没有设定其与各子级动态表单数据的关联信息“parentCode”，请检查!!");
    /*
     * 当formObject主业务模型完成创建/修改过程，并调用本方法试图进行主模型结构中动态表单结构的数据创建时，
     * 处理过程如下：
     * 1、取得当前主业务模型下，所有需要进行做动态表单模型处理的字段/属性
     * 2、由于本功能支持一个主业务模型中存在多个抽象的Map形式的字段，所以需要一个一个字段进行处理
     *   2.1、根据字段/属性，对应的DynamicFieldConfiguration信息，找出每个Key对应的Value值
     *   2.2、开始逐条创建动态表单信息
     * */
    
    // 1、=======
    Set<String> dynamicFieldCodes = dynamicFieldConfigurationMapping.keySet();
    if(CollectionUtils.isEmpty(dynamicFieldCodes)) {
      return;
    }
    
    // 2、=======
    for (String dynamicFieldCode : dynamicFieldCodes) {
      Field dynamicField = this.findDynamicField(dynamicFieldCode);
      if(dynamicField == null) {
        continue;
      }
      // 获取数据，一定是一种Map性质的数据
      Map<String , ?> dynamicFieldValue = null;
      try {
        dynamicFieldValue = (Map<String , ?>)dynamicField.get(formObject);
        if(CollectionUtils.isEmpty(dynamicFieldValue)) {
          continue;
        }
      } catch (IllegalArgumentException | IllegalAccessException e) {
        LOGGER.error(e.getMessage() , e);
        continue;
      }
      
      // 2.1、======= 
      // 注意，根据DynamicFieldConfiguration中的设定，这个value值可能是对象，也可能是集合
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      if(dynamicKeys == null || dynamicKeys.length == 0) {
        continue;
      }
      Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
      Class<?> collectionClass = dynamicFieldConfiguration.getCollectionClass();
      DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = this.findDynamicFormServiceOpStrategy(this.dynamicFormServiceOpStraties, isValueCollection, collectionClass);
      if(currentDynamicFormServiceOpStrategy == null) {
        LOGGER.warn("进行动态模型数据保存时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json为数组），请检查!!");
        continue;
      }
      
      // 2.2、======== 
      // 首先从dynamicFieldConfiguration中，找到各种计算的前置要素
      for (String dynamicKey : dynamicKeys) {
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
        if(operationStrategy == null) {
          continue;
        }
        Object dynamicKeyObject = dynamicFieldValue.get(dynamicKey);
        if(!isModify) {
          // 没有对象信息，就不用进行新增
          if(dynamicKeyObject == null) {
            continue;
          }
          currentDynamicFormServiceOpStrategy.createOp(operationStrategy, dynamicKey, dynamicKeyObject, formObject, parentCode);
        } else {
          currentDynamicFormServiceOpStrategy.modifyOp(operationStrategy, dynamicKey, dynamicKeyObject, formObject, parentCode);
        }
      }
      
      // 2.3、====== TODO 确定一下主业务模型的关联性是否建立好了
    }
  }
  
  @Override
  @Transactional
  public void modifyDynamicDetails(T formObject , String parentCode) {
    this.saveDynamicDetails(formObject, parentCode, true); 
  }

  @Override
  @Transactional
  public void deleteDynamicDetails(String parentCode) {
    Validate.notBlank(parentCode , "全面删除时，发现错误的主业务关联编号信息，请检查!!");
    Set<String> dynamicFieldCodes = dynamicFieldConfigurationMapping.keySet();
    if(CollectionUtils.isEmpty(dynamicFieldCodes)) {
      return;
    }
    
    for (String dynamicFieldCode : dynamicFieldCodes) {
      Field dynamicField = this.findDynamicField(dynamicFieldCode);
      if(dynamicField == null) {
        continue;
      }
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      if(dynamicKeys == null || dynamicKeys.length == 0) {
        continue;
      }
      
      // 一个一个的删除
      // 全部删除过程比较简单，所以不涉及使用DynamicFormServiceOpStrategy进行策略封装
      for (String dynamicKey : dynamicKeys) {
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
        if(operationStrategy == null) {
          continue;
        } 
        operationStrategy.onDynamicFormsDelete(parentCode, dynamicKey);
      } 
    } 
  } 

  @SuppressWarnings("unchecked")
  @Override
  @Transactional
  public void deleteDynamicDetails(String parentCode , String dynamicFieldCode, String dynamicKey, String[] detailCodes) {
    Validate.notBlank(parentCode , "部分删除时，发现错误的主业务关联编号信息，请检查!!");
    Validate.notBlank(dynamicFieldCode , "部分删除时，发现错误的主业务属性/字段名信息，请检查!!");
    Validate.notBlank(dynamicKey , "部分删除时，发现错误的主业务属性/字段名信息下的业务号信息，请检查!!");
    Validate.isTrue(detailCodes != null && detailCodes.length > 0 , "部分删除时，至少需要指定一条需要删除的信息，请检查!!");
    Set<String> dynamicFieldCodes = dynamicFieldConfigurationMapping.keySet();
    if(CollectionUtils.isEmpty(dynamicFieldCodes)) {
      return;
    }
    
    // 注意，只对要求的dynamicFieldCode（主模型的动态属性/字段名进行删除）
    for (String dynamicFieldItem : dynamicFieldCodes) {
      if(!StringUtils.equals(dynamicFieldCode, dynamicFieldItem)) {
        continue;
      }
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      if(dynamicKeys == null || dynamicKeys.length == 0) {
        continue;
      }
      
      // 注意，只删除对应的dynamicKey信息下的部分数据
      for (String dynamicKeyItem : dynamicKeys) {
        if(!StringUtils.equals(dynamicKey, dynamicKeyItem)) {
          continue;
        }
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
        OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
        if(operationStrategy == null || !isValueCollection) {
          continue;
        } 
        // 注意，只能是DynamicFormOperationStrategy性质的操作策略
        DynamicFormsOperationStrategy<DynamicForm> dynamicFormsOperationStrategy = (DynamicFormsOperationStrategy<DynamicForm>)operationStrategy;
        dynamicFormsOperationStrategy.onDynamicFormsDelete(parentCode, dynamicKey, detailCodes);
      }
    }
  }

  @Override
  public void perfectDynamicDetails(T formObject, String parentCode) {
    /*
     * 对传入的主业务模型下面的各个Map形式的抽象属性/字段，进行数据完善的操作
     * */
    if(formObject == null || StringUtils.isBlank(parentCode)) {
      return;
    }
    Set<String> dynamicFieldCodes = dynamicFieldConfigurationMapping.keySet();
    if(CollectionUtils.isEmpty(dynamicFieldCodes)) {
      return;
    }
    
    // 开始一个字段一个字段的进行完善
    for (String dynamicFieldCode : dynamicFieldCodes) {
      Field dynamicField = this.findDynamicField(dynamicFieldCode);
      if(dynamicField == null) {
        continue;
      }
      dynamicField.setAccessible(true);
      
      // 1、======= 根据dynamicFieldCode调取字段配置信息
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      if(dynamicKeys == null || dynamicKeys.length == 0) {
        continue;
      }
      Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
      Class<?> collectionClass = dynamicFieldConfiguration.getCollectionClass();
      DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = this.findDynamicFormServiceOpStrategy(this.dynamicFormServiceOpStraties, isValueCollection, collectionClass);
      if(currentDynamicFormServiceOpStrategy == null) {
        LOGGER.warn("进行动态模型数据完善时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json为数组），请检查!!");
        continue;
      }
      
      // 2、======== 调用对应的 operationStrategy进行具体值的读取
      for (String dynamicKey : dynamicKeys) {
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
        if(operationStrategy == null) {
          continue;
        }
        currentDynamicFormServiceOpStrategy.findAndMergeByParentCode(operationStrategy , dynamicKey, dynamicField, formObject, parentCode);
      }
    }
  }

  @SuppressWarnings("unchecked")
  @Override
  public <D> D findDetailByParentCodeAndDynamicFieldAndDynamicKey(String parentCode, String dynamicFieldCode, String dynamicKey, Class<D> dclass) {
    if(StringUtils.isAnyBlank(parentCode , dynamicFieldCode , dynamicKey) || dclass == null) {
      return null;
    }
    
    // 1、====== 根据dynamicFieldCode调取字段配置信息
    DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
    String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
    if(dynamicKeys == null || dynamicKeys.length == 0) {
      return null;
    }
    Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
    if(isValueCollection) {
      LOGGER.warn("进行动态模型数据查询时，发现“" + dynamicFieldCode + "”字段的类型，不能为集合，请检查!!");
      return null;
    }
    Class<?> collectionClass = dynamicFieldConfiguration.getCollectionClass();
    DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = this.findDynamicFormServiceOpStrategy(this.dynamicFormServiceOpStraties, isValueCollection, collectionClass);
    if(currentDynamicFormServiceOpStrategy == null) {
      LOGGER.warn("进行动态模型数据查询时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json为数组），请检查!!");
      return null;
    }
    // 验证传入的dynamicKey是否匹配
    if(StringUtils.indexOfAny(dynamicKey, dynamicKeys) == -1) {
      return null;
    }
    
    // 2、===== 取得值，进行转换，并在验证后进行返回
    String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
    OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
    if(operationStrategy == null) {
      return null;
    }
    Validate.isTrue(dclass.isAssignableFrom(operationStrategy.dynamicFormClass()) , "进行动态模型数据查询时，发现要求返回的类型和查询出的数据类型不一致，请检查!!");    
    D result = (D)currentDynamicFormServiceOpStrategy.findByParentCode(operationStrategy, dynamicKey, parentCode);
    return result;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <D> Collection<D> findDetailsByParentCodeAndDynamicFieldAndDynamicKey(String parentCode, String dynamicFieldCode, String dynamicKey, Class<D> dclass) {
    if(StringUtils.isAnyBlank(parentCode , dynamicFieldCode , dynamicKey) || dclass == null) {
      return null;
    }
    
    // 1、====== 根据dynamicFieldCode调取字段配置信息
    DynamicFieldConfiguration<T> dynamicFieldConfiguration = dynamicFieldConfigurationMapping.get(dynamicFieldCode);
    String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
    if(dynamicKeys == null || dynamicKeys.length == 0) {
      return null;
    }
    Boolean isValueCollection = dynamicFieldConfiguration.isValueCollection();
    if(!isValueCollection) {
      LOGGER.warn("进行动态模型数据查询时，发现“" + dynamicFieldCode + "”字段的类型为集合，请检查!!");
      return null;
    }
    Class<?> collectionClass = dynamicFieldConfiguration.getCollectionClass();
    DynamicFormServiceOpStrategy currentDynamicFormServiceOpStrategy = this.findDynamicFormServiceOpStrategy(this.dynamicFormServiceOpStraties, isValueCollection, collectionClass);
    if(currentDynamicFormServiceOpStrategy == null) {
      LOGGER.warn("进行动态模型数据查询时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json为数组），请检查!!");
      return null;
    }
    // 验证传入的dynamicKey是否匹配
    if(StringUtils.indexOfAny(dynamicKey, dynamicKeys) == -1) {
      return null;
    }
    
    // 2、===== 取得值，进行转换，并在验证后进行返回
    String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
    OperationStrategy<? extends DynamicForm> operationStrategy = DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties);
    if(operationStrategy == null) {
      return null;
    }
    Validate.isTrue(dclass.isAssignableFrom(operationStrategy.dynamicFormClass()) , "进行动态模型数据查询时，发现要求返回的类型和查询出的数据类型不一致，请检查!!");    
    Collection<D> results = (Collection<D>)currentDynamicFormServiceOpStrategy.findByParentCode(operationStrategy, dynamicKey, parentCode);
    return results;
  }
}
