package com.biz.crm.common.form.local.service.op;

import java.lang.reflect.Field;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.biz.crm.common.form.local.service.DynamicFormServiceOpStrategy;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.DynamicFormsOperationStrategy;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * 该服务操作策略用于处理Map结构中，其Value部分为List性质的集合对象的情况
 * @author yinwenjie
 */
@Component
public class ListDynamicFormServiceOpStrategy extends AbstractDynamicFormServiceOpStrategy implements DynamicFormServiceOpStrategy {
  /**
   * 日志
   */
  private static final Logger LOGGER = LoggerFactory.getLogger(ListDynamicFormServiceOpStrategy.class);
  
  @Override
  public int sort() {
    return 98;
  }

  @Override
  public boolean support(boolean isValueCollection, Class<?> collectionClass) {
    // 如果主模型上映射动态表单结构的字段/属性，其Map类的Value部分，是一个List性质的集合。
    // 则本处理策略可以进行支持
    return isValueCollection && List.class.isAssignableFrom(collectionClass);
  }
  
  @Override
  public <T> Map<String , T> createMapbeforeOp() {
    Map<String , T> targetMapValueList = Maps.newLinkedHashMap();
    return targetMapValueList;
  } 
  
  @SuppressWarnings("unchecked")
  @Override
  public <T> void convert(Map<String , T> targetMapValueList , Object mainForm, Field dynamicField , String dynamicKey
                          , JSON jsonValue, Class<? extends DynamicForm> targetDynamicFormClass 
                          , boolean isValueCollection, Class<?> collectionClass) {
    
    String dynamicFieldCode = dynamicField.getName();
    if(!isValueCollection || !(jsonValue instanceof JSONArray)) {
      LOGGER.warn("进行动态模型数据转换时，发现“" + dynamicFieldCode + "”的类型和json结构的值不能对应（json不为数组），请检查!!");
      return;
    } 
    
    // 开始正式进行转换并完成赋值
    T targetValues = (T)JSON.parseArray(jsonValue.toJSONString(), targetDynamicFormClass);
    Field[] declaredFields = targetDynamicFormClass.getDeclaredFields();
    // 开始进行验证
    if(Collection.class.isAssignableFrom(targetValues.getClass())) {
      Collection<?> collections = (Collection<?>)targetValues;
      for (Object collectionItem : collections) {
        // 一般不会出现这种情况，但为了保证边界，还是做了判定
        if(!DynamicForm.class.isAssignableFrom(collectionItem.getClass())) {
          continue;
        }
        for (Field declaredField : declaredFields) {
          declaredField.setAccessible(true);
          this.fieldValidate(declaredField, (DynamicForm)collectionItem, mainForm);
        }
      }
    }
    
    targetMapValueList.put(dynamicKey, targetValues);
  }

  @SuppressWarnings("unchecked")
  @Override
  public void createOp(OperationStrategy<? extends DynamicForm> operationStrategy , 
                           String dynamicKey , Object dynamicKeyObject ,
                           Object formObject , String parentCode) {
    Validate.isTrue(DynamicFormsOperationStrategy.class.isAssignableFrom(operationStrategy.getClass()), "不正确的动态表单(创建)操作策略，请检查!!");
    DynamicFormsOperationStrategy<DynamicForm> dynamicFormsOperationStrategy = (DynamicFormsOperationStrategy<DynamicForm>)operationStrategy;
    List<DynamicForm> dynamicForms = (List<DynamicForm>)dynamicKeyObject;
    dynamicFormsOperationStrategy.onDynamicFormsCreate(dynamicForms,  dynamicKey, parentCode, formObject);
  }

  @SuppressWarnings("unchecked")
  @Override
  public void modifyOp(OperationStrategy<? extends DynamicForm> operationStrategy,
      String dynamicKey, Object dynamicKeyObject, Object formObject, String parentCode) {
    Validate.isTrue(DynamicFormsOperationStrategy.class.isAssignableFrom(operationStrategy.getClass()), "不正确的动态表单(修改)操作策略，请检查!!");
    DynamicFormsOperationStrategy<DynamicForm> dynamicFormsOperationStrategy = (DynamicFormsOperationStrategy<DynamicForm>)operationStrategy;
    List<DynamicForm> dynamicForms = (List<DynamicForm>)dynamicKeyObject;
    dynamicFormsOperationStrategy.onDynamicFormsModify(dynamicForms,  dynamicKey, parentCode, formObject);
  }
  
  @SuppressWarnings("unchecked")
  @Override
  public Object findByParentCode(OperationStrategy<? extends DynamicForm> operationStrategy, String dynamicKey, String parentCode) {
    // 使用该操作策略进行处理的，一定是主业务模型数据中，Map性质的字段其Value部分为List的情况
    Validate.isTrue(DynamicFormsOperationStrategy.class.isAssignableFrom(operationStrategy.getClass()), "不正确的动态表单(查询/完善)操作策略，请检查!!");
    DynamicFormsOperationStrategy<DynamicForm> dynamicFormsOperationStrategy = (DynamicFormsOperationStrategy<DynamicForm>)operationStrategy;
    Collection<DynamicForm> results = dynamicFormsOperationStrategy.findByParentCode(dynamicKey , parentCode);
    return results;
  }

  @SuppressWarnings("unchecked")
  @Override
  public void findAndMergeByParentCode(OperationStrategy<? extends DynamicForm> operationStrategy,
      String dynamicKey, Field dynamicField, Object formObject, String parentCode) {
    Collection<DynamicForm> results = (Collection<DynamicForm>)this.findByParentCode(operationStrategy, dynamicKey, parentCode);
    if(CollectionUtils.isEmpty(results)) {
      return;
    }
    
    // 以K-V形式，设置到主模型上面
    Map<String , List<DynamicForm>> dynamicFieldValue = null;
    try {
      dynamicFieldValue = (Map<String , List<DynamicForm>>)dynamicField.get(formObject);
      if(dynamicFieldValue == null) {
        dynamicFieldValue = Maps.newLinkedHashMap();
        dynamicField.set(formObject, dynamicFieldValue);
      }
    } catch (IllegalArgumentException | IllegalAccessException e) {
      LOGGER.error(e.getMessage() , e);
      return;
    }
    
    // 一定是一个Map，且Value部分为List;
    dynamicFieldValue.put(dynamicKey, Lists.newArrayList(results));
  }
}
