package com.biz.crm.common.form.local.builder;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Triple;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;

import com.biz.crm.common.form.local.utils.DynamicFormUtils;
import com.biz.crm.common.form.sdk.DynamicFieldConfiguration;
import com.biz.crm.common.form.sdk.DynamicFormService;
import com.biz.crm.common.form.sdk.DynamicFormServiceBuilder;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.OperationStrategy;
import com.google.common.collect.Maps;

public class DynamicFormServiceLocalBuilder<T> implements DynamicFormServiceBuilder<T> {

  /**
   * 主业务模型的class信息
   */
  private Class<T> formClass;
  
  /**
   * 已被设定的主模型中的动态表单结构对应的字段/属性信息
   * key：表示主模型中，某个需要进行动态表单结构处理的字段/属性 名
   * Value：这个需要进行动态表单结构处理的字段/属性，所设定的动态表单结构配置信息
   */
  private Map<String , DynamicFieldConfiguration<T>> dynamicFieldConfigurationMapping = Maps.newLinkedHashMap();
  
  private ApplicationContext applicationContext;
  
  @Autowired
  private List<OperationStrategy<? extends DynamicForm>> operationStraties; 
  
  private static final Logger LOGGER = LoggerFactory.getLogger(DynamicFormServiceLocalBuilder.class);
  
  public DynamicFormServiceLocalBuilder(Class<T> formClass , ApplicationContext applicationContext) {
    this.applicationContext = applicationContext;
    Validate.notNull(applicationContext , "未发现application服务上下文!!");
    this.formClass = formClass;
  }
  
  @Override
  public <R> DynamicFieldConfiguration<T> dynamicField(String field) {
    Validate.notBlank(field , "生成动态表单服务时，构建者必须传入正确的field信息");
    Field declaredField = null;
    try {
      declaredField = formClass.getDeclaredField(field);
    } catch (NoSuchFieldException | SecurityException e) {
      LOGGER.error(e.getMessage() , e);
      throw new IllegalArgumentException("生成动态表单服务时，构建者未找到正确的field字段信息，请检查", e);
    }
    
    // 字段类型必须是map性质
    Triple<Class<?>, Boolean, Class<?> > fieldAnalysisResult = DynamicFormUtils.analysisMapFieldType(declaredField);
    Class<?> valueClass = fieldAnalysisResult.getLeft();
    Boolean isValueCollection = fieldAnalysisResult.getMiddle();
    Class<?> collectionClass = fieldAnalysisResult.getRight();
    
    // 对DynamicFieldConfiguration进行构建
    DynamicFieldLocalConfiguration dynamicFieldLocalConfiguration = new DynamicFieldLocalConfiguration(this, field, valueClass, isValueCollection , collectionClass);
    dynamicFieldConfigurationMapping.put(field, dynamicFieldLocalConfiguration);
    return dynamicFieldLocalConfiguration;
  }

  @SuppressWarnings("unchecked")
  @Override
  public DynamicFormService<T> build() {
    /*
     * 构造步骤为：
     * 1、首选检测主模型，是否至少有一个字段设定了动态表单结构
     * 对于每一个动态表单结构，都要检查其设定的关联信息是否存在
     * 2、接着，构建为这个带有动态表单结构的主模型服务的DynamicFormService
     * */
    
    // 1、========
    Validate.isTrue(!CollectionUtils.isEmpty(dynamicFieldConfigurationMapping) , "生成动态表单服务时，至少设定一个主模型上的字段/属性，为动态表单模型结构!!");
    for (Map.Entry<String,DynamicFieldConfiguration<T>> entrySetItem : dynamicFieldConfigurationMapping.entrySet()) {
      DynamicFieldConfiguration<T> dynamicFieldConfiguration = entrySetItem.getValue();
      String field = dynamicFieldConfiguration.getField();
      Validate.notBlank(field , "生成动态表单服务时，发现至少一个主模型设置项没有正确设定字段/属性信息");
      String[] dynamicKeys = dynamicFieldConfiguration.findDynamicKeys();
      Validate.isTrue(dynamicKeys != null && dynamicKeys.length > 0 , "生成动态表单服务时，发现至少一个主模型设置设定的字段/属性信息没有设定映射信息，请检查!!");
      
      for (String dynamicKey : dynamicKeys) {
        String dynamicFormCode = dynamicFieldConfiguration.findDynamicFormCodeByDynamicKey(dynamicKey);
        Validate.notBlank(dynamicFormCode , "生成动态表单服务时，发现至少一个字段/属性，设定的映射信息，找不到映射值，请检查!!");
        Validate.notNull(DynamicFormUtils.findByDynamicFormCode(dynamicFormCode, operationStraties) , "生成动态表单服务时，未发现匹配的处理策略（%s）" , dynamicFormCode);
      }
    }
    
    // 2、=======
    DynamicFormService<T> dynamicFormService = this.applicationContext.getBean(DynamicFormService.class , this.formClass , dynamicFieldConfigurationMapping);
    return dynamicFormService;
  }

  public class DynamicFieldLocalConfiguration implements DynamicFieldConfiguration<T> {
    
    private DynamicFormServiceLocalBuilder<T> dynamicFormServiceBuilder;
    /**
     * 字段的英文名（这种字段必须是Map性质的字段）
     */
    private String field;
    /**
     * 这个Map性质的字段，对应的value的类型
     */
    private Class<?> valueClass;
    /**
     * 这个Map性质的字段，其value部分是否为集合
     */
    private Boolean isValueCollection;
    /**
     * 如果isValueCollection为true,那么这里记录集合的类型（注意，可能不是某种具体的类型）
     */
    private Class<?> collectionClass;
    
    private Map<String , String> dynamicMapping = Maps.newLinkedHashMap();
    
    public DynamicFieldLocalConfiguration(DynamicFormServiceLocalBuilder<T> dynamicFormServiceBuilder , 
                                          String field ,
                                          Class<?> valueClass , 
                                          Boolean isValueCollection, 
                                          Class<?> collectionClass) {
      this.dynamicFormServiceBuilder = dynamicFormServiceBuilder;
      this.field = field;
      this.valueClass = valueClass;
      this.isValueCollection = isValueCollection;
      this.collectionClass = collectionClass;
    }
    
    @Override
    public DynamicFieldConfiguration<T> addDynamicMapping(String dynamicKey, String dynamicFormCode) {
      if(StringUtils.isAnyBlank(dynamicKey , dynamicFormCode)) {
        throw new IllegalArgumentException("错误的入参信息，dynamicKey和dynamicFormCode必须都进行填写");
      }
      this.dynamicMapping.put(dynamicKey, dynamicFormCode);
      return this;
    }

    @Override
    public DynamicFormServiceBuilder<T> config() {
      return this.dynamicFormServiceBuilder;
    }

    @Override
    public String findDynamicFormCodeByDynamicKey(String dynamicKey) {
      return this.dynamicMapping.get(dynamicKey);
    }
    
    @Override
    public String[] findDynamicKeys() {
      Set<String> keys = this.dynamicMapping.keySet();
      if(CollectionUtils.isEmpty(keys)) {
        return null;
      }
      return keys.toArray(new String[] {});
    }
    
    public String getField() {
      return field;
    }

    public Class<?> getValueClass() {
      return valueClass;
    }

    public boolean isValueCollection() {
      return isValueCollection;
    }

    public Class<?> getCollectionClass() {
      return collectionClass;
    }
  }
}
