package com.biz.crm.common.form.local.service;

import java.lang.reflect.Field;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.biz.crm.common.form.sdk.model.DynamicForm;
import com.biz.crm.common.form.sdk.model.OperationStrategy;

/**
 * 这是动态表单结构标品处理逻辑中需要的一个操作策略器</br>
 * 由于用户可以定义千奇百怪的Map性质的字段，所以为了解决这些字段的特异性，DynamicFormServiceImpl需要些很多if，</br>
 * 而这些if放置在将会堵死代码的扩展性，所以改用策略模式，以便放开DynamicFormServiceImpl的扩展性
 * @author yinwenjie
 *
 */
public interface DynamicFormServiceOpStrategy {
  
  /**该操作策略的匹配优先级，值越小匹配优先级就越高
   * @return
   */
  public int sort();
  
  /**
   * 该方法的返回值，将确定当前的动态表单模型的处理过程，本操作策略是否能够支持
   * @param isValueCollection 当前主业务模型上映射动态表单结构的字段/属性，一定是一个Map结构，其Key也一定是一个字符串性质的类型。但关键是其Value可能是DynamicForm类型本身，也可能是一个集合；而本字段就是在说明是否是一个集合
   * @param collectionClass 如果是一个集合，那么该入参就是在指示集合的具体类型；否则该参数为null
   * @return
   */
  public boolean support(boolean isValueCollection , Class<?> collectionClass);
  
  /**
   * 如果support返回为true，该方法会在convert方法前运行，用于创建一个最终存储转换数据的具体Map性质的集合
   * @return
   */
  public <T> Map<String , T> createMapbeforeOp();
  
  /**
   * 如果support返回的是true，那么该方法在进行JSON->动态模型转换操作时的实现就会被触发。</br>
   * 在该方法中，操作者需要将一段json信息（注意，可能是json数组，也可能就是单个的json对象）转换为对应的动态表单数据信息，并设定到主模型中
   * @param mainForm 主模型对象（可能是VO，也可能是DTO，等任意性质的对象）
   * @param dynamicField 主模型上关联动态表单结构信息的字段/属性
   * @param jsonValue
   * @param targetDynamicFormClass 动态表单结构的目标类型
   * @param isValueCollection 这个字段/属性一定是一个Map，但其Value部分是否是一个集合，就需要该属性进行指定了
   * @param collectionClass 如果是一个集合，那么该入参就是在指示集合的具体类型；否则该参数为null
   */
  public <T> void convert(Map<String , T> targetMap , Object mainForm , Field dynamicField , String dynamicKey , JSON jsonValue , Class<? extends DynamicForm> targetDynamicFormClass , boolean isValueCollection , Class<?> collectionClass);
  
  /**
   * 如果support返回的是true，那么该方法在主模型mainForm创建完毕，并进行其下各个Map性质的字段/属性创建时，会被触发。</br>
   * 以便操作者能够完成每一个/一组动态表单数据的保存过程
   * @param operationStrategy 当前保存过程匹配的操作策略
   * @param dynamicKey Map性质的抽象字段/属性，其中已经设定的一个key信息
   * @param dynamicKeyObject 这个key信息对应的value信息，注意，可能是单个对象，也可能是某种collection性质的集合。但是一定会有值
   * @param formObject 主业务模型对象
   * @param parentCode 主业务模型和本次正在保存的这个/这组抽象数据的关联业务编号
   */
  public void createOp(OperationStrategy<? extends DynamicForm> operationStrategy ,
                           String dynamicKey , Object dynamicKeyObject ,
                           Object formObject , String parentCode);
  
  /**
   * 其意义和createOp方法意义相似，唯一不同的是，该方法是在主模型的修改场景中使用的
   */
  public void modifyOp(OperationStrategy<? extends DynamicForm> operationStrategy ,
                       String dynamicKey , Object dynamicKeyObject ,
                       Object formObject , String parentCode);
  
  /**
   * 查询动态表单数据（注意可能是单个对象，也可能是一个集合），并直接返回给调用者
   * @param operationStrategy 当前保存过程匹配的操作策略
   * @param dynamicKey 指定的动态表单key
   * @param parentCode 主业务数据模型上的可以唯一参考的标识编号
   * @return 注意，这个返回值可能为null，可能是一个标准的对象，也可能是一个集合对象
   */
  public Object findByParentCode(OperationStrategy<? extends DynamicForm> operationStrategy, String dynamicKey, String parentCode);
  
  /**
   * 查询动态表单数据（注意可能是单个对象，也可能是一个集合），并且组装到主数据表单的特定字段中
   * @param operationStrategy 当前保存过程匹配的操作策略
   * @param dynamicKey 指定的动态表单key
   * @param dynamicField 动态表单在
   * @param formObject 主业务数据模型对象
   * @param parentCode 主业务数据模型上的可以唯一参考的标识编号
   */
  public void findAndMergeByParentCode(OperationStrategy<? extends DynamicForm> operationStrategy , String dynamicKey , Field dynamicField , Object formObject , String parentCode);
}
