package com.biz.crm.common.form.local.service.internal;

import com.biz.crm.common.form.sdk.dto.DynamicFormFieldMappingRebindDto;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.transaction.Transactional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.biz.crm.common.form.local.entity.DynamicFormFieldMapping;
import com.biz.crm.common.form.local.repository.DynamicFormFieldMappingRepository;
import com.biz.crm.common.form.sdk.DynamicFormFieldMappingService;
import com.biz.crm.common.form.sdk.model.OperationStrategyService;
import com.biz.crm.common.form.sdk.vo.DynamicChildrenFormVo;
import com.biz.crm.common.form.sdk.vo.DynamicFieldVo;
import com.biz.crm.common.form.sdk.vo.DynamicFormVo;
import com.biz.crm.common.form.sdk.vo.IDynamicForm;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

/**
 * @author yinwenjie
 */
@Service
public class DynamicFormFieldMappingServiceImpl implements DynamicFormFieldMappingService {
  
  @Autowired
  private OperationStrategyService operationStrategyService;

  @Autowired
  private DynamicFormFieldMappingRepository dynamicFormFieldMappingRepository;
  
  @Override
  public DynamicFormVo findByDynamicFormCode(String dynamicFormCode) {
    if(StringUtils.isBlank(dynamicFormCode)) {
      return null;
    }
    return this.operationStrategyService.findByDynamicFormCode(dynamicFormCode);
  }

  @Override
  public DynamicFormVo findByDynamicFormCodeAndMappingCode(String dynamicFormCode, String mappingCode) {
    if(StringUtils.isAnyBlank(dynamicFormCode , mappingCode)) {
      return null;
    }
    // 如果不存在动态表单，映射关系没有意义
    DynamicFormVo dynamicForm = this.operationStrategyService.findByDynamicFormCode(dynamicFormCode);
    if(dynamicForm == null || CollectionUtils.isEmpty(dynamicForm.getDynamicFields())) {
      return null;
    }
    // 检查是否存在映射关系，没有映射关系，就返回所有字段
    List<DynamicFormFieldMapping> dynamicFormFields = this.dynamicFormFieldMappingRepository.findByDynamicFormCodeAndMappingCode(dynamicFormCode, mappingCode);
    if(CollectionUtils.isEmpty(dynamicFormFields)) {
      return dynamicForm;
    }
    // 开始进行两个树结构的匹配（简单的剪枝过程）
    this.buildDynamicFields("", dynamicForm, dynamicFormFields);
    return dynamicForm;
  }
  
  /**
   * 这是一个递归方法，用于根据上级字段信息，构建前者的所有子级字段结构
   * @param parentFieldCode 这个父级字段特别需要注意，一层父级为：field；二层父级为field1.field2，以此类推
   * @param currentDynamiForm
   * @param dynamicFormFields
   */
  private void buildDynamicFields(String parentFieldCode , IDynamicForm currentDynamiForm , List<DynamicFormFieldMapping> dynamicFormFields) {
    /*
     * 1、从dynamicFormFields中找到key的前缀为parentFieldCode的字段信息，构造成一个K-V结构，以便后续使用，并基于此开始循环构建
     * 注意为null的情况，以及注意parentFieldCode为空的情况
     * 
     * 2、循环中，开始进行字段排序后的匹配过程，注意，只涉及对本级字段的梳理
     * 
     * 3、然后再在循环中，递归分析该级表单的下级表单，注意，下级表单所关联的本级表单字段，一定要在勾选的字段清单中
     * */
    // 1、=======
    // 现在从动态表单原始的字段集合中，过滤出映射的字段，映射成K-V结构，后续好使用
    List<DynamicFieldVo> dynamicFields = currentDynamiForm.getDynamicFields();
    List<DynamicChildrenFormVo> childrenForms = currentDynamiForm.getChildrenForms();
    if(CollectionUtils.isEmpty(dynamicFields)) {
      return;
    }
    //当map 中的键有重复的时候使用entity1的值作为键
    Map<String , DynamicFieldVo> dynamicFieldMappings = dynamicFields.stream().filter(
        item -> StringUtils.isNotBlank(item.getFieldCode()))
        .collect(Collectors.toMap(DynamicFieldVo::getFieldCode, item -> item,(entity1,entity2) -> entity1)
    );
    // 找到本层级在数据库中存储的设定字段
    List<DynamicFormFieldMapping> currentLevelDynamicFormFields = null;
    String prefixKey = "";
    if(StringUtils.isNotBlank(parentFieldCode)) {
      prefixKey = StringUtils.join(parentFieldCode , ".");
      currentLevelDynamicFormFields = dynamicFormFields.stream().filter(item -> StringUtils.indexOf(item.getFieldCode(), StringUtils.join(parentFieldCode , ".")) == 0).collect(Collectors.toList());
    } else {
      currentLevelDynamicFormFields = dynamicFormFields.stream().filter(item -> StringUtils.indexOf(item.getFieldCode(), ".") == -1).collect(Collectors.toList());
    }
    // 如果条件成立，说明没有配置本级表单的任何字段，不再做后续处理即可
    if(CollectionUtils.isEmpty(currentLevelDynamicFormFields)) {
      currentDynamiForm.setDynamicFields(null);
      currentDynamiForm.setChildrenForms(null);
      return;
    }
    
    List<DynamicFieldVo> resultDynamicFields = Lists.newArrayList();
    List<DynamicChildrenFormVo> resultChildrenForms = Lists.newArrayList();
    Map<String , DynamicChildrenFormVo> resultChildrenFormsMapping = Maps.newHashMap();
    if(!CollectionUtils.isEmpty(childrenForms)) {
      // 做成一个K-V结构，Key就是本级各个具有子级表单的字段的fieldCode
      resultChildrenFormsMapping = childrenForms.stream().collect(Collectors.toMap(DynamicChildrenFormVo::getParentFieldCode, item -> item));
    }
    for (DynamicFormFieldMapping currentDynamicFormFieldMapping : currentLevelDynamicFormFields) {
      // 2、=======
      String dynamicFieldCode = currentDynamicFormFieldMapping.getFieldCode();
      String dynamicFieldCodeNoPrefix = StringUtils.substringAfter(dynamicFieldCode, prefixKey);
      if(StringUtils.isBlank(dynamicFieldCode)) {
        continue;
      }
      DynamicFieldVo currentDynamicFieldVo = dynamicFieldMappings.get(dynamicFieldCodeNoPrefix);
      if(currentDynamicFieldVo == null) {
        continue;
      }
      // 这里要根据DynamicFormFieldMapping设定的nullable信息，修改dynamicField中的requed信息
      if(currentDynamicFormFieldMapping.getNullable() != null) {
        currentDynamicFieldVo.setRequired(!currentDynamicFormFieldMapping.getNullable());
      }
      // 这里如果保存的关系DynamicFormFieldMapping中有设置fieldName则以保存的为准，保存的为空则以注解为准
      if(StringUtils.isNotBlank(currentDynamicFormFieldMapping.getFieldName())) {
        currentDynamicFieldVo.setFieldName(currentDynamicFormFieldMapping.getFieldName());
      }
      resultDynamicFields.add(currentDynamicFieldVo);
      
      // 3、======
      // 如果当前层没有子级表单，或者当前字段不是一个关联下级表单的字段，则该步骤被忽略
      DynamicChildrenFormVo currentDynamicChildrenForm = resultChildrenFormsMapping.get(dynamicFieldCodeNoPrefix);
      if(CollectionUtils.isEmpty(childrenForms) || currentDynamicChildrenForm == null) {
        continue;
      }
      this.buildDynamicFields(dynamicFieldCode, currentDynamicChildrenForm, dynamicFormFields);
      resultChildrenForms.add(currentDynamicChildrenForm);
    }
    currentDynamiForm.setDynamicFields(resultDynamicFields);
    currentDynamiForm.setChildrenForms(resultChildrenForms);
  }

  @Transactional
  @Override
  public void rebinding(String dynamicFormCode, String mappingCode, List<String> fieldCodes) {
    Validate.isTrue(!CollectionUtils.isEmpty(fieldCodes) , "进行字段绑定/重绑定时，被绑定的字段信息必须传入");
    int len = fieldCodes.size();
    Boolean[] nullableArray = new Boolean[len];
    Arrays.fill(nullableArray, Boolean.TRUE);
    this.rebinding(dynamicFormCode, mappingCode, fieldCodes, Lists.newArrayList(nullableArray));
  }

  @Transactional
  @Override
  public void rebinding(String dynamicFormCode, String mappingCode, List<String> fieldCodes, List<Boolean> nullables) {
    this.rebinding(dynamicFormCode, mappingCode, fieldCodes, nullables, null , null);
  }
  
  @Override
  public void rebinding(String dynamicFormCode, String mappingCode, List<String> fieldCodes, List<Boolean> nullables, List<Integer> sortIndexs ,List<String> groupIndexs) {
    DynamicFormFieldMappingRebindDto rebindDto = new DynamicFormFieldMappingRebindDto();
    rebindDto.setDynamicFormCode(dynamicFormCode);
    rebindDto.setFieldNames(null);
    rebindDto.setFieldCodes(fieldCodes);
    rebindDto.setNullables(nullables);
    rebindDto.setMappingCode(mappingCode);
    rebindDto.setGroupIndexs(groupIndexs);
    rebindDto.setSortIndexs(sortIndexs);
    this.rebinding(rebindDto);
  }

  @Override
  public void rebinding(DynamicFormFieldMappingRebindDto dto) {
    /*
     * 重绑定过程为：
     * 1、首先进行有效性验证
     * 2、然后验证或者重新给定排序索引号
     * 3、验证或者重新给定分组信息
     * 3、然后删除指定dynamicFormCode指定mappingCode下所有的绑定信息
     * 4、最后再重新进行绑定
     * */
    // 1、========
    String dynamicFormCode = dto.getDynamicFormCode();
    String mappingCode = dto.getMappingCode();
    List<String> fieldCodes = dto.getFieldCodes();
    List<String> fieldNames = dto.getFieldNames();
    List<Boolean> nullables = dto.getNullables();
    List<Integer> sortIndexs = dto.getSortIndexs();
    List<String> groupIndexs = dto.getGroupIndexs();

    Validate.isTrue(!CollectionUtils.isEmpty(fieldCodes) , "进行字段绑定/重绑定时，被绑定的字段信息必须传入");
    Validate.isTrue(!CollectionUtils.isEmpty(nullables) , "进行字段绑定/重绑定时，被绑定的字段必须设定“是否必填信息”");
    Validate.notBlank(dynamicFormCode , "进行字段绑定/重绑定时，动态表单编号dynamicFormCode必须传入");
    DynamicFormVo currentDynamicForm = this.operationStrategyService.findByDynamicFormCode(dynamicFormCode);
    // 将整个表单中的属性结构进行降维，形成一维结构后续号进行验证判定
    List<String> reducedsFieldCodes = Lists.newArrayList();
    Map<String , DynamicFieldVo> dynamicFieldMapping = Maps.newLinkedHashMap();
    this.findAllFieldCodes("", currentDynamicForm.getChildrenForms(), currentDynamicForm.getDynamicFields(), reducedsFieldCodes , dynamicFieldMapping);
    Validate.isTrue(!CollectionUtils.isEmpty(reducedsFieldCodes) , "进行字段绑定/重绑定时，未发现当前动态表单设置了任何字段注解（@DynamicField），请检查");
    Validate.notNull(currentDynamicForm , "进行字段绑定/重绑定时，未找到指定的动态表单");
    Validate.notBlank(mappingCode , "进行字段绑定/重绑定时，映射业务编号mappingCode必须传入");
    // 检查fieldCodes的重复性
    Set<String> distinctFieldCodes = fieldCodes.stream().distinct().collect(Collectors.toSet());
    Validate.isTrue(distinctFieldCodes.size() == fieldCodes.size() , "进行字段绑定/重绑定时，至少存在一个重复的字段信息，请检查!!");
    // 必须不存在差集
    Validate.isTrue(Sets.difference(distinctFieldCodes , Sets.newHashSet(reducedsFieldCodes)).isEmpty() , "进行字段绑定/重绑定时，发现至少一个要求绑定的字段不属于指定的动态表单，请检查!!");
    // 检验必填字段的设定，特别是那些设置为true的字段，是否能够被设置为true；
    Validate.isTrue(distinctFieldCodes.size() == nullables.size() , "进行字段绑定/重绑定时，发现字段和字段的必填项设定不一致，请检查!!");
    if (!CollectionUtils.isEmpty(fieldNames)) {
      // 检验字段名称的设定
      Validate.isTrue(distinctFieldCodes.size() == fieldNames.size() , "进行字段绑定/重绑定时，发现字段和字段名称设定不一致，请检查!!");
    }
    // 检查所有字段都在currentDynamicFields之内，还要检查必填项的设定是否正确
    int index = 0;
    for (String fieldCode : fieldCodes) {
      DynamicFieldVo dynamicField = dynamicFieldMapping.get(fieldCode);
      Validate.notNull(dynamicField , "进行字段绑定/重绑定时，在操作时可以发现字段%s并不在动态表单中，请检查" , fieldCode);
      boolean isRequired = dynamicField.isRequired();
      // 如果该字段已在技术层面设置为了必填，则字段绑定设置中，该字段一定是必填
      if(isRequired) {
        Validate.isTrue(!nullables.get(index) , "进行字段绑定/重绑定时，发现字段[%s]在技术层面设置为必填，所以该字段不能由使用者设置为非必填" , fieldCode);
      }
      index++;
    }

    // 2、=======
    List<Integer> currentSortIndes = sortIndexs;
    // 如果条件成立，则重新完善排序索引号，否则进行验证
    if(CollectionUtils.isEmpty(currentSortIndes)) {
      currentSortIndes = Lists.newArrayList();
      for (int sortIndex = 0 ; sortIndex < fieldCodes.size() ; sortIndex++) {
        currentSortIndes.add(sortIndex);
      }
    } else {
      Validate.isTrue(fieldCodes.size() == currentSortIndes.size() , "进行字段绑定/重绑定时，发现传入的排序信息和字段信息的数量不吻合");
      Validate.isTrue(currentSortIndes.size() == (int)currentSortIndes.stream().filter(item -> item != null).distinct().count(), "进行字段绑定/重绑定时，发现传入的排序信息存在空值或者重复值，请检查!!");
    }

    // 3、======
    List<String> currentGroupIndexs = groupIndexs;
    if(CollectionUtils.isEmpty(currentGroupIndexs)) {
      currentGroupIndexs = Lists.newArrayList();
      for (int groupIndex = 0 ; groupIndex < fieldCodes.size() ; groupIndex++) {
        currentGroupIndexs.add(null);
      }
    } else {
      Validate.isTrue(fieldCodes.size() == currentGroupIndexs.size() , "进行字段绑定/重绑定时，发现传入的分组信息和字段信息的数量不吻合");
    }

    // 3、=======
    this.dynamicFormFieldMappingRepository.deleteByDynamicFormCodeAndMappingCode(dynamicFormCode, mappingCode);
    this.dynamicFormFieldMappingRepository.flush();

    // 4、=======
    index = 0;
    for (String fieldCode : fieldCodes) {
      DynamicFormFieldMapping dynamicFormFieldMapping = new DynamicFormFieldMapping();
      Boolean nullable = nullables.get(index);
      Integer sortIndex = currentSortIndes.get(index);
      String groupIndex = currentGroupIndexs.get(index);
      if (!CollectionUtils.isEmpty(fieldNames)) {
        dynamicFormFieldMapping.setFieldName(fieldNames.get(index));
      }
      dynamicFormFieldMapping.setDynamicFormCode(dynamicFormCode);
      dynamicFormFieldMapping.setFieldCode(fieldCode);
      dynamicFormFieldMapping.setMappingCode(mappingCode);
      dynamicFormFieldMapping.setNullable(nullable);
      dynamicFormFieldMapping.setSortIndex(sortIndex);
      dynamicFormFieldMapping.setGroupIndex(groupIndex);
      this.dynamicFormFieldMappingRepository.save(dynamicFormFieldMapping);
      index++;
    }
  }

  /**
   * 该方法负责将现在动态表单的中的所有字段进行降维处理
   * @param childrenForms
   * @param dynamicFields
   * @param reducedsFieldCodes 降维后的所有字段存储在这里
   */
  private void findAllFieldCodes(String parentFieldCode , List<DynamicChildrenFormVo> childrenForms , List<DynamicFieldVo> dynamicFields , List<String> reducedsFieldCodes ,  Map<String , DynamicFieldVo> dynamicFieldMapping) {
    // 如果条件成立，说明要累加当前表单的字段信息
    if(!CollectionUtils.isEmpty(dynamicFields)) {
      for (DynamicFieldVo dynamicField : dynamicFields) {
        String fieldCode = dynamicField.getFieldCode();
        if(StringUtils.isBlank(parentFieldCode)) {
          reducedsFieldCodes.add(fieldCode);
          dynamicFieldMapping.put(fieldCode, dynamicField);
        } else {
          reducedsFieldCodes.add(StringUtils.join(parentFieldCode , "." , fieldCode));
          dynamicFieldMapping.put(StringUtils.join(parentFieldCode , "." , fieldCode), dynamicField);
        }
      }
    }
    
    // 广度遍历，处理当前表单的下级表单
    if(!CollectionUtils.isEmpty(childrenForms)) {
      for (DynamicChildrenFormVo dynamicChildrenForm : childrenForms) {
        String currentParentFieldCode = null;
        if(StringUtils.isBlank(parentFieldCode)) {
          currentParentFieldCode = dynamicChildrenForm.getParentFieldCode();
        } else {
          currentParentFieldCode = StringUtils.join(parentFieldCode , "." , dynamicChildrenForm.getParentFieldCode());
        }
        this.findAllFieldCodes(currentParentFieldCode, dynamicChildrenForm.getChildrenForms(), dynamicChildrenForm.getDynamicFields(), reducedsFieldCodes , dynamicFieldMapping);
      }
    }
  }
}
